package org.mule.weave.v2.interpreted.node.executors

import org.mule.weave.v2.core.functions.BaseBinaryFunctionValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.UnexpectedFunctionCallTypesException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.Frame
import org.mule.weave.v2.interpreted.node.FunctionDispatchingHelper.findMatchingFunctionWithCoercion
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.model.values.ValuesHelper
import org.mule.weave.v2.parser.location.WeaveLocation

import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicReference

class BinaryFunctionExecutor(
  override val node: ValueNode[_],
  val name: String,
  val firstArgConstantType: Boolean,
  val secondArgConstantType: Boolean,
  override val showInStacktrace: Boolean = false,
  override val location: WeaveLocation) extends BinaryExecutor with Product4[ValueNode[_], String, Boolean, Boolean] {

  private val validated: AtomicBoolean = new AtomicBoolean(false)

  private val cachedCoercedDispatch: AtomicReference[(Type, Type, Seq[Int])] = new AtomicReference()

  override def executeBinary(fv: Value[_], sv: Value[_])(implicit ctx: ExecutionContext): Value[Any] = {
    val activeFrame: Frame = ctx.executionStack().activeFrame()
    try {
      activeFrame.updateCallSite(node)
      val functionValue = target().asInstanceOf[BaseBinaryFunctionValue]
      //This is the fast path
      if (validated.get()) {
        //If values are literal we do not need to validate every just the first time when we load the operation
        //Then is ok every time as its type will NEVER change
        val firstValue = if (!firstArgConstantType && functionValue.leftParam.typeRequiresMaterialization) {
          fv.materialize
        } else {
          fv
        }

        val secondValue = if (!secondArgConstantType && functionValue.rightParam.typeRequiresMaterialization) {
          sv.materialize
        } else {
          sv
        }
        if ((firstArgConstantType || functionValue.L.accepts(firstValue)) && (secondArgConstantType || functionValue.R.accepts(secondValue))) {
          return doCall(functionValue, firstValue, secondValue)
        }
      }
      val firstValue = if (functionValue.leftParam.typeRequiresMaterialization) {
        fv.materialize
      } else {
        fv
      }

      val secondValue = if (functionValue.rightParam.typeRequiresMaterialization) {
        sv.materialize
      } else {
        sv
      }

      if (cachedCoercedDispatch.get() != null) {
        val coercedOperation = cachedCoercedDispatch.get
        //If values are literal we do not need to validate every just the first time when we load the coerced operation
        //Then is ok every time as its type will NEVER change
        if ((firstArgConstantType || coercedOperation._1.accepts(firstValue)) &&
          (secondArgConstantType || coercedOperation._2.accepts(secondValue))) {
          val maybeFirstValue = if (functionValue.L.accepts(firstValue)) Some(firstValue) else functionValue.L.coerceMaybe(firstValue)
          val maybeSecondValue = if (functionValue.R.accepts(secondValue)) Some(secondValue) else functionValue.R.coerceMaybe(secondValue)
          if (maybeFirstValue.isDefined && maybeSecondValue.isDefined) {
            return doCall(functionValue, maybeFirstValue.get, maybeSecondValue.get)
          }
        }
      }

      if (functionValue.L.accepts(firstValue) && functionValue.R.accepts(secondValue)) {
        validated.set(true)
        doCall(functionValue, firstValue, secondValue)
      } else {
        val materializedValues: Array[Value[Any]] = ValuesHelper.array(firstValue.materialize, secondValue.materialize)
        val functionToCallWithCoercion: Option[(Int, Array[Value[_]], Seq[Int])] = findMatchingFunctionWithCoercion(materializedValues, Array(functionValue), this)
        functionToCallWithCoercion match {
          case Some((_, argumentsWithCoercion, paramsToCoerce)) => {
            //Cache the coercion use the base type to avoid Memory Leaks as Types may have references to Streams or Objects
            cachedCoercedDispatch.set((firstValue.valueType.baseType, secondValue.valueType.baseType, paramsToCoerce))
            val firstCoercedValue: Value[_] = argumentsWithCoercion(0)
            val secondCoercedValue = argumentsWithCoercion(1)
            doCall(functionValue, firstCoercedValue, secondCoercedValue)
          }
          case None =>
            throw new UnexpectedFunctionCallTypesException(location, name, materializedValues, Seq(functionValue.functionParamTypes.map(_.theType)))
        }
      }
    } finally {
      activeFrame.cleanCallSite()
    }
  }

  private def doCall(functionValue: BaseBinaryFunctionValue, firstValue: Value[_], secondValue: Value[_])(implicit ctx: ExecutionContext) = {
    try {
      functionValue.call(firstValue, secondValue)
    } catch {
      case ex: ExecutionException =>
        if (showInStacktrace) {
          ex.addCallToStacktrace(location, name())
        }
        throw ex
    }
  }

  def target()(implicit ctx: ExecutionContext): FunctionValue = {
    node.execute.asInstanceOf[FunctionValue]
  }

  override def _1: ValueNode[_] = node

  override def _2: String = name

  override def _3: Boolean = firstArgConstantType

  override def _4: Boolean = secondArgConstantType

  override def execute(arguments: Array[Value[_]])(implicit ctx: ExecutionContext): Value[Any] = {
    executeBinary(arguments(0), arguments(1))
  }

  override def name()(implicit ctx: ExecutionContext): String = {
    this.name
  }

}
