package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.WeaveLocation
/**
  * Represents a reference to a WeaveType
  */
trait TypeRef {
  /**
    * Returns the referenced type
    *
    * @return The referenced type
    */
  def ref(): WeaveType

  /**
    * The resolver used to resolve this ref if any
    *
    * @return Returns the resolver used to resolve this reference if any
    */
  def refResolver(): Option[WeaveTypeReferenceResolver] = None
}

case class LiteralReference(literalType: WeaveType, override val refResolver: Option[WeaveTypeReferenceResolver]) extends TypeRef() {
  override def ref(): WeaveType = {
    literalType
  }
}

class DefaultTypeReference(refName: NameIdentifier, maybeTypeArguments: Option[Seq[WeaveType]], val referenceResolver: WeaveTypeReferenceResolver) extends TypeRef {
  private var referencedType: WeaveType = _
  private var resolvedType: Option[WeaveType] = _

  override def ref(): WeaveType = {
    if (referencedType == null) {
      val resolvedReferenceType = referenceResolver.resolveType(refName, maybeTypeArguments)
      resolvedType = resolvedReferenceType
      referencedType = resolvedReferenceType match {
        case Some(rrt) => {
          val cloned = WeaveTypeCloneHelper.clone(rrt, refName.location())
          cloned
        }
        case None => {
          AnyType()
        }
      }
    }
    referencedType
  }

  override def refResolver(): Option[WeaveTypeReferenceResolver] = Some(referenceResolver)
}

class TypeSelectorReference(nsPrefix: Option[NamespaceNode], refName: String, referencedType: ReferenceType, refLocation: WeaveLocation, referenceResolver: WeaveTypeReferenceResolver) extends TypeRef {

  var resultType: WeaveType = _

  override def ref(): WeaveType = {
    if (resultType == null) {
      val propertyPairs = TypeHelper.selectPropertyPairs(referencedType, refName).filter(kvpt => {
        kvpt.key match {
          case KeyType(NameType(Some(QName(_, None))), _) if nsPrefix.isEmpty => true
          case KeyType(NameType(Some(QName(_, Some(ns)))), _) if nsPrefix.isDefined => referenceResolver.resolveNamespace(nsPrefix.get).contains(ns)
          case _ => false
        }
      })
      resultType = propertyPairs match {
        case Seq()      => AnyType()
        case Seq(first) => WeaveTypeCloneHelper.clone(first.value, refLocation)
        case _          => UnionType(propertyPairs.map(_.value))
      }
    }
    resultType
  }
}