package org.mule.weave.v2.ts

import org.mule.weave.v2.ts.Kind.ARRAY
import org.mule.weave.v2.ts.Kind.FIELD
import org.mule.weave.v2.ts.Kind.Kind
import org.mule.weave.v2.ts.Kind.RETURN_TYPE
import org.mule.weave.v2.utils.WeaveTypeEmitter

object SelectionPath {
  def apply(): SelectionPath = ObjectSelectionPath()
}

trait SelectionPath {

  def isEmpty: Boolean

  def field(element: String): SelectionPath

  def itemArray(): SelectionPath

  def arg(name: String, weaveType: WeaveType, argIndex: Int): SelectionPath

  def returnType(argumentsLength: Int, weaveType: WeaveType): SelectionPath

  def toString(): String
}

case class ArgumentSelectionPath(previousPath: SelectionPath, name: String, weaveType: WeaveType, argumentIndex: Int, childExpression: SelectionPath) extends SelectionPath {

  def isEmpty: Boolean = false

  def field(element: String): SelectionPath = {
    ArgumentSelectionPath(previousPath, name, weaveType, argumentIndex, childExpression.field(element))
  }

  def itemArray(): SelectionPath = {
    ArgumentSelectionPath(previousPath, name, weaveType, argumentIndex, childExpression.itemArray())
  }

  override def toString(): String = {
    val arguments: String = "_, " * argumentIndex
    previousPath + "(" + arguments + name + ": " + WeaveTypeEmitter.toPresentableString(weaveType, "", "") + (if (!childExpression.isEmpty) "." + childExpression.toString else "") + ")"
  }

  override def arg(name: String, weaveType: WeaveType, argIndex: Int): SelectionPath = {
    ArgumentSelectionPath(this, name, weaveType, argIndex, SelectionPath())
  }

  override def returnType(argumentsLength: Int, weaveType: WeaveType): SelectionPath = {
    ArgumentSelectionPath(previousPath, name, weaveType, argumentIndex, childExpression.returnType(argumentsLength, weaveType))
  }
}

case class ObjectSelectionPath(elements: Seq[(String, Kind)] = Seq()) extends SelectionPath {

  def isEmpty: Boolean = elements.isEmpty

  def field(element: String): SelectionPath = {
    ObjectSelectionPath(elements :+ (element, FIELD))
  }

  def itemArray(): SelectionPath = {
    ObjectSelectionPath(elements :+ ("[*]", FIELD))
  }

  override def toString(): String = {
    if (elements.isEmpty)
      ""
    else {
      val value1 = elements.head._1
      elements.tail.foldLeft(value1)((acc, value) => {
        value._2 match {
          case RETURN_TYPE => acc + value._1
          case FIELD       => acc + "." + value._1
          case ARRAY       => acc + value._1
        }
      })
    }
  }

  override def arg(name: String, weaveType: WeaveType, argIndex: Int): SelectionPath = {
    ArgumentSelectionPath(this, name, weaveType, argIndex, SelectionPath())
  }

  override def returnType(argumentsLength: Int, weaveType: WeaveType): SelectionPath = {
    val args = List.fill(argumentsLength)("_").mkString("(", ",", ")")
    ObjectSelectionPath(elements :+ (args + " -> " + WeaveTypeEmitter.toPresentableString(weaveType, "", ""), RETURN_TYPE))
  }
}