package org.mule.weave.v2.ts

import org.mule.weave.v2.parser.ast.QName

import scala.collection.mutable.ArrayBuffer

object ObjectTypeHelper {

  /**
    * Subtract an object from another one the best way possible, this is not
    * an exact subtraction, but an approximation.
    *
    * For instance, {a: 12 | 34, b: 56 | 78} \ {a: 12, b: 56}
    * will return {a: 12 | 34, b: 56 | 78}, while a more exact result could be
    * {a: 12, b: 56 | 78} | {a: 12 | 34, b: 78}.
    *
    * @param orig The original Object
    * @param toRemove The object to remove
    * @param ctx The type resolution context
    * @return The resulting Type
    */
  def subtractObjects(orig: ObjectType, toRemove: ObjectType, ctx: WeaveTypeResolutionContext, typeHelper: TypeHelper): WeaveType = {
    if (typeHelper.canBeAssignedTo(orig, toRemove, ctx)) {
      // If orig is a subtype of toRemove, return Nothing
      NothingType()
    } else {
      val result = toRemove match {
        case ObjectType(Seq(KeyValuePairType(KeyType(NameType(Some(remName)), Seq()), remValue, _, _)), false, _) =>
          val newProps = orig.properties map {
            case KeyValuePairType(KeyType(NameType(Some(name)), attrs), value, optional, false) if name.matchesPattern(remName) =>
              val newValue = typeHelper.subtractType(value, remValue, ctx)
              KeyValuePairType(KeyType(NameType(Some(name)), attrs), newValue, optional)
            case kvp => kvp
          }

          ObjectType(newProps, orig.close, orig.ordered)

        case ObjectType(Seq(KeyValuePairType(KeyType(NameType(Some(remName)), Seq()), remValue, _, _)), true, _) =>
          // If both are closed, and have the same only property
          val newProps = orig.properties match {
            case Seq(KeyValuePairType(KeyType(NameType(Some(name)), attrs), value, optional, false)) if name.matchesPattern(remName) =>
              val newValue = typeHelper.subtractType(value, remValue, ctx)
              Seq(KeyValuePairType(KeyType(NameType(Some(name)), attrs), newValue, optional))
            case other => other
          }
          ObjectType(newProps, orig.close, orig.ordered)

        case _ => orig
      }
      if (typeHelper.isEmptyType(result)) {
        NothingType()
      } else {
        result
      }
    }

  }

  /**
    * Selects a property in the specified object
    *
    * @param propertyName The property name to select
    * @param from         The object to select from
    * @return The Value of the property with the name specified if found
    */
  def selectProperty(propertyName: QName, from: ObjectType): Option[KeyValuePairType] = {
    selectProperty(propertyName, from.properties)
  }

  /**
    * Selects a property in the specified object
    *
    * @param propertyName The property name to select
    * @param properties         The object to select from
    * @return The Value of the property with the name specified if found
    */
  def selectProperty(propertyName: QName, properties: Seq[KeyValuePairType]): Option[KeyValuePairType] = {
    properties
      .find((kvpt) =>
        kvpt.key match {
          case KeyType(NameType(Some(name)), _) => name.matchesPattern(propertyName)
          case _                                => false
        })
      .map({
        case kvp @ KeyValuePairType(KeyType(NameType(Some(name)), attrs), values, optional, repeated) => {
          if (name.matchesAllNs) {
            KeyValuePairType(KeyType(NameType(Some(propertyName)), attrs), values, optional, repeated)
          } else {
            kvp
          }
        }
        case t => throw new MatchError(t)
      })
  }

  def selectAttribute(propertyName: QName, properties: Seq[NameValuePairType]): Option[NameValuePairType] = {
    properties.find((nvpt) =>
      nvpt.name match {
        case NameType(Some(name)) => name.equals(propertyName)
        case _                    => false
      })
  }

  /**
    * Selects a value in the specified object
    *
    * @param propertyName The property name to select
    * @param from         The object to select from
    * @return The Value of the property with the name specified if found
    */
  def selectPropertyValue(propertyName: QName, from: ObjectType): Option[WeaveType] = {
    selectProperty(propertyName, from).map(_.value)
  }

  /**
    * Selects all properties values that matches the specified
    *
    * @param propertyName The property name to select
    * @param from         The object to select from
    * @return The Value of the property with the name specified if found
    */
  def selectAllProperties(propertyName: QName, from: ObjectType): Seq[KeyValuePairType] = {
    val objectProperties = from.properties
    selectAllProperties(propertyName, objectProperties)
  }

  /**
    * Selects all properties values over the list of properties
    *
    * @param propertyName      The name of the property to search
    * @param keyValuePairTypes The properties to search in
    * @return The list of matching properties
    */
  def selectAllProperties(propertyName: QName, keyValuePairTypes: Seq[KeyValuePairType]): Seq[KeyValuePairType] = {
    keyValuePairTypes.filter((kvpt) =>
      kvpt.key match {
        case KeyType(NameType(Some(name)), _) => name.equals(propertyName)
        case _                                => false
      })
  }

  /**
    * Matches every left property with a name, with the corresponding right property
    *
    * @param lprop left properties
    * @param rprop right properties
    * @return
    */
  def matchLeftProperties(lprop: Seq[KeyValuePairType], rprop: Seq[KeyValuePairType], substitutionMode: Boolean): Seq[(Option[KeyValuePairType], Option[KeyValuePairType])] = {
    var answer: Seq[(Option[KeyValuePairType], Option[KeyValuePairType])] = lprop.flatMap({
      case kvp @ KeyValuePairType(KeyType(NameType(Some(name)), _), _, _, _) =>
        val r: Option[KeyValuePairType] = selectProperty(name, rprop)
        Seq((Some(kvp), r))
      case kvp @ KeyValuePairType(rt, _, _, _) if (!rt.isInstanceOf[KeyType] && !substitutionMode) =>
        //If it is AbstractType Parameter then we are going to ignore
        //As we don't know to what to compare
        val rProperties: Seq[KeyValuePairType] = rprop.filter((r) => {
          TypeHelper.areEqualStructurally(r.key, rt)
        })
        Seq((Some(kvp), rProperties.headOption))
      case _ => Seq()
    })

    val lDef: Option[KeyValuePairType] = selectDefaultProperty(lprop)

    // Add every rprop that hasnt been used with ldef
    val unused_rprops: Seq[KeyValuePairType] =
      rprop
        .collect({
          case rkvp @ KeyValuePairType(KeyType(NameType(_), _), _, _, _) => rkvp
        })
        .filter((rkvp) => {
          !answer.exists((matchedProp) => {
            matchedProp._2.isDefined && (rkvp eq matchedProp._2.get)
          })
        })

    answer = answer ++ (unused_rprops.map(kvp => (lDef, Some(kvp))))

    // Add lDef if not optional {_: String} = {}
    if (lDef.isDefined && !lDef.get.optional && unused_rprops.isEmpty) {
      answer = answer :+ (lDef, None)
    }

    answer
  }

  /**
    * Match all properties with the right counterpart semantically, that means
    * that a default will be deduced if one is not present.
    *
    * @param lprop
    * @param rprop
    * @param lClose
    * @param rClose
    * @return
    */
  def matchAllProperties(lprop: Seq[KeyValuePairType], rprop: Seq[KeyValuePairType], lClose: Boolean, rClose: Boolean): Seq[(WeaveType, WeaveType)] = {
    var lDefUsed: Boolean = false
    val lDef: KeyValuePairType =
      selectDefaultProperty(lprop).getOrElse({
        if (lClose) {
          KeyValuePairType(KeyType(NameType()), NothingType(), optional = true)
        } else {
          KeyValuePairType(KeyType(NameType()), AnyType(), optional = true)
        }
      })

    var rDefUsed = false
    val rDef: KeyValuePairType = selectDefaultProperty(rprop).getOrElse({
      if (rClose) {
        KeyValuePairType(KeyType(NameType()), NothingType(), optional = true)
      } else {
        KeyValuePairType(KeyType(NameType()), AnyType(), optional = true)
      }
    })

    val names: Seq[QName] = (lprop ++ rprop)
      .collect({
        case KeyValuePairType(KeyType(NameType(Some(name)), _), _, _, _) if !name.matchesAllNs => name
      })
      .distinct

    var answer: Seq[(WeaveType, WeaveType)] = names.map((n) => {
      (selectProperty(n, lprop), selectProperty(n, rprop)) match {
        case (Some(l), Some(r)) =>
          val p = (l, r)
          p
        case (Some(l), None) =>
          if (!l.optional) {
            rDefUsed = true
          }
          (l, rDef)
        case (None, Some(r)) =>
          if (!r.optional) {
            lDefUsed = true
          }
          (lDef, r)
        case a => throw new MatchError(a)
      }
    })

    // If the default values haven't been matched, match them
    // together
    if (lDefUsed) {
      lDef.optional = true
    }
    if (rDefUsed) {
      rDef.optional = true
    }
    (selectDefaultProperty(lprop), selectDefaultProperty(rprop)) match {
      case (None, None) =>
      case _ =>
        // I think this can go without the guard
        answer = answer :+ (lDef, rDef)
    }

    // Complete the answer with every KVPT that
    // doesn't have a Name

    val lprop_rest: Seq[(WeaveType, Left[KeyValuePairType, Nothing])] = lprop
      .filter({
        case KeyValuePairType(KeyType(NameType(_), _), _, _, _) => false
        case _ => true
      })
      .map(kvp => (kvp.key, Left(kvp)))

    val rprop_rest: Seq[(WeaveType, Right[Nothing, KeyValuePairType])] = rprop
      .filter({
        case KeyValuePairType(KeyType(NameType(_), _), _, _, _) => false
        case _ => true
      })
      .map(kvp => (kvp.key, Right(kvp)))

    val allDynamicProperties: Seq[(WeaveType, Either[KeyValuePairType, KeyValuePairType])] = lprop_rest.++(rprop_rest)
    val usedProperties: ArrayBuffer[(WeaveType, Either[KeyValuePairType, KeyValuePairType])] = ArrayBuffer()
    val prop_rest = ArrayBuffer[Seq[Either[KeyValuePairType, KeyValuePairType]]]()

    allDynamicProperties.foreach((p) => {
      if (!usedProperties.contains(p)) {
        val tuples: Seq[(WeaveType, Either[KeyValuePairType, KeyValuePairType])] = allDynamicProperties
          .filter((prop) => {
            TypeHelper.areEqualStructurally(p._1, prop._1)
          })
        usedProperties.++=(tuples)
        prop_rest.+=(tuples.map(_._2))
      }
    })

    val grouped_rest = prop_rest.map(kvps => {
      val left_props = kvps.collect({
        case Left(kvp) => kvp
      })
      val right_props = kvps.collect({
        case Right(kvp) => kvp
      })

      val left: WeaveType = left_props.size match {
        case 0 => lDef
        case 1 => left_props.head
        case _ => IntersectionType(left_props)
      }

      val right: WeaveType = right_props.size match {
        case 0 => rDef
        case 1 => right_props.head
        case _ => IntersectionType(right_props)
      }

      (left, right)
    })
    answer = answer ++ grouped_rest

    answer
  }

  def selectDefaultProperty(properties: Seq[KeyValuePairType]): Option[KeyValuePairType] = {
    properties.find((kvpt) =>
      kvpt.key match {
        case KeyType(NameType(None), _) => true
        case _                          => false
      })
  }

  /**
    * Select semantically the required KeyValuePairType
    *
    * If it's part of the object, it is returned, otherwise it's copied from
    * the default, if there is one, with the name changed, and with the
    * optional and repeated flags enabled.
    * If there's no default KVPT, it creates one depending on whether
    * the object is open (default is `_?: Any`) or closed
    * (default is `_?: Nothing`)
    *
    * @param name
    * @param props
    * @param closed
    * @return
    */
  def selectSemanticProperty(name: QName, props: Seq[KeyValuePairType], closed: Boolean): KeyValuePairType = {
    selectProperty(name, props) match {
      case Some(value) => value
      case None =>
        selectDefaultProperty(props) match {
          case Some(KeyValuePairType(KeyType(_, attrs), value, _, _)) =>
            KeyValuePairType(KeyType(NameType(Some(name)), attrs), value, true, true)
          case _ =>
            KeyValuePairType(KeyType(NameType(Some(name)), Seq()), if (closed) NothingType() else AnyType(), true, true)
        }
    }
  }

}
