package org.mule.weave.v2.sdk.selectors

import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.CustomTypeResolver
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NullType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.RecursionDetector
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext

trait BaseValueSelectorCustomTypeResolver extends CustomTypeResolver {

  def getSelectorNameType(selector: WeaveType): Option[NameType] = {
    selector match {
      case StringType(strValue)                  => Some(NameType(strValue.map(QName(_))))
      case nt: NameType                          => Some(nt)
      case tp: TypeParameter if tp.top.isDefined => tp.top.flatMap(getSelectorNameType)
      case _                                     => None
    }
  }

  override def resolve(actualTypes: Seq[WeaveType], ctx: WeaveTypeResolutionContext, node: TypeNode, resolvedReturnType: WeaveType): Option[WeaveType] = {
    val leftType: WeaveType = actualTypes.head
    resolve(leftType, actualTypes(1), ctx, node, false) match {
      case NoMatch(closed) => {
        if (closed) {
          Some(NullType().markSelectionMissed())
        } else {
          Some(AnyType())
        }
      }
      // If we set None then is not going to keep propagating. The other option is to return Any.
      // For now to continue with old behaviour we will keep None as is the safe
      case Unknown            => None
      case Matched(weaveType) => Some(weaveType)
    }
  }

  def resolve(leftType: WeaveType, nameType: WeaveType, ctx: WeaveTypeResolutionContext, node: TypeNode, insideArray: Boolean, recursionDetector: RecursionDetector[SelectionResult] = RecursionDetector((_, _) => Unknown)): SelectionResult = {
    leftType match {
      case ArrayType(of) => {
        resolveSelectionOverArray(nameType, ctx, node, of)
      }
      case IntersectionType(of) => {
        TypeHelper.resolveIntersection(of) match {
          case IntersectionType(of) => {
            val resolved =
              of.map((element) => resolve(element, nameType, ctx, node, insideArray, recursionDetector))
                .find((sr) => sr.hasResult())
            resolved.getOrElse(NoMatch(closed = false))
          }
          case otherType => {
            resolve(otherType, nameType, ctx, node, insideArray, recursionDetector)
          }
        }
      }
      case ut: UnionType => {
        val weaveType = if (insideArray) {
          val unifiedObjects = TypeHelper.unify(ut.of.filter((wtype) => isObjectType(wtype)))
          TypeHelper.resolveUnion(TypeHelper.simplify(unifiedObjects))
        } else {
          TypeHelper.resolveUnion(TypeHelper.simplify(ut))
        }
        weaveType match {
          case UnionType(elements) => {
            val results: Seq[SelectionResult] = elements
              .map((unionTypePart) => {
                resolve(unionTypePart, nameType, ctx, node, insideArray, recursionDetector)
              })
            val resolved: Seq[WeaveType] =
              results
                .collect({
                  case Matched(weaveType) => weaveType
                })
            if (resolved.isEmpty) {
              val isOpen = results.exists {
                case NoMatch(closed) => !closed
                case _               => true
              }
              NoMatch(!isOpen)
            } else {
              Matched(TypeHelper.unify(resolved))
            }
          }
          case theType => {
            resolve(theType, nameType, ctx, node, insideArray, recursionDetector)
          }
        }
      }
      case TypeParameter(_, topType, bottomType, _, _) => {
        resolve(topType.orElse(bottomType).getOrElse(AnyType()), nameType, ctx, node, insideArray, recursionDetector)
      }
      case rt: ReferenceType => {
        recursionDetector.resolve(rt, (r) => resolve(r, nameType, ctx, node, insideArray, recursionDetector))
      }
      case _ => select(leftType, nameType, ctx, node, insideArray)
    }
  }

  private def isObjectType(wtype: WeaveType, recursionDetector: RecursionDetector[Boolean] = RecursionDetector((_, _) => false)): Boolean = {
    wtype match {
      case _: ObjectType => true
      case it: IntersectionType => {
        TypeHelper.resolveIntersection(it.of) match {
          case _: ObjectType        => true
          case IntersectionType(of) => of.exists(isObjectType(_, recursionDetector))
          case _                    => false
        }
      }
      case ut: UnionType => {
        ut.of.exists((wt) => isObjectType(wt, recursionDetector))
      }
      case referenceType: ReferenceType => {
        recursionDetector.resolve(referenceType, (wt) => isObjectType(wt, recursionDetector))
      }
      case _ => false
    }
  }

  protected def resolveSelectionOverArray(nameType: WeaveType, ctx: WeaveTypeResolutionContext, node: TypeNode, of: WeaveType): SelectionResult = {
    val result = resolve(of, nameType, ctx, node, true)
    result match {
      case NoMatch(closed) => {
        if (closed) {
          NoMatch(closed)
        } else {
          Matched(ArrayType(AnyType()).markOptional())
        }
      }
      case Unknown => Unknown
      case Matched(weaveType) => {
        Matched(ArrayType(weaveType).withOptional(weaveType.isOptional()))
      }
    }
  }

  /**
    * Returns the selection of selecting the nameType over the leftType.
    *  - If there is no match because it is dynamic or either no name was found the NoMatch result should be returned.
    *     - If the no match was over a closed object then it should be true
    *     - If the no match was over an open object then it should be false
    *  - If the result is Unknown for example selecting over Nothing or Any then Unknown should be result
    *  - If there is a match then a Matched(weaveType) object should be return
    */
  def select(leftType: WeaveType, nameType: WeaveType, ctx: WeaveTypeResolutionContext, node: TypeNode, insideArray: Boolean): SelectionResult

  def containsNameType(leftType: WeaveType, nameType: WeaveType): Boolean = {
    val name: NameType = getSelectorNameType(nameType).get
    leftType match {
      case ObjectType(properties, _, _) => {
        name.value match {
          case Some(selectedName) => {
            properties
              .exists((prop) => {
                prop.key match {
                  case KeyType(NameType(Some(propName)), _) => {
                    propName.selectedBy(selectedName)
                  }
                  case _ => false
                }
              })
          }
          case _ => false
        }
      }
      case _ => false
    }
  }
}

sealed trait SelectionResult {
  def hasResult(): Boolean

  def maybeResult(): Option[WeaveType]
}

/**
  * When no match was found. This mean that
  *
  * @param closed
  */
case class NoMatch(closed: Boolean) extends SelectionResult {
  override def hasResult(): Boolean = false

  override def maybeResult(): Option[WeaveType] = None
}

/**
  * When nothing can be inferred
  */
object Unknown extends SelectionResult {
  override def hasResult(): Boolean = false

  override def maybeResult(): Option[WeaveType] = None
}

case class Matched(weaveType: WeaveType) extends SelectionResult {
  override def hasResult(): Boolean = true

  override def maybeResult(): Option[WeaveType] = Some(weaveType)
}
