package org.mule.weave.v2.sdk

import java.net.URL
import java.util
import java.util.Collections

import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import scala.collection.JavaConverters._

import scala.collection.mutable
import scala.io.Source

class ClassLoaderWeaveResourceResolver(classloaders: Seq[() => ClassLoader]) extends WeaveResourceResolver {

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    val path = NameIdentifierHelper.toWeaveFilePath(name, "/")
    resolvePath(path)
  }

  override def resolvePath(path: String): Option[WeaveResource] = {
    val weaveFilePath: String = if (path.startsWith("/")) path.substring(1) else path
    var resources: util.Enumeration[URL] = Collections.emptyEnumeration()
    val classLoadersIt = classloaders.toIterator
    while (classLoadersIt.hasNext && !resources.hasMoreElements) {
      val classLoader = classLoadersIt.next()()
      resources = classLoader.getResources(weaveFilePath)
    }
    if (resources.hasMoreElements) {
      val element: URL = resources.nextElement()
      val source = Source.fromInputStream(element.openStream(), "UTF-8")
      try {
        Some(WeaveResource(element.toExternalForm, source.mkString))
      } finally {
        source.close()
      }
    } else {
      None
    }
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    val path = NameIdentifierHelper.toWeaveFilePath(name, "/")
    lookupResources(path)
  }

  def lookupResources(path: String): Seq[WeaveResource] = {
    val weaveFilePath: String = if (path.startsWith("/")) path.substring(1) else path
    var resources: util.Enumeration[URL] = Collections.emptyEnumeration();
    val classLoadersIt = classloaders.toIterator
    val result = mutable.ArrayBuffer[WeaveResource]()
    while (classLoadersIt.hasNext) {
      val classLoader = classLoadersIt.next()()
      resources = classLoader.getResources(weaveFilePath)
      result.++=(resources.asScala.map((resource) => {
        val source = Source.fromInputStream(resource.openStream(), "UTF-8")
        try {
          WeaveResource(resource.toExternalForm, source.mkString)
        } finally {
          source.close()
        }
      }))
    }
    result
  }
}

object ClassLoaderWeaveResourceResolver {
  def apply(): ClassLoaderWeaveResourceResolver = {
    val mainClassloader = getClass.getClassLoader
    new ClassLoaderWeaveResourceResolver(Seq(() => mainClassloader, () => Thread.currentThread().getContextClassLoader))
  }

  def noContextClassloader(): ClassLoaderWeaveResourceResolver = {
    val mainClassloader = getClass.getClassLoader
    new ClassLoaderWeaveResourceResolver(Seq(() => mainClassloader))
  }

  def contextClassloaderOnly(): ClassLoaderWeaveResourceResolver = {
    new ClassLoaderWeaveResourceResolver(Seq(() => Thread.currentThread().getContextClassLoader))
  }

  def providedClassLoader(classLoaders: Seq[ClassLoader]): ClassLoaderWeaveResourceResolver = {
    val mainClassloader = getClass.getClassLoader
    val all = Seq(mainClassloader) ++ classLoaders
    new ClassLoaderWeaveResourceResolver(all.map(cl => () => cl))
  }

}
