package org.mule.weave.v2.scope

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.AstNodeHelper.containsChild
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.ts.TypeInformationProviderService
import org.mule.weave.v2.ts.WeaveType

class DependenciesAnalyzerService(editor: WeaveEditorSupport) {

  val typeInformationProviderService = new TypeInformationProviderService(editor)

  def externalScopeDependencies(startBlock: Int, endBlock: Int, topScope: Option[VariableScope]): Array[VariableDependency] = {
    var result = Array[VariableDependency]()
    val scope = editor.scopeGraph()
    if (scope.isDefined) {
      val scopesNavigator: ScopesNavigator = scope.get
      val maybeNode: Option[AstNode] = scopesNavigator.rootScope.astNavigator().nodeAt(startBlock, endBlock)
      if (maybeNode.isDefined) {
        val identifiers: Seq[NameIdentifier] = calculateVariableDependenciesOf(maybeNode.get, scopesNavigator, topScope)
        val dependencies: Array[VariableDependency] = identifiers
          .groupBy((ni) => ni)
          .values
          .map((nis) => {
            val ni = nis.head
            VariableDependency(ni.name, typeInformationProviderService.typeOf(ni), nis.map(_.location()).toArray)
          }).toArray
        val sortedDependencies = dependencies.sortBy(_.name.toUpperCase)
        result = sortedDependencies
      }
    }
    result
  }

  /**
    * Returns the list of variable that are declared under the specified node that are not locally resolved
    *
    * @param node            The node to check
    * @param scopesNavigator The scope navigator
    * @return The list of fields.
    */
  private def calculateVariableDependenciesOf(node: AstNode, scopesNavigator: ScopesNavigator, topScope: Option[VariableScope]): Seq[NameIdentifier] = {
    val scope: VariableScope = scopesNavigator.scopeOf(node).get
    val variableReferences: Seq[AstNode] =
      if (node.isInstanceOf[VariableReferenceNode]) {
        Seq(node)
      } else {
        AstNodeHelper.collectChildrenWith(node, classOf[VariableReferenceNode])
      }
    variableReferences
      .collect({
        case vrn: VariableReferenceNode => {
          vrn.variable
        }
      })
      .filter((variableReference) => {
        val reference: Option[Reference] = scope.resolveVariable(variableReference.localName())
        reference match {
          case Some(reference) => {
            val maybeScope = scopesNavigator.scopeOf(reference.referencedNode)
            if (maybeScope.isDefined) {
              val declarationScope: VariableScope = maybeScope.get
              if (reference.isLocalReference) {
                val isDeclaredOutSideTheSelection: Boolean = !containsChild(node, reference.referencedNode)
                val isNotVisibleInsideTargetScope: Boolean = topScope.isEmpty || declarationScope.index > topScope.get.index
                isDeclaredOutSideTheSelection && isNotVisibleInsideTargetScope
              } else {
                false
              }
            } else {
              false
            }
          }
          case None => {
            false
          }
        }
      })

  }
}

case class VariableDependency(name: String, weaveType: Option[WeaveType], usages: Array[WeaveLocation])
