package org.mule.weave.v2.parser.phase

import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.ast.types.TypeParameterNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.ts.Constraint
import org.mule.weave.v2.ts.ConstraintSet
import org.mule.weave.v2.ts.DoubleRecursionDetector
import org.mule.weave.v2.ts.EmptyConstrainProblem
import org.mule.weave.v2.ts.NoSolutionSet
import org.mule.weave.v2.ts.ScopeGraphTypeReferenceResolver
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeResolutionContext

/**
  * Typechecks all generics used in type references. e.g:
  *
  * type O<T <: {}> = {}
  * var o: O<Number> = {}
  *          ^^^^^^ <-- Marks this assignment as invalid
  *
  * Assumptions:
  * - Amount of type params matches the definition is already validated on a previous phase e.g
  * type O<T> = {}
  * var o: O<Number, String> = ??? // This error failed on a previous phase (TypeParameterCheckerPhase)
  *
  */
class TypeReferenceGenericsChecker[R <: AstNode, T <: TypeCheckingResult[R]]() extends VerificationPhase[T] {
  override def verify(source: T, context: ParsingContext): Unit = {
    val typeGraph: TypeGraph = source.typeGraph
    val referenceResolver = source.scope.referenceResolver
    val typeResolutionContext = new WeaveTypeResolutionContext(typeGraph)
    val scope = source.scope

    val typeReferenceNodes = scope.astNavigator().allWithType(classOf[TypeReferenceNode])
    val referencesWithTypeArgs = typeReferenceNodes.filter(_.typeArguments.nonEmpty)

    referencesWithTypeArgs.foreach(rn => {
      val maybeReference = scope.resolveVariable(rn.variable)
      if (maybeReference.nonEmpty) {
        val reference = maybeReference.get
        val typeDeclarationNode = reference.scope.astNavigator().parentOf(reference.referencedNode).get
        typeDeclarationNode match {
          case td: TypeDirective =>
            val typeArguments = rn.typeArguments.map(_.map(WeaveType(_, referenceResolver)))
            typeArguments match {
              case Some(typeArguments) =>
                td.typeParametersListNode match {
                  case Some(typeParameters) =>
                    val parameters: Seq[TypeParameterNode] = typeParameters.typeParameters
                    val paramTypes = td.typeParametersListNode.get.typeParameters.map(tp => {
                      referenceResolver.asInstanceOf[ScopeGraphTypeReferenceResolver].resolveTypeParameter(tp)
                    })
                    if (parameters.size == typeArguments.size) {
                      val constraintSet: ConstraintSet = paramTypes
                        .zip(typeArguments)
                        .map(t => Constraint.collectConstrains(t._1, t._2, typeResolutionContext))
                        .foldLeft[ConstraintSet](EmptyConstrainProblem)(_.merge(_))

                      constraintSet match {
                        case NoSolutionSet(problems) => {
                          problems.foreach(p => context.messageCollector.warning(p._2, p._1))
                        }
                        case _ =>
                      }
                    }
                  case None =>
                }
              case None =>
            }
          case _ =>
        }
      }
    })
  }
}
