package org.mule.weave.v2.parser

import org.mule.weave.v2.api.tooling.ast.DWAstNodeKind
import org.mule.weave.v2.grammar.Identifiers
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.LiteralValueAstNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameter
import org.mule.weave.v2.parser.ast.header.directives.ContentType
import org.mule.weave.v2.parser.ast.header.directives.DataFormatId
import org.mule.weave.v2.parser.ast.header.directives.VersionMajor
import org.mule.weave.v2.parser.ast.header.directives.VersionMinor
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.UriNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

trait ErrorAstNode extends LiteralValueAstNode {
  def message(): Message
}

case class MissingExpressionErrorAstNode(errorMessage: String) extends ErrorAstNode {
  override protected def doClone(): AstNode = MissingExpressionErrorAstNode(errorMessage)

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_EXPRESSION_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_EXPRESSION_ERROR_NODE
}

case class MissingFunctionDefinitionErrorAstNode(errorMessage: String) extends ErrorAstNode {
  override protected def doClone(): AstNode = MissingFunctionDefinitionErrorAstNode(errorMessage)

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_FUNCTION_DEFINITION_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_FUNCTION_DEFINITION_ERROR_NODE
}

case class MissingSelectorExpressionErrorAstNode(errorMessage: String) extends ErrorAstNode {
  override protected def doClone(): AstNode = MissingSelectorExpressionErrorAstNode(errorMessage)

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_SELECTOR_EXPRESSION_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_SELECTOR_EXPRESSION_ERROR_NODE
}

case class MissingTokenErrorAstNode(errorMessage: String, token: String) extends ErrorAstNode {
  override protected def doClone(): AstNode = MissingTokenErrorAstNode(errorMessage, token)

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_TOKEN_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_TOKEN_ERROR_NODE
}

class MissingKeySelector(errorMessage: String) extends NameNode(StringNode(""), None) with ErrorAstNode {
  override def children(): Seq[AstNode] = Seq()

  override protected def doClone(): AstNode = new MissingKeySelector(errorMessage)

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_KEY_SELECTOR_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_KEY_SELECTOR_NODE
}

class InvalidNameIdentifierError(errorMessage: String, prefix: Seq[String]) extends NameIdentifier(prefix.mkString(NameIdentifier.SEPARATOR) + NameIdentifier.SEPARATOR + NameIdentifier.INSERTED_FAKE_VARIABLE_NAME, None) with ErrorAstNode {
  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.INVALID_NAME_IDENTIFIER_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.INVALID_NAME_IDENTIFIER_ERROR
}

class MissingNameIdentifierError(errorMessage: String) extends NameIdentifier(NameIdentifier.INSERTED_FAKE_VARIABLE_NAME, None) with ErrorAstNode {
  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_NAME_IDENTIFIER_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_NAME_IDENTIFIER_NODE
}

class MissingFunctionParameter() extends FunctionParameter(NameIdentifier.INSERTED_FAKE_VARIABLE) with ErrorAstNode {
  override val literalValue: String = "Missing function parameter e.g name: String"

  override def message(): Message = {
    Message(MessageKind.MISSING_FUNCTION_PARAMETER_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_FUNCTION_PARAMETER
}

class InvalidVariableNameIdentifierError(name: String) extends NameIdentifier(name, None) with ErrorAstNode {

  val reason =
    if (Identifiers.keywords.contains(name))
      s"The name `${name}` is a reserved word. List of reserved words: \n - ${Identifiers.keywords.mkString("\n - ")}"
    else
      s"The name `${name}` violates DataWeave naming rules. It should start with a `Alpha` Character followed by zero or more `AlphaNumeric` or `_`"

  override val literalValue: String = name

  override def message(): Message = {
    Message(MessageKind.INVALID_VARIABLE_NAME_IDENTIFIER_MESSAGE_KIND, s"Invalid variable name identifier. Reason: ${reason}", ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.INVALID_VARIABLE_NAME_IDENTIFIER_ERROR_NODE
}

class InvalidFieldNameIdentifierError(name: String) extends NameIdentifier(name, None) with ErrorAstNode {

  val reason =
    if (Identifiers.keywords.contains(name))
      s"The name `${name}` is a reserved word. List of reserved words: \n - ${Identifiers.keywords.mkString("\n - ")}"
    else
      s"The name `${name}` violates DataWeave field naming rules. It should start with a `Alpha` Character followed by zero or more `AlphaNumeric` or `_`"

  override val literalValue: String = name

  override def message(): Message = {
    Message(MessageKind.INVALID_FIELD_NAME_IDENTIFIER_MESSAGE_KIND, s"Invalid field name identifier. Reason: ${reason}. Possible solution wrap `${name}` into quotes e.g payload.'${name}'", ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.INVALID_FIELD_NAME_IDENTIFIER_NODE
}

class MissingFormatDefinition(errorMessage: String) extends ContentType("invalid/mime") with ErrorAstNode {

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_FORMAT_DEFINITION_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_FORMAT_NODE
}

class MissingDataFormatDefinition(errorMessage: String) extends DataFormatId("invalid") with ErrorAstNode {

  override val literalValue: String = errorMessage

  override def message(): Message = {
    Message(MessageKind.MISSING_DATA_FORMAT_DEFINITION_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_DATA_FORMAT
}

class MissingVersionMajor() extends VersionMajor("-1") with ErrorAstNode {
  override val literalValue: String = "Missing Version major"

  override def message(): Message = {
    Message(MessageKind.MISSING_VERSION_MAJOR_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_VERSION_MAJOR_NODE

}

class MissingVersionMinor() extends VersionMinor("-1") with ErrorAstNode {
  override val literalValue: String = "Missing Version minor"

  override def message(): Message = {
    Message(MessageKind.MISSING_VERSION_MINOR_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_VERSION_MINOR_NODE
}

class MissingUri() extends UriNode("invalidUri") with ErrorAstNode {
  override def message(): Message = {
    Message(MessageKind.INVALID_URI_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_URI_NODE
}

class MissingSchemaNode() extends SchemaNode(Seq()) with ErrorAstNode {
  override val literalValue: String = "Missing Schema"

  override def message(): Message = {
    Message(MessageKind.MISSING_SCHEMA_MESSAGE_KIND, literalValue, ParsingPhaseCategory)
  }

  override def getKind(): String = DWAstNodeKind.MISSING_SCHEMA_NODE
}
