package org.mule.weave.v2.mapping

import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.parser.MappingParser
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.FunctionParameters
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.selectors.NullSafeNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.KeyValuePairNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.utils.CacheBuilder

import scala.annotation.tailrec
import scala.collection.mutable

class DataMappingLoader(val parsingContext: ParsingContext) {

  private val implicitInputs = mutable.ArrayBuffer[String]()

  def addImplicitInput(name: String): DataMappingLoader = {
    this.implicitInputs.+=(name)
    this
  }

  def inferAssignments(weaveResource: WeaveResource): Seq[FieldAssignment] = {
    implicitInputs.foreach((implicitInput) => {
      parsingContext.addImplicitInput(implicitInput, None)
    })
    val parser = MappingParser.parse(MappingParser.scopePhase(), weaveResource, parsingContext)
    val result = parser.getResult()

    val documentNode = result.astNode
    val scopeNav = result.scope
    val resolver = new SourcePathResolver(scopeNav)
    val mapping = DataMapping(None, NamePathElement.fromString("/"), NamePathElement.fromString("/"))
    inferAssignments(documentNode.root, mapping, "", "", "")(scopeNav, resolver).apply(mapping)
    val assignments = collectAssignments(mapping)
    assignments
  }

  def collectAssignments(mapping: DataMapping): Seq[FieldAssignment] = {
    mapping.fieldAssignments().toSeq ++ mapping.childMappings().flatMap(collectAssignments).toSeq
  }

  /**
    * Returns the action that the parent should apply to the mapping
    */
  def inferAssignments(node: AstNode, mapping: DataMapping, currTargetPath: String, sourceSuffix: String, currSourcePath: String)(implicit scopesNavigator: ScopesNavigator, resolver: SourcePathResolver): MappingAction = {
    node match {
      case doc: DocumentNode =>
        val action = inferAssignments(doc.root, mapping, currTargetPath, sourceSuffix, currSourcePath)
        action.apply(mapping)
        NopAction

      case kvp: KeyValuePairNode =>
        val target = getKeyName(kvp.key)
        val newTargetPath = s"$currTargetPath/$target"
        val action = inferAssignments(kvp.value, mapping, newTargetPath, sourceSuffix, currSourcePath)
        action.apply(mapping)
        NopAction

      case BinaryOpNode(ValueSelectorOpId, lhs, NameNode(StringNode(name, _), _, _), _) =>
        val newSourceSuffix = s"/$name$sourceSuffix"
        inferAssignments(lhs, mapping, currTargetPath, newSourceSuffix, currSourcePath)

      case ref @ VariableReferenceNode(NameIdentifier(name, _), _) =>
        val sourcePath = resolver.resolve(ref, currSourcePath, sourceSuffix)
        (mapping: DataMapping) => {
          mapping.addArrow(NamePathElement.fromString(sourcePath), NamePathElement.fromString(currTargetPath), None, None)
        }

      case FunctionCallNode(VariableReferenceNode(NameIdentifier("map", _), _), FunctionCallParametersNode(args), _, _) =>
        val lhs = args(0)
        val rhs = args(1)
        val source = resolver.resolve(lhs, currSourcePath, sourceSuffix) + "/[]"
        val target = s"$currTargetPath/[]"
        val childMapping = mapping.findOrCreateInnerMapping(NamePathElement.fromString(source), NamePathElement.fromString(target))
        inferAssignments(rhs, childMapping, target, sourceSuffix, source)

      case x =>
        for (child <- x.children()) {
          val action = inferAssignments(child, mapping, currTargetPath, sourceSuffix, currSourcePath)
          action.apply(mapping)
        }
        NopAction
    }
  }

  trait MappingAction {
    def apply(mapping: DataMapping): Unit
  }

  object NopAction extends MappingAction {
    override def apply(mapping: DataMapping): Unit = {}
  }

  private def getKeyName(keyValue: AstNode): String = {
    keyValue match {
      case KeyNode(StringNode(str, _), _, _, _) => str
      case _                                    => ""
    }
  }
}

class SourcePathResolver(scopesNavigator: ScopesNavigator) {
  private val astNavigator: AstNavigator = scopesNavigator.rootScope.astNavigator()

  //TODO: use this instead of passing the sourcePath as a string
  private val cache = CacheBuilder
    .apply((key: (AstNode, String, String)) => doResolve(key._1, key._2, key._3))
    .maximumSize(100)
    .build()

  def resolve(node: AstNode, sourcePathPrefix: String, sourcePathSuffix: String): String = {
    val resolvedPath = cache.get(node, sourcePathPrefix, sourcePathSuffix)
    resolvedPath
  }

  @tailrec
  private def doResolve(node: AstNode, sourcePathPrefix: String = "", sourcePathSuffix: String = ""): String = {
    node match {
      case NullSafeNode(x, _) => doResolve(x, sourcePathPrefix, sourcePathSuffix)
      case BinaryOpNode(ValueSelectorOpId, lhs, NameNode(StringNode(name, _), _, _), _) =>
        val suffix = s"/$name$sourcePathSuffix"
        doResolve(lhs, sourcePathPrefix, suffix)
      case ref @ VariableReferenceNode(NameIdentifier(name, _), _) =>
        val maybeRefScope = scopesNavigator.scopeOf(ref)
        val maybeRefParentScope = maybeRefScope.flatMap(_.parentScope)
        val declaration = scopesNavigator.resolveReference(ref).get
        val maybeDeclScope = scopesNavigator.scopeOf(declaration)

        val isLambda = maybeDeclScope.exists(x => isLambdaArg(x.astNode, name))

        if (maybeDeclScope.isDefined && (maybeRefScope == maybeDeclScope || maybeRefParentScope == maybeDeclScope)) {
          //if it's declared in this scope use the source prefix
          val resolvedPath = if (isLambda) s"$sourcePathPrefix$sourcePathSuffix" else s"$sourcePathPrefix/$name$sourcePathSuffix"
          resolvedPath
        } else {
          //else go up the AST resolving the path until the root is reached
          val suffix = if (isLambda) sourcePathSuffix else s"$name$sourcePathSuffix"
          resolveVariablePath(declaration, suffix)
        }
      case _ =>
        throw new RuntimeException(s"Unable to handle ${node.getClass}")
    }
  }

  private def isLambdaArg(node: AstNode, name: String): Boolean = {
    node match {
      case FunctionNode(FunctionParameters(params), _, _, _) if params.exists(_.variable.name == name) =>
        true
      case _ =>
        false
    }
  }

  private def resolveVariablePath(node: AstNode, sourcePathSuffix: String = ""): String = {
    //Go up the AST resolving the path until the root is reached
    node match {
      case _: DocumentNode => sourcePathSuffix
      case FunctionCallNode(VariableReferenceNode(NameIdentifier("map", _), _), FunctionCallParametersNode(args), _, _) =>
        val lhs = args.head
        //TODO: add array and source path of lhs
        doResolve(lhs, "", s"/[]/$sourcePathSuffix")
      case _ =>
        val parent = astNavigator.parentOf(node).get //there's always a parent unless it's the root
        resolveVariablePath(parent, sourcePathSuffix)
    }
  }

}

