package org.mule.weave.v2.grammar.location

import org.mule.weave.v2.parser.SafeStringBasedParserInput
import org.mule.weave.v2.parser.annotation.EnclosedMarkAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.WeaveLocationCapable
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.location.ParserPosition
import org.mule.weave.v2.parser.location.Position
import org.mule.weave.v2.parser.location.WeaveLocation
import org.parboiled2.Parser

trait PositionTracking {
  this: PositionTracking with Parser =>

  def pushPosition = rule {
    push(ParserPosition(cursor, stringInput))
  }

  def stringInput: SafeStringBasedParserInput = {
    input.asInstanceOf[SafeStringBasedParserInput]
  }

  def injectPosition[A] = rule {
    run { (startPos: ParserPosition, node: A) =>
      {
        node match {
          case pn: WeaveLocationCapable =>
            pn._location = Some(createWeaveLocation(startPos))
          case Some(pn: WeaveLocationCapable) =>
            pn._location = Some(createWeaveLocation(startPos))
        }
        node
      }
    }
  }

  def createWeaveLocation(startPos: ParserPosition): WeaveLocation = {
    WeaveLocation(startPos, ParserPosition(cursor, stringInput), resourceName)
  }

  def injectUnaryPosition[A <: AstNode] = rule {
    run { (node: A) =>
      doInjectBinaryPosition(node)
    }
  }

  protected def resolveStartPosition(astNode: AstNode): Position = {
    val maybeEnclosedMarkAnnotations = astNode.annotationsBy(classOf[EnclosedMarkAnnotation]).sortBy(_.location.startPosition.index)(Ordering[Int].reverse).headOption
    val position = if (maybeEnclosedMarkAnnotations.nonEmpty) {
      maybeEnclosedMarkAnnotations.get.location.startPosition
    } else {
      astNode.location().startPosition
    }
    position
  }

  protected def doInjectBinaryPosition[A <: AstNode](node: A) = {
    node match {
      case pn: WeaveLocationCapable =>
        pn._location = Some(WeaveLocation(node.children().head.location().startPosition, ParserPosition(cursor, stringInput), resourceName))
    }
    node
  }

  def injectBinaryPosition[A <: AstNode] = rule {
    run { (node: A) =>
      {
        node match {
          case pn: WeaveLocationCapable =>
            pn._location = Some(WeaveLocation(node.children().head.location().startPosition, ParserPosition(cursor, stringInput), resourceName))
        }
        node
      }
    }
  }

  def resourceName: NameIdentifier

}
