package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.literals.BaseExpression
import org.mule.weave.v2.grammar.literals.Literals
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.grammar.structure.Attributes
import org.mule.weave.v2.parser.annotation.BracketSelectorAnnotation
import org.mule.weave.v2.parser.annotation.DescendantsIncludeThisAnnotation
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.operators.UnaryOpNode
import org.mule.weave.v2.parser.ast.selectors.ExistsSelectorNode
import org.mule.weave.v2.parser.ast.selectors.NullSafeNode
import org.mule.weave.v2.parser.ast.selectors.NullUnSafeNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.parboiled2._
import org.parboiled2.support.hlist.::
import org.parboiled2.support.hlist.HNil

import scala.annotation.switch

trait Selectors extends PositionTracking with Tokens with Functions with Literals with Attributes with BaseExpression with ErrorRecovery {
  this: Selectors with Grammar =>

  def selectors = rule {
    ws ~ oneOrMore(ws ~ (baseSelector | functionCall)) ~ optional(quiet(selectorModifier))
  }

  def selectorModifier = rule {
    (run {
      (cursorChar: @switch) match {
        case '?' => keyExistsModifier
        case '!' => nullUnSafeModifier
        case _   => nullSafeModifier
      }
    }) ~ injectBinaryPosition
  }

  def baseSelector: Rule[AstNode :: HNil, AstNode :: HNil] = rule {
    (run {
      (cursorChar: @switch) match {
        case '.' => dotSelector
        case '[' => bracketSelector
        case _   => MISMATCH
      }
    }) ~ injectBinaryPosition
  }

  def dotSelector = rule {
    dot ~!~ baseDotSelector
  }

  def baseDotSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Selector Identifier") {
    (run {
      (cursorChar: @switch) match {
        case '.' => descendantsSelector
        case '@' => arrayAttributeSelector | attributeSelector | allAttributeSelector
        case '^' => schemaSelector | allSchemaSelector
        case '#' => namespaceSelector
        case '&' => objectKeyFilter
        case _   => arrayValueSelector | arrayAttributeSelectorAlternative | valueSelector
      }
    })
  }

  def descendantChildSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Selector Identifier") {
    (run {
      (cursorChar: @switch) match {
        case '@' => (arrayAttributeSelector | attributeSelector | allAttributeSelector) ~> markDescendantIncludeThisSelector
        case '^' => (schemaSelector | allSchemaSelector) ~> markDescendantIncludeThisSelector
        case '#' => namespaceSelector ~> markDescendantIncludeThisSelector
        case '&' => objectKeyFilter ~> markDescendantIncludeThisSelector
        case '[' => bracketSelector
        case _   => (arrayValueSelector | valueSelectorWithoutRecovery) ~> markDescendantIncludeThisSelector
      }
    })
  }

  def baseBracketSelector: Rule[AstNode :: HNil, AstNode :: HNil] = rule {
    ((run {
      (cursorChar: @switch) match {
        case '@'        => attributeSelector | arrayAttributeSelector | allAttributeSelector
        case '?'        => filterSelector
        case '"' | '\'' => valueSelector
        case '&'        => objectKeyDynamicSelector
        case '*'        => arrayValueSelector | arrayAttributeSelectorAlternative | dynamicArrayValueSelector
        case _          => dynamicSelector | valueSelector
      }
    }) ~> markBracketSelector)
  }

  def descendantsSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Descendant Selector e.g `payload..fieldName`") {
    standaloneDescendantsSelector ~ optional(descendantChildSelector)
  }

  def standaloneDescendantsSelector: Rule[AstNode :: HNil, Nothing :: HNil] = rule {
    ch('.') ~> createDescendantsSelectorNode ~ injectUnaryPosition
  }

  def nonStandaloneDescendantsSelector = rule {
    &(ch('.')) ~> createDescendantsSelectorNode
  }

  val createDescendantsSelectorNode = (selectable: AstNode) => {
    UnaryOpNode(DescendantsSelectorOpId, selectable) :: HNil
  }

  def allAttributeSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("All Attribute Selector e.g `payload.@`") {
    ch('@') ~> createAllAttributeSelectorNode ~ injectUnaryPosition
  }

  val createAllAttributeSelectorNode = (selectable: AstNode) => {
    UnaryOpNode(AllAttributesSelectorOpId, selectable) :: HNil
  }

  def allSchemaSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("All Metadata Selector e.g `payload.^`") {
    (ch('^') ~> createAllSchemaSelectorNode) ~ injectUnaryPosition
  }

  def namespaceSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Namespace Selector e.g `payload.#`") {
    (ch('#') ~> createNamespaceSelectorNode) ~ injectUnaryPosition
  }

  def objectKeyFilter: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Object Key Pair Selector e.g `payload.&fieldName`") {
    ch('&') ~ ws ~ fieldName ~> createObjectKeyFilterNode
  }

  def objectKeyDynamicSelector: Rule[AstNode :: HNil, AstNode :: HNil] = rule {
    ch('&') ~ expr ~> createObjectKeyFilterNode
  }

  val createObjectKeyFilterNode = (selectable: AstNode, name: AstNode) => {
    BinaryOpNode(ObjectKeyValueSelectorOpId, selectable, name) :: HNil
  }

  val createAllSchemaSelectorNode = (selectable: AstNode) => {
    UnaryOpNode(AllSchemaSelectorOpId, selectable) :: HNil
  }

  val createNamespaceSelectorNode = (selectable: AstNode) => {
    UnaryOpNode(NamespaceSelectorOpId, selectable) :: HNil
  }

  def attributeSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Attribute Value Selector e.g `payload.@attributeName`") {
    ch('@') ~ !ch('*') ~ attributeNameWithoutAnnotations ~> createAttributeSelectorNode
  }

  def arrayAttributeSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("MultiValue Attribute Selector e.g `payload.@*attributeName`") {
    ch('@') ~ ch('*') ~!~ attributeNameWithoutAnnotations ~> createArrayAttributeSelectorNode
  }

  def arrayAttributeSelectorAlternative: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("MultiValue Attribute Selector e.g `payload.*@attributeName`") {
    ch('*') ~ ch('@') ~!~ attributeNameWithoutAnnotations ~> createArrayAttributeSelectorNode
  }

  val createAttributeSelectorNode = (selectable: AstNode, name: NameNode) => {
    BinaryOpNode(AttributeValueSelectorOpId, selectable, name) :: HNil
  }

  val createArrayAttributeSelectorNode = (selectable: AstNode, name: NameNode) => {
    BinaryOpNode(MultiAttributeValueSelectorOpId, selectable, name) :: HNil
  }

  def schemaSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Metadata Selector e.g `payload.^metadataFieldName`") {
    ch('^') ~ attributeNameWithoutAnnotations ~> createSchemaSelectorNode
  }

  val createSchemaSelectorNode = (selectable: AstNode, name: NameNode) => {
    BinaryOpNode(SchemaValueSelectorOpId, selectable, name) :: HNil
  }

  def valueSelectorWithoutRecovery: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Value Selector e.g `payload.fieldName`") {
    (fieldNameWithoutAnnotations) ~> createValueSelectorNode
  }

  def valueSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Value Selector e.g `payload.fieldName`") {
    (fieldNameWithoutAnnotations | missingSelectorName("Missing Field Name for Selector Expression.")) ~> createValueSelectorNode
  }

  def arrayValueSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("MultiValue Selector e.g `payload.*fieldName`") {
    starToken ~ fieldNameWithoutAnnotations ~> createArrayValueSelectorNode
  }

  val createValueSelectorNode = (selectable: AstNode, name: AstNode) => {
    BinaryOpNode(ValueSelectorOpId, selectable, name) :: HNil
  }

  val createArrayValueSelectorNode = (selectable: AstNode, name: AstNode) => {
    BinaryOpNode(MultiValueSelectorOpId, selectable, name) :: HNil
  }

  val createDynamicArrayValueSelectorNode = (selectable: AstNode, name: AstNode) => {
    BinaryOpNode(MultiValueSelectorOpId, selectable, name) :: HNil
  }

  def bracketSelector = rule {
    squareBracketOpen ~!~ baseBracketSelector ~ (squareBracketEnd | fail("']' for the selector expression."))
  }

  val markBracketSelector: AstNode => AstNode = (node: AstNode) => {
    node.annotate(BracketSelectorAnnotation())
    node
  }

  val markDescendantIncludeThisSelector: AstNode => AstNode = (node: AstNode) => {
    node.children().head.annotate(DescendantsIncludeThisAnnotation())
    node
  }

  def filterSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Filter selector e.g `payload[?$.fieldName == 'condition']`") {
    quiet(ch('?')) ~ (filterSelectorLambda) ~> createFilterSelectorNode
  }

  val createFilterSelectorNode = (selectable: AstNode, fn: FunctionNode) => {
    BinaryOpNode(FilterSelectorOpId, selectable, fn) :: HNil
  }

  def dynamicSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Dynamic selector e.g `payload[vars.fieldName]`") {
    expr ~ !ch('#') ~> createDynamicSelectorNode
  }

  def dynamicArrayValueSelector: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Dynamic MultiValue Selector e.g `payload[*vars.fieldName]`") {
    atomic(ch('*') ~ ws ~ expr ~ !ch('#')) ~> createDynamicArrayValueSelectorNode
  }

  val createDynamicSelectorNode = (selectable: AstNode, expr: AstNode) => {
    BinaryOpNode(DynamicSelectorOpId, selectable, expr) :: HNil
  }

  def keyExistsModifier: Rule[AstNode :: HNil, AstNode :: HNil] = namedRule("Key Exists e.g `payload.fieldName?`") {
    quiet(questionMark) ~> createExistsSelectorNode
  }

  val createExistsSelectorNode = (selectable: AstNode) => {
    ExistsSelectorNode(selectable) :: HNil
  }

  def nullSafeModifier: Rule[AstNode :: HNil, NullSafeNode :: HNil] = rule {
    !exclamationMark ~> createNullSafeNode
  }

  val createNullUnSafeNode = (selectable: AstNode) => {
    NullUnSafeNode(selectable)
  }

  val createNullSafeNode = (selectable: AstNode) => {
    NullSafeNode(selectable)
  }

  def nullUnSafeModifier: Rule[AstNode :: HNil, AstNode :: HNil] = rule {
    exclamationMark ~> createNullUnSafeNode
  }

  val passthru = (selectable: AstNode) => {
    selectable :: HNil
  }

}
