package org.mule.weave.v2.grammar

import org.mule.weave.v2.grammar.literals.Literals
import org.mule.weave.v2.grammar.location.PositionTracking
import org.mule.weave.v2.grammar.structure.Attributes
import org.mule.weave.v2.grammar.structure.Namespaces
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.patterns._
import org.mule.weave.v2.parser.ast.structure.RegexNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.parboiled2.Rule1

trait Patterns extends PositionTracking with Tokens with Variables with Selectors with Literals with Namespaces with Attributes {
  this: Patterns with Grammar =>

  val createPatternsNode = (patterns: Seq[PatternExpressionNode]) => {
    PatternExpressionsNode(patterns)
  }

  val createRegexPattern = (rx: AstNode, fn: AstNode) => {
    RegexPatternNode(rx, NameIdentifier.$, fn)
  }

  val createNamedRegexPattern = (name: NameIdentifier, rx: RegexNode, fn: AstNode) => {
    RegexPatternNode(rx, name, fn)
  }

  val createTypePattern = (t: AstNode, fn: AstNode) => {
    TypePatternNode(t, NameIdentifier.$, fn)
  }

  val createNamedTypePattern = (name: NameIdentifier, t: AstNode, fn: AstNode) => {
    TypePatternNode(t, name, fn)
  }

  val createLiteralPattern = (l: AstNode, fn: AstNode) => {
    LiteralPatternNode(l, NameIdentifier.$, fn)
  }

  val createNamedLiteralPattern = (name: NameIdentifier, l: AstNode, fn: AstNode) => {
    LiteralPatternNode(l, name, fn)
  }

  val createNamedExpressionPattern = (name: NameIdentifier, expr: AstNode, fn: AstNode) => {
    ExpressionPatternNode(expr, name, fn)
  }

  val createDefaultPattern = (variableName: Option[NameIdentifier], fn: AstNode) => {
    DefaultPatternNode(fn, variableName.getOrElse(NameIdentifier.$))
  }

  val createEmptyArrayPattern = (fn: AstNode) => {
    EmptyArrayPatternNode(fn)
  }

  val createDeconstructArrayPattern = (head: NameIdentifier, tail: NameIdentifier, fn: AstNode) => {
    DeconstructArrayPatternNode(head, tail, fn)
  }

  val createEmptyObjectPattern = (fn: AstNode) => {
    EmptyObjectPatternNode(fn)
  }

  val createDeconstructObjectPattern = (headKey: NameIdentifier, headValue: NameIdentifier, tail: NameIdentifier, fn: AstNode) => {
    DeconstructObjectPatternNode(headKey, headValue, tail, fn)
  }

  def patterns: Rule1[PatternExpressionsNode] = rule {
    pushPosition ~ (curlyBracketsStart ~!~ zeroOrMore(pattern).separatedBy(quiet(oneOrMore(comment | whiteSpaceOrNewLineChar))) ~!~ (curlyBracketsEnd | fail("'}' for the Pattern Matching.")) ~> createPatternsNode) ~ injectPosition
  }

  def pattern: Rule1[PatternExpressionNode] = rule {
    pushPosition ~ ((caseKeyword ~!~ (regexPattern | namedRegexPattern | typePattern | namedTypePattern | literalPattern | namedLiteralPattern | namedExpressionPattern | deconstructPattern)) | defaultPattern) ~ injectPosition
  }

  def deconstructPattern: Rule1[PatternExpressionNode] = rule {
    emptyArrayPattern | deconstructArrayPattern | deconstructObjectPattern | emptyObjectPattern
  }

  def emptyArrayPattern: Rule1[EmptyArrayPatternNode] = rule {
    (squareBracketOpen ~ squareBracketEnd ~ lambdaMark ~!~ expr ~> createEmptyArrayPattern)
  }

  def deconstructArrayPattern: Rule1[DeconstructArrayPatternNode] = rule {
    (squareBracketOpen ~ namedVariable ~ tilde ~ namedVariable ~ squareBracketEnd ~ lambdaMark ~!~ expr ~> createDeconstructArrayPattern)
  }

  def emptyObjectPattern = rule {
    (curlyBracketsStart ~ curlyBracketsEnd ~ lambdaMark ~!~ expr ~> createEmptyObjectPattern)
  }

  def deconstructObjectPattern: Rule1[DeconstructObjectPatternNode] = rule {
    (curlyBracketsStart ~ namedVariable ~ ws ~ objFieldSep ~ namedVariable ~ ws ~ tilde ~ namedVariable ~ curlyBracketsEnd ~ lambdaMark ~!~ expr ~> createDeconstructObjectPattern)
  }

  def regexPattern: Rule1[PatternExpressionNode] = rule {
    (matchesKeyword ~ regexLiteral ~ lambdaMark ~!~ expr ~> createRegexPattern)
  }

  def namedRegexPattern: Rule1[PatternExpressionNode] = rule {
    (namedVariable ~ ws ~ matchesKeyword ~ regexLiteral ~ lambdaMark ~!~ expr ~> createNamedRegexPattern)
  }

  def typePattern: Rule1[PatternExpressionNode] = rule {
    (isKeyword ~ typeExpression ~ lambdaMark ~!~ expr ~> createTypePattern)
  }

  def namedTypePattern: Rule1[PatternExpressionNode] = rule {
    (namedVariable ~ fws ~ isKeyword ~ typeExpression ~ lambdaMark ~!~ expr ~> createNamedTypePattern)
  }

  def literalPattern: Rule1[PatternExpressionNode] = rule {
    ((number | anyDateLiteral | trueLiteral | falseLiteral | nullLiteral | string) ~ lambdaMark ~!~ expr ~> createLiteralPattern)
  }

  def namedLiteralPattern: Rule1[PatternExpressionNode] = rule {
    (namedVariable ~ ws ~ objFieldSep ~ (number | anyDateLiteral | trueLiteral | falseLiteral | string) ~ lambdaMark ~!~ expr ~> createNamedLiteralPattern)
  }

  def namedExpressionPattern: Rule1[PatternExpressionNode] = rule {
    (namedVariable ~ fws ~ ifKeyword ~!~ ((ws ~ ifCondition) | (fws ~ expr)) ~ lambdaMark ~!~ expr ~> createNamedExpressionPattern)
  }

  def defaultPattern: Rule1[PatternExpressionNode] = rule {
    pushPosition ~ (elseKeyword ~ optional(nameIdentifierNode) ~ lambdaMark ~!~ expr ~> createDefaultPattern) ~ injectPosition
  }
}
