package org.mule.weave.v2.editor.quickfix

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.completion.Template
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.WeaveTextDocument
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.DirectiveNode
import org.mule.weave.v2.parser.ast.header.directives.OutputDirective
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode

abstract class TemplateBasedQuickFixAction extends QuickFixAction {

  override def run(document: WeaveTextDocument): Unit = {
    val maybeDocument = editorSupport.astDocument()
    maybeDocument match {
      case Some(astDocument) => {
        astDocument match {
          case mn: ModuleNode => {
            doRunExtractTemplate(document, mn.directives, true)
          }
          case ast: DocumentNode => {
            val header = ast.header
            if (AstNodeHelper.notInjectedNode(header)) {
              doRunExtractTemplate(document, header.directives, false)
            } else {
              doRunExtractTemplate(document, Seq(), false)
            }
          }
        }
      }
      case None =>
    }
  }

  protected def createTemplate(): Template

  val editorSupport: WeaveEditorSupport

  /**
    * The element where the template is going to be executed
    */
  val element: AstNode

  private def doRunExtractTemplate(document: WeaveTextDocument, directives: Seq[DirectiveNode], isModule: Boolean): Unit = {
    val lastOption = directives
      .filter(AstNodeHelper.notInjectedNode)
      .filter((d) => !d.isInstanceOf[OutputDirective])
      .lastOption
    val index = editorSupport.astNavigator().flatMap(astNavigator => {
      val mayBeParent = astNavigator.parentWithType(element, classOf[DirectiveNode])
      mayBeParent.map((parent) => {
        parent.location().startPosition.index
      })
    }).orElse(
      lastOption.map(_.location().endPosition.index + 1))
    val template: Template = createTemplate()
    if (index.isEmpty && !isModule) {
      template.prepend("%dw 2.0\n")
      template.addnl("---")
    }
    document.runTemplate(template, index.getOrElse(0))
  }

}
