package org.mule.weave.v2.editor

import org.mule.weave.v2.editor.VirtualFileSystem.pathToUrl
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.sdk.ChainedWeaveResourceResolver
import org.mule.weave.v2.sdk.NameIdentifierHelper
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.sdk.WeaveResourceResolver

import java.util
import java.util.Collections
import java.util.UUID
import scala.collection.JavaConverters.asJavaIterableConverter
import scala.collection.JavaConverters.asJavaIteratorConverter
import scala.collection.JavaConverters.asScalaIteratorConverter
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

/**
  * Virtual File System used by the IDE to interact with the tooling services.
  */
trait VirtualFileSystem {

  //TODO change this to start using url
  /**
    * Returns the virtual file for this path. And return null if not found
    *
    * @param path
    * @return
    */
  def file(path: String): VirtualFile

  /**
    * Removes the change listener
    *
    * @param listener
    */
  def removeChangeListener(listener: ChangeListener): Unit

  /**
    * Register change a change listener
    *
    * @param cl
    */
  def changeListener(cl: ChangeListener)

  /**
    * Triggers the change of a file with the given path. Internal use only
    *
    * @param vf
    */
  def onChanged(vf: VirtualFile): Unit

  /**
    * Return a WeaveResourceResolver based on this Virtual File System
    *
    * @return
    */
  def asResourceResolver: WeaveResourceResolver = new VirtualFSResourceProvider(this)

  /**
    * Returns all the files
    * @return
    */
  def listFiles(): java.util.Iterator[VirtualFile] = {
    Collections.emptyIterator()
  }

}

object VirtualFileSystem {
  val VIRTUAL_SCHEME = "virtual:"

  def pathToUrl(path: String): String = {
    VIRTUAL_SCHEME + (if (path.startsWith("/")) path else ("/" + path))
  }
}

/**
  * A view of a File. This has all the required things that we need of a File implement our tooling API
  */
trait VirtualFile {

  /**
    * Returns the Virtual File System this file belongs to
    *
    * @return
    */
  def fs(): VirtualFileSystem

  /**
    * Returns the content of this File
    *
    * @return
    */
  def read(): String

  /**
    * Writes the content to the file
    *
    * @param content The new content
    * @return True if the content was written
    */
  def write(content: String): Boolean

  /**
    * Returns true if this file is read only
    *
    * @return True if the file is read only
    */
  def readOnly(): Boolean

  /**
    * The url of this file.
    * I.E file://a/b.dwl or virtual:/foo/bar.dwl for in memory files¬
    *
    * @return The URL String
    */
  def url(): String

  /**
    * Returns the path of this file.
    * This path doesn't have the scheme is only the path part
    *
    * @return the path of this file
    */
  def path(): String

  /**
    * Returns the WeaveResource representation of this virtual file. A WeaveResource is just a Url and a Content
    *
    * @return The WeaveResource representation of this file
    */
  def asResource(): WeaveResource = WeaveResource(this)

  /**
    * Returns the name identifier of this file
    *
    * @return
    */
  def getNameIdentifier: NameIdentifier

  override def toString: String = url()
}

object EmptyVirtualFileSystem extends SimpleVirtualFileSystem(mutable.Map[String, String]())

class SimpleVirtualFileSystem(val fs: mutable.Map[String, String]) extends VirtualFileSystem {

  val listeners: ArrayBuffer[ChangeListener] = ArrayBuffer[ChangeListener]()

  override def file(path: String): VirtualFile = {
    fs.get(path)
      .map((_) => {
        new SimpleVirtualFile(path, this)
      })
      .orNull
  }

  override def changeListener(cl: ChangeListener): Unit = {
    listeners.+=(cl)
  }

  override def onChanged(vf: VirtualFile): Unit = {
    listeners.foreach((cl) => cl.onChanged(vf))
  }

  override def removeChangeListener(service: ChangeListener): Unit = {
    listeners.remove(listeners.indexOf(service))
  }

  override def listFiles(): util.Iterator[VirtualFile] = {
    fs.toIterator
      .map((entry) => file(entry._1))
      .asJava
  }

  def updateContent(path: String, content: String): Unit = {
    fs.update(path, content) //Update the content
    onChanged(file(path)) //trigger changes
  }
}

object SimpleVirtualFileSystem {
  def apply(fs: Map[String, String]): SimpleVirtualFileSystem = new SimpleVirtualFileSystem(mutable.Map(fs.toSeq: _*))
}

class ReadOnlyVirtualFile(val path: String, var content: String, val fs: VirtualFileSystem) extends VirtualFile {

  override def read(): String = content

  override def write(content: String): Boolean = {
    false
  }

  override def readOnly(): Boolean = true

  override def url(): String = {
    pathToUrl(path)
  }

  override def getNameIdentifier: NameIdentifier = {
    NameIdentifierHelper.fromWeaveFilePath(path)
  }
}

class SimpleVirtualFile(val path: String, val fs: SimpleVirtualFileSystem) extends VirtualFile {

  override def read(): String = {
    fs.fs(path)
  }

  override def write(content: String): Boolean = {
    fs.fs.put(path, content)
    true
  }

  override def readOnly(): Boolean = true

  override def url(): String = {
    pathToUrl(path)
  }

  override def getNameIdentifier: NameIdentifier = {
    NameIdentifierHelper.fromWeaveFilePath(path)
  }
}

/**
  * Virtual File System Changes Listener
  */
trait ChangeListener {

  /**
    * Triggered when a file was deleted
    *
    * @param vf The file that was deleted
    */
  def onDeleted(vf: VirtualFile): Unit = {}

  /**
    * Triggered when a file was modified
    *
    * @param vf The file that was modified
    */
  def onChanged(vf: VirtualFile): Unit = {}

  /**
    * When a new file was created
    *
    * @param vf The new file
    */
  def onCreated(vf: VirtualFile): Unit = {}
}

class VirtualFSResourceProvider(vfs: VirtualFileSystem) extends WeaveResourceResolver {

  override def resolve(name: NameIdentifier): Option[WeaveResource] = {
    val path = NameIdentifierHelper.toWeaveFilePath(name)
    resolvePath(path)
  }

  override def resolvePath(path: String): Option[WeaveResource] = {
    val virtualFile: VirtualFile = vfs.file(path)
    Option(virtualFile).map((vfs) => {
      vfs.asResource()
    })
  }

  override def resolveAll(name: NameIdentifier): Seq[WeaveResource] = {
    resolve(name).toSeq
  }

}

object VirtualFSResourceProvider {
  def apply(vfs: VirtualFileSystem): VirtualFSResourceProvider = new VirtualFSResourceProvider(vfs)
}

class CompositeFileSystem(val fs: VirtualFileSystem, val parent: VirtualFileSystem) extends VirtualFileSystem {
  override def file(path: String): VirtualFile = {
    val file = fs.file(path)
    if (file != null) {
      file
    } else {
      parent.file(path)
    }
  }

  override def changeListener(cl: ChangeListener): Unit = {
    fs.changeListener(cl)
    parent.changeListener(cl)
  }

  override def onChanged(virtualFile: VirtualFile): Unit = {
    fs.onChanged(virtualFile)
    parent.onChanged(virtualFile)
  }

  override def removeChangeListener(service: ChangeListener): Unit = {
    fs.removeChangeListener(service)
    parent.removeChangeListener(service)
  }

  override def asResourceResolver: WeaveResourceResolver = {
    new ChainedWeaveResourceResolver(Seq(fs.asResourceResolver, parent.asResourceResolver))
  }

  override def listFiles(): util.Iterator[VirtualFile] = {
    Seq(fs, parent).toIterator
      .flatMap(_.listFiles().asScala)
      .asJava
  }
}

object CompositeFileSystem {
  def apply(fs: VirtualFileSystem, parent: VirtualFileSystem): CompositeFileSystem = new CompositeFileSystem(fs, parent)
}

class ModularizeFileSystem(modules: mutable.Map[String, VirtualFileSystem] = new mutable.HashMap()) extends VirtualFileSystem {

  override def file(path: String): VirtualFile = {
    modules
      .find(_._2.file(path) != null)
      .map(_._2.file(path))
      .orNull
  }

  def removeModule(name: String): Boolean = {
    modules.remove(name).isDefined
  }

  def addModule(name: String, virtualFileSystem: VirtualFileSystem): Unit = {
    modules.update(name, virtualFileSystem)
  }

  def getModule(name: String): VirtualFileSystem = {
    modules.get(name).orNull
  }

  def getModules(): mutable.Map[String, VirtualFileSystem] = {
    modules
  }

  override def changeListener(cl: ChangeListener): Unit = {
    modules.foreach(_._2.changeListener(cl))
  }

  override def onChanged(virtualFile: VirtualFile): Unit = {
    modules.foreach(_._2.onChanged(virtualFile))
  }

  override def removeChangeListener(service: ChangeListener): Unit = {
    modules.foreach(_._2.removeChangeListener(service))
  }

  override def listFiles(): util.Iterator[VirtualFile] = {
    modules.values
      .flatMap(_.listFiles().asScala)
      .toIterator
      .asJava

  }
}
