package org.mule.weave.v2.completion

import org.mule.weave.v2.WeaveEditorSupport
import org.mule.weave.v2.completion.AutoCompletionService.INSERTED_TEXT
import org.mule.weave.v2.editor.QuickFixAction
import org.mule.weave.v2.editor.indexing.DocumentKind
import org.mule.weave.v2.editor.indexing.EmptyIndexService
import org.mule.weave.v2.editor.indexing.LocatedResult
import org.mule.weave.v2.editor.indexing.WeaveDocument
import org.mule.weave.v2.editor.indexing.WeaveIndexService
import org.mule.weave.v2.editor.quickfix.DeclareNamespaceDeclarationAction
import org.mule.weave.v2.grammar.AllAttributesSelectorOpId
import org.mule.weave.v2.grammar.AllSchemaSelectorOpId
import org.mule.weave.v2.grammar.AttributeValueSelectorOpId
import org.mule.weave.v2.grammar.DescendantsSelectorOpId
import org.mule.weave.v2.grammar.DynamicSelectorOpId
import org.mule.weave.v2.grammar.MultiAttributeValueSelectorOpId
import org.mule.weave.v2.grammar.MultiValueSelectorOpId
import org.mule.weave.v2.grammar.ObjectKeyValueSelectorOpId
import org.mule.weave.v2.grammar.SchemaValueSelectorOpId
import org.mule.weave.v2.grammar.ValueSelectorOpId
import org.mule.weave.v2.parser.DocumentParser
import org.mule.weave.v2.parser.ErrorAstNode
import org.mule.weave.v2.parser.MissingExpressionErrorAstNode
import org.mule.weave.v2.parser.MissingFormatDefinition
import org.mule.weave.v2.parser.MissingNameIdentifierError
import org.mule.weave.v2.parser.MissingSelectorExpressionErrorAstNode
import org.mule.weave.v2.parser.MissingTokenErrorAstNode
import org.mule.weave.v2.parser.MissingVersionMajor
import org.mule.weave.v2.parser.MissingVersionMinor
import org.mule.weave.v2.parser.ast.AstNode
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.CommentNode
import org.mule.weave.v2.parser.ast.CommentType
import org.mule.weave.v2.parser.ast.QName
import org.mule.weave.v2.parser.ast.annotation.AnnotationArgumentsNode
import org.mule.weave.v2.parser.ast.annotation.AnnotationNode
import org.mule.weave.v2.parser.ast.conditional.IfNode
import org.mule.weave.v2.parser.ast.functions.DoBlockNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallNode
import org.mule.weave.v2.parser.ast.functions.FunctionCallParametersNode
import org.mule.weave.v2.parser.ast.functions.FunctionNode
import org.mule.weave.v2.parser.ast.functions.UsingNode
import org.mule.weave.v2.parser.ast.header.HeaderNode
import org.mule.weave.v2.parser.ast.header.directives._
import org.mule.weave.v2.parser.ast.module.ModuleNode
import org.mule.weave.v2.parser.ast.operators.BinaryOpNode
import org.mule.weave.v2.parser.ast.operators.UnaryOpNode
import org.mule.weave.v2.parser.ast.structure.ArrayNode
import org.mule.weave.v2.parser.ast.structure.AttributesNode
import org.mule.weave.v2.parser.ast.structure.BooleanNode
import org.mule.weave.v2.parser.ast.structure.DocumentNode
import org.mule.weave.v2.parser.ast.structure.KeyNode
import org.mule.weave.v2.parser.ast.structure.KeyValuePairNode
import org.mule.weave.v2.parser.ast.structure.NameNode
import org.mule.weave.v2.parser.ast.structure.NameValuePairNode
import org.mule.weave.v2.parser.ast.structure.NamespaceNode
import org.mule.weave.v2.parser.ast.structure.NumberNode
import org.mule.weave.v2.parser.ast.structure.ObjectNode
import org.mule.weave.v2.parser.ast.structure.StringNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaNode
import org.mule.weave.v2.parser.ast.structure.schema.SchemaPropertyNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.TypeSelectorNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNodeWithSchema
import org.mule.weave.v2.parser.ast.updates.ArrayIndexUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.AttributeNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.FieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.MultiFieldNameUpdateSelectorNode
import org.mule.weave.v2.parser.ast.updates.UpdateExpressionNode
import org.mule.weave.v2.parser.ast.updates.UpdateNode
import org.mule.weave.v2.parser.ast.updates.UpdateSelectorNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier
import org.mule.weave.v2.parser.ast.variables.VariableReferenceNode
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.parser.phase.PhaseResult
import org.mule.weave.v2.parser.phase.TypeCheckingResult
import org.mule.weave.v2.parser.phase.VersionCheckerPhase
import org.mule.weave.v2.scope.AstNavigator
import org.mule.weave.v2.scope.Reference
import org.mule.weave.v2.scope.ScopesNavigator
import org.mule.weave.v2.scope.VariableScope
import org.mule.weave.v2.sdk.WeaveResource
import org.mule.weave.v2.ts.AnyType
import org.mule.weave.v2.ts.ArrayType
import org.mule.weave.v2.ts.BooleanType
import org.mule.weave.v2.ts.DateTimeType
import org.mule.weave.v2.ts.FunctionType
import org.mule.weave.v2.ts.IntersectionType
import org.mule.weave.v2.ts.KeyType
import org.mule.weave.v2.ts.KeyValuePairType
import org.mule.weave.v2.ts.LocalDateTimeType
import org.mule.weave.v2.ts.LocalDateType
import org.mule.weave.v2.ts.LocalTimeType
import org.mule.weave.v2.ts.MetadataConstraint
import org.mule.weave.v2.ts.NameType
import org.mule.weave.v2.ts.NameValuePairType
import org.mule.weave.v2.ts.NamespaceType
import org.mule.weave.v2.ts.NumberType
import org.mule.weave.v2.ts.ObjectType
import org.mule.weave.v2.ts.PeriodType
import org.mule.weave.v2.ts.ReferenceType
import org.mule.weave.v2.ts.StringType
import org.mule.weave.v2.ts.TimeType
import org.mule.weave.v2.ts.TimeZoneType
import org.mule.weave.v2.ts.TypeGraph
import org.mule.weave.v2.ts.TypeHelper
import org.mule.weave.v2.ts.TypeNode
import org.mule.weave.v2.ts.TypeParameter
import org.mule.weave.v2.ts.TypeSelectorType
import org.mule.weave.v2.ts.TypeType
import org.mule.weave.v2.ts.UnionType
import org.mule.weave.v2.ts.WeaveType
import org.mule.weave.v2.ts.WeaveTypeCloneHelper
import org.mule.weave.v2.utils.AsciiDocMigrator
import org.mule.weave.v2.utils.StringEscapeHelper
import org.mule.weave.v2.utils.WeaveTypeEmitter
import org.mule.weave.v2.utils.WeaveTypeParser
import org.mule.weave.v2.weavedoc.WeaveDocParser
import org.mule.weave.v2.weavedoc.WeaveDocumentation

import java.util
import scala.collection.JavaConverters.asScalaIteratorConverter
import scala.collection.mutable.ArrayBuffer

class AutoCompletionService(val editor: WeaveEditorSupport, provider: DataFormatDescriptorProvider = EmptyDataFormatDescriptorProvider, indexService: WeaveIndexService = EmptyIndexService, configuration: AutoCompletionConfiguration = AutoCompletionConfiguration()) {

  val keyWords: Array[Suggestion] = Array(
    Suggestion("output", Template().add("output ").choice(provider.dataFormats().map(_.mimeType)), Some("Specified the output type.")),
    Suggestion("var", Template().add("var ").placeHolder().add(" = ").placeHolder(), Some("Declares a new output.")),
    Suggestion("type", Template().add("type ").placeHolder().add(" = ").placeHolder(), Some("Declares a new type.")),
    Suggestion("ns", Template().add("ns ").placeHolder("ns1").add(" ").placeHolder("http://mycompany.com"), Some("Declares a new namespace.")),
    Suggestion("fun", Template().add("fun ").placeHolder().add("(").placeHolder().add(") = ").placeHolder(), Some("Declares a new function.")),
    Suggestion("import", Template().add("import ").placeHolder("*").add(" from ").placeHolder(), Some("Import element from module.")),
    Suggestion("annotation", Template().add("annotation ").placeHolder("").add(" ( ").placeHolder().add(")"), Some("Declares a new Annotation.")))

  val binaryOperators: Array[Suggestion] = Array(
    Suggestion("update", Template().add(" update {\n\tcase ").placeHolder().add(" -> ").placeHolder("???").add("\n}"), Some("Update the value with.")),
    Suggestion("match", Template().add(" match {\n\tcase ").placeHolder().add(" -> ").placeHolder("???").add("\n}"), Some("Pattern match with.")),
    Suggestion("default", Template().add(" default ").placeHolder(" ???"), Some("Defaults to a new value when the left part is `null`.")))

  val temporalFields: Array[Suggestion] = Array(
    Suggestion("nanoseconds", Some("The Nanoseconds part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("milliseconds", Some("The Milliseconds part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("seconds", Some("The Seconds part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("minutes", Some("The Minutes part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("hour", Some("The Hour part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("day", Some("The Day part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("month", Some("The Month part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("year", Some("The Year part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("quarter", Some("The Nanoseconds part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("dayOfWeek", Some("The Day of The Week of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("dayOfYear", Some("The Day of The year of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("offsetSeconds", Some("The Offset Seconds part of this time expression."), NumberType(), SuggestionType.Field),
    Suggestion("timezone", Some("The timezone of this time expression."), TimeZoneType(), SuggestionType.Field))

  val periodFields: Array[Suggestion] = Array(
    Suggestion("secs", Some("Returns the seconds of the given Period."), NumberType(), SuggestionType.Field),
    Suggestion("minutes", Some("Returns the minutes of the given Period."), NumberType(), SuggestionType.Field),
    Suggestion("hours", Some("Returns the hours of the given Period."), NumberType(), SuggestionType.Field),
    Suggestion("weeks", Some("Returns the weeks of the given Period."), NumberType(), SuggestionType.Field),
    Suggestion("years", Some("Returns the years of the given Period."), NumberType(), SuggestionType.Field),
    Suggestion("days", Some("Returns the days of the given Period."), NumberType(), SuggestionType.Field))

  val templates: Array[Suggestion] = Array(Suggestion("if/else", Template().add("if(").placeHolder().add(") ").placeHolder().add(" else ").placeHolder(), Some("If else template.")))

  def suggest(): SuggestionResult = {
    //Special case where the document is empty
    if (editor.content().trim.isEmpty) {
      val result = new AutoCompletionService(editor.withNewContent(editor.content() + AutoCompletionService.INSERTED_TEXT), provider, indexService).suggest()
      SuggestionResult(result.suggestions ++ keyWords, 0, 0)
    } else {
      val maybeAstNodeNavigator = editor.astNavigator()
      maybeAstNodeNavigator match {
        case Some(astNavigator) => {
          val maybeAstNode: Option[AstNode] = editor.elementAtCursor()
          suggestFor(maybeAstNode, editor.cursorLocation(), astNavigator, editor.resource)
        }
        case None => {
          emptySuggestion(editor.cursorLocation())
        }
      }
    }
  }

  def getLeafOnTheLeft(astNode: AstNode, cursorLocation: Int): Option[AstNode] = {
    if (astNode.location().contains(cursorLocation) || astNode.location().endPosition.index < cursorLocation) {
      astNode match {
        case vrn: VariableReferenceNode => Some(vrn)
        case vrn: TypeReferenceNode => Some(vrn)
        case fcn: FunctionCallNode if !astNode.location().contains(cursorLocation) => Some(fcn)
        case _ if astNode.isLeaf() => Some(astNode)
        case bon: BinaryOpNode => Some(bon)
        case uon: UnaryOpNode => Some(uon)
        case _ if astNode.location().contains(cursorLocation) || astNode.isInstanceOf[KeyValuePairNode] => {
          val container: Option[AstNode] = astNode
            .children()
            .find((node) => {
              node.children().nonEmpty && node.location().contains(cursorLocation)
            })
          container match {
            case Some(node) => getLeafOnTheLeft(node, cursorLocation)
            case None => {
              val last: Option[AstNode] = astNode
                .children()
                .filter((astNode) => astNode.location().endPosition.index < cursorLocation)
                .lastOption
              if (last.isDefined) {
                getLeafOnTheLeft(last.get, cursorLocation)
              } else {
                Some(astNode)
              }
            }
          }
        }
        case _ => Some(astNode)
      }
    } else {
      None
    }
  }

  def suggestKeysForCompletion(on: ObjectNode): Seq[Suggestion] = {
    val typePhase: Option[TypeGraph] = editor.reverseTypeGraph()
    typePhase match {
      case Some(typeGraph) => {
        val maybeNode: Option[TypeNode] = typeGraph.findNode(on)
        maybeNode match {
          case Some(node) => {
            val maybeType: Seq[WeaveType] = node
              .outgoingEdges()
              .flatMap(_.mayBeExpectedPropagatedType())
            maybeType
              .flatMap(suggestObjectKeys(_, on))
          }
          case None => Seq()
        }
      }
      case None => Seq()
    }

  }

  private def suggestObjectKeys(weaveType: WeaveType, on: ObjectNode): Seq[Suggestion] = {
    weaveType match {
      case ot: ObjectType => {
        val filtered = ot.properties.filterNot((keyvaluepair) => {
          if (keyvaluepair.repeated) {
            false
          } else {
            val keyToFind: WeaveType = keyvaluepair.key
            existsInObject(on, keyToFind)
          }
        })
        keyValuePairsToSuggestion(filtered, on, false)
      }
      case it: IntersectionType => {
        it.of.flatMap((wt) => suggestObjectKeys(wt, on))
      }
      case ut: UnionType => {
        ut.of.flatMap((wt) => suggestObjectKeys(wt, on))
      }
      case rt: ReferenceType => {
        suggestObjectKeys(rt.resolveType(), on)
      }
      case _ => Seq()
    }

  }

  def suggestAttributes(keyNode: KeyNode): Seq[Suggestion] = {
    val typePhase: Option[TypeGraph] = editor.reverseTypeGraph()
    typePhase match {
      case Some(typeGraph) => {
        val maybeNode = typeGraph.findNode(keyNode)
        maybeNode match {
          case Some(node) => {
            val outgoingNodes: Seq[WeaveType] =
              node
                .outgoingEdges()
                .flatMap(_.mayBeExpectedPropagatedType())

            val maybeKeyType: Option[KeyType] =
              outgoingNodes
                .collectFirst({
                  case keyType: KeyType => keyType
                })

            maybeKeyType
              .map((ot) => {
                attributesToSuggestions(ot, keyNode)
              })
              .getOrElse(Seq())
          }
          case None => Seq()
        }
      }
      case None => Seq()
    }

  }

  private def attributesToSuggestions(ot: KeyType, location: AstNode): Seq[Suggestion] = {
    ot.attrs.flatMap((nvp) =>
      nvp.name match {
        case NameType(Some(name)) => {
          val localName = name.name
          val prefix = localName + ": "
          val nameSuggestion: Suggestion = Suggestion(localName, Template(prefix), nvp.getDocumentation(), nvp.value)
          val smartSuggestion: Seq[Suggestion] = suggestExpressionsBasedOnNameType(localName, prefix, nvp.value, location)
          smartSuggestion :+ nameSuggestion
        }
        case _ => Seq()
      })
  }

  private def existsInObject(on: ObjectNode, keyToFind: WeaveType): Boolean = {
    keyToFind match {
      case KeyType(NameType(Some(name)), _) => {
        val exists = on
          .children()
          .exists({
            case KeyValuePairNode(key, _, _) => {
              key match {
                case KeyNode(StringNode(keyValue, _), _, _, _) => {
                  keyValue.equals(name.name)
                }
                case _ => false
              }
            }
            case _ => false
          })

        exists
      }
      case _ => false
    }
  }

  def collectLiterals(weaveType: WeaveType, on: AstNode, typeGraph: TypeGraph): Seq[Suggestion] = {
    weaveType match {
      case StringType(Some(sl)) => {
        if (!(on.isInstanceOf[BooleanNode] || on.isInstanceOf[NumberNode]))
          on match {
            case node: FunctionCallParametersNode =>
              val functionCallParameterArgs = node.args
              if (functionCallParameterArgs.nonEmpty && functionCallParameterArgs.last.isInstanceOf[StringNode] && functionCallParameterArgs.last.asInstanceOf[StringNode].value == "") {
                Seq(Suggestion(sl, weaveType.getDocumentation(), weaveType, SuggestionType.Enum))
              } else {
                Seq(Suggestion(wrapIntoQuotesIfRequired(sl, on), weaveType.getDocumentation(), weaveType, SuggestionType.Enum))
              }
            case _ => {
              Seq(Suggestion(wrapIntoQuotesIfRequired(sl, on), weaveType.getDocumentation(), weaveType, SuggestionType.Enum))
            }
          }
        else
          Seq()
      }
      case BooleanType(Some(bl), _) => {
        if (!(on.isInstanceOf[StringNode] || on.isInstanceOf[NumberNode]))
          Seq(Suggestion(bl.toString, weaveType.getDocumentation(), weaveType, SuggestionType.Enum))
        else
          Seq()
      }
      case NumberType(Some(nl)) => {
        if (!(on.isInstanceOf[BooleanNode] || on.isInstanceOf[StringNode]))
          Seq(Suggestion(nl, weaveType.getDocumentation(), weaveType, SuggestionType.Enum))
        else
          Seq()
      }
      case it: IntersectionType => {
        it.of.flatMap((wt) => collectLiterals(wt, on, typeGraph))
      }
      case ut: UnionType => {
        ut.of.flatMap((wt) => collectLiterals(wt, on, typeGraph))
      }
      case rt: ReferenceType => {
        collectLiterals(rt.resolveType(), on, typeGraph)
      }
      case ft: FunctionType => {
        on match {
          case functionCallParametersNode: FunctionCallParametersNode =>
            val index: Int = argumentIndex(functionCallParametersNode, nodeAtCursor())
            val functionParamTypes: Seq[WeaveType] = ft.params.map(_.wtype)
            if (index < functionParamTypes.size) {
              collectLiterals(functionParamTypes(index), on, typeGraph)
            } else {
              Seq()
            }
          case _ =>
            Seq()
        }
      }
      case _ => Seq()
    }
  }

  private def nodeAtCursor(): Option[AstNode] = {
    val currentLocation: Int = editor.cursorLocation()
    val maybeNode: Option[AstNode] = editor.astNavigator().flatMap((navigator) => navigator.nodeAt(currentLocation))
    maybeNode
  }

  private def wrapIntoQuotesIfRequired(sl: String, on: AstNode): String = {
    on match {
      case _: StringNode => sl
      case _             => "\"" ++ sl ++ "\""
    }
  }

  def suggestParametersExpression(fcn: FunctionCallParametersNode, navigator: AstNavigator): Seq[Suggestion] = {
    val typePhase: Option[TypeGraph] = editor.reverseTypeGraph()
    typePhase match {
      case Some(typeGraph) => {
        val index = argumentIndex(fcn, nodeAtCursor())
        val maybeTypeNode: Option[TypeNode] =
          navigator.parentOf(fcn) match {
            case Some(FunctionCallNode(vr, _, _, _)) => typeGraph.findNode(vr)
            case _                                   => None
          }

        maybeTypeNode match {
          case Some(node) => {
            val maybeType: Seq[Suggestion] = node
              .incomingEdges()
              .map(_.mayBeIncomingType())
              .flatMap((wt: Option[WeaveType]) =>
                wt match {
                  case Some(ft @ FunctionType(_, _, _, ov, _, _)) => if (ov.nonEmpty) ov else Seq(ft)
                  case _ => Seq()
                })
              .flatMap((ft) => {
                if (index < ft.params.size) {
                  val parameter = ft.params(index)
                  suggestExpressionsBasedOnNameType(parameter.name, "", parameter.wtype, nodeAtCursor().getOrElse(fcn))
                } else {
                  Seq()
                }
              })
            maybeType
          }
          case None => Seq()
        }
      }
      case _ => Seq()

    }
  }

  private def argumentIndex(fcn: FunctionCallParametersNode, maybeNode: Option[AstNode]) = {
    if (maybeNode.isDefined) {
      val argumentIndex: Int = fcn.args.indexWhere((node) => (node eq maybeNode.get) || AstNodeHelper.containsChild(node, maybeNode.get))
      if (argumentIndex == -1) 0 else argumentIndex
    } else {
      fcn.args.length - 1
    }
  }

  def suggestLiterals(on: AstNode, navigator: AstNavigator): Seq[Suggestion] = {
    val typePhase: Option[TypeGraph] = editor.reverseTypeGraph()
    val parentNode: Option[AstNode] = navigator.parentOf(on)
    if (parentNode.isDefined && parentNode.get.isInstanceOf[FunctionCallParametersNode]) {
      suggestLiterals(parentNode.get, navigator)
    } else {
      typePhase match {
        case Some(typeGraph) => {
          val maybeNode = on match {
            case FunctionCallParametersNode(_) =>
              navigator.parentOf(on) match {
                case Some(FunctionCallNode(vr, _, _, _)) => typeGraph.findNode(vr)
                case _                                   => None
              }
            case _ => typeGraph.findNode(on)
          }
          maybeNode match {
            case Some(node) => {
              if (on.isInstanceOf[FunctionCallParametersNode]) {
                val maybeType: Seq[Suggestion] = node
                  .incomingEdges()
                  .map(_.mayBeIncomingType())
                  .flatMap((wt: Option[WeaveType]) =>
                    wt match {
                      case Some(FunctionType(_, _, _, ov, _, _)) => if (ov.nonEmpty) ov else Seq(wt.get)
                      case Some(_)                               => Seq(wt.get)
                      case _                                     => Seq()
                    })
                  .flatMap((ft) => collectLiterals(ft, on, typeGraph))
                maybeType
              } else {
                val maybeType: Seq[WeaveType] = node
                  .outgoingEdges()
                  .flatMap(_.mayBeExpectedPropagatedType())
                maybeType
                  .flatMap(collectLiterals(_, on, typeGraph))
              }
            }
            case None => Seq()
          }
        }
        case None => Seq()
      }
    }
  }

  @scala.annotation.tailrec
  private def suggestFor(elementNode: Option[AstNode], cursorLocation: Int, astNavigator: AstNavigator, weaveResource: WeaveResource): SuggestionResult = {
    elementNode match {
      case Some(dn: DocumentNode) => {
        val astNode = AstNodeHelper.lastNode(dn.root)
        astNode match {
          case _: ErrorAstNode => suggestFor(Some(astNode), cursorLocation, astNavigator, weaveResource)
          case _ => {
            val suggestionSeq = suggestOperationsAvailableFor(dn.root)
            SuggestionResult(suggestionSeq.toArray, cursorLocation, cursorLocation)
          }
        }

      }
      case Some(_: CommentNode) => {
        emptySuggestion(cursorLocation)
      }
      case Some(dn: ModuleNode) => {
        val elements: Seq[DirectiveNode] = dn.elements
        val suggestionSeq: Seq[Suggestion] = suggestForDirectives(elements)
        SuggestionResult(suggestionSeq.toArray, cursorLocation, cursorLocation)
      }
      case Some(an: ArrayNode) if previousCharAnyOf(weaveResource, cursorLocation, Seq('[', ',', '(')) => {
        val maybeNavigator = editor.scopeGraph().flatMap(_.scopeOf(an))
        maybeNavigator match {
          case Some(value) => {
            val visibleVariables = suggestVisibleVariables(value, editor.typeGraph())
            SuggestionResult(visibleVariables.toArray, cursorLocation, cursorLocation)
          }
          case None => emptySuggestion(cursorLocation)
        }
      }
      case Some(on: ObjectNode) if on.location().contains(cursorLocation) && (on.children().isEmpty || previousCharAnyOf(weaveResource, cursorLocation, Seq('{', ',', '('))) => {
        SuggestionResult(suggestKeysForCompletion(on).toArray, cursorLocation, cursorLocation)
      }
      case Some(on: StringNode) if (astNavigator.parentWithType(on, classOf[NameValuePairNode]).isDefined) => {
        astNavigator.parentWithType(on, classOf[KeyNode]) match {
          case Some(kn: KeyNode) => {
            SuggestionResult(suggestAttributes(kn).toArray, cursorLocation, cursorLocation)
          }
          case _ => emptySuggestion(cursorLocation)
        }
      }
      case Some(on: StringNode) if (astNavigator.parentWithType(on, classOf[KeyNode]).isDefined) => {
        val maybeObjectNode = astNavigator.parentWithType(on, classOf[ObjectNode])
        maybeObjectNode match {
          case Some(objectNode) =>
            SuggestionResult(suggestKeysForCompletion(objectNode).toArray, cursorLocation, cursorLocation)
          case None => {
            emptySuggestion(cursorLocation)
          }
        }
      }
      case Some(attr: AttributesNode) if attr.location().contains(cursorLocation) => {
        astNavigator.parentOf(attr) match {
          case Some(kn: KeyNode) => {
            SuggestionResult(suggestAttributes(kn).toArray, cursorLocation, cursorLocation)
          }
          case _ => emptySuggestion(cursorLocation)
        }
      }
      case Some(kvpn: KeyValuePairNode) => {
        if (kvpn.key.location().contains(cursorLocation)) {
          suggestFor(Some(kvpn.key), cursorLocation, astNavigator, weaveResource)
        } else {
          suggestFor(Some(kvpn.value), cursorLocation, astNavigator, weaveResource)
        }
      }
      case Some(err: MissingTokenErrorAstNode) => {
        SuggestionResult(Array(Suggestion(err.token, Template(err.token), None)), cursorLocation, cursorLocation)
      }
      case Some(err: MissingVersionMajor) => {
        SuggestionResult(Array(Suggestion("Version Major", Template(VersionCheckerPhase.getWeaveVersion.major.toString), None)), cursorLocation, cursorLocation)
      }
      case Some(err: MissingVersionMinor) => {
        SuggestionResult(
          Array(
            // TODO add supportedVersionMinor
            Suggestion("Version Minor", Template(VersionCheckerPhase.getWeaveVersion.minor.toString), None)),
          cursorLocation,
          cursorLocation)
      }
      case Some(err: MissingExpressionErrorAstNode) if (!isSchemaPropertyNodeExpression(err, astNavigator) && !isDirectiveOptionNodeExpression(err, astNavigator)) => {
        val suggestions = editor
          .scopeGraph()
          .flatMap((sn) => sn.scopeOf(err))
          .map((vs) => {
            val literals: Seq[Suggestion] = suggestLiterals(err, astNavigator)
            val visibleVariables: Seq[Suggestion] = suggestVisibleVariables(vs, editor.typeGraph())
            val optionsWithoutTypeAndNamespaces: Seq[Suggestion] = filterTypesAndNamespaces(visibleVariables)
            val result: Array[Suggestion] = suggestModulesWith()
            val expressionSuggestions: Seq[Suggestion] = {
              val maybeNameValuePairNode: Option[NameValuePairNode] = astNavigator.parentWithType(err, classOf[NameValuePairNode])
              if (maybeNameValuePairNode.isDefined) {
                suggestExpressionsBasedOnContextNode(err, maybeNameValuePairNode.get)
              } else {
                val maybeKeyValuePairNode: Option[KeyValuePairNode] = astNavigator.parentWithType(err, classOf[KeyValuePairNode])
                if (maybeKeyValuePairNode.isDefined) {
                  suggestExpressionsBasedOnContextNode(err, maybeKeyValuePairNode.get)
                } else {
                  Seq()
                }
              }
            }
            expressionSuggestions ++ literals ++ optionsWithoutTypeAndNamespaces ++ result ++ templates
          })
          .getOrElse(Seq())
        SuggestionResult(suggestions.toArray, cursorLocation.min(err.location().startPosition.index), cursorLocation.max(err.location().startPosition.index))
      }
      case Some(_: MissingFormatDefinition | _: InputDirective) => {
        suggestMimesAndIds(Right(cursorLocation))
      }
      case Some(formatId: DataFormatId) => {
        suggestMimesAndIds(Left(formatId))
      }
      case Some(ni: NameIdentifier) if astNavigator.isParentOfType(ni, classOf[ImportedElement]) => {
        val element = astNavigator.parentOf(ni).get.asInstanceOf[ImportedElement]
        val importDirective = astNavigator.parentOf(element)
        importDirective match {
          case Some(_: ImportDirective) => {
            suggestVisibleModules(ni, cursorLocation, appendSeparator = false)
          }
          case Some(ies: ImportedElements) => {
            val importDirective = astNavigator.parentWithType(ies, classOf[ImportDirective]).get
            val suggestions = suggestModuleElements(importDirective.importedModule.elementName)
              .getOrElse(Seq())
              .map((suggestion) => {
                //The template for functions has () we only need the name
                suggestion.copy(template = Template(suggestion.name))
              })
            val startIndex = resolveStartIndex(ni, cursorLocation)
            val endIndex = resolveEndIndex(ni, cursorLocation)
            SuggestionResult(suggestions.toArray, startIndex, endIndex)
          }
          case _ => SuggestionResult(Array(), ni.location().startPosition.index, ni.location().endPosition.index)
        }
      }
      case Some(ifen: IfNode) => {
        suggestFor(Some(ifen.ifExpr), cursorLocation, astNavigator, weaveResource)
      }
      case Some(ni) if astNavigator.parentWithType(ni, classOf[SchemaPropertyNode]).isDefined => {
        val schemaPropertyNode = astNavigator.parentWithType(ni, classOf[SchemaPropertyNode]).get
        val maybeType = astNavigator.parentWithType(schemaPropertyNode, classOf[WeaveTypeNodeWithSchema])
        val result: Array[Suggestion] = maybeType match {
          case Some(theType) => {
            ni match {
              case sn: StringNode if (astNavigator.isDescendantOf(schemaPropertyNode.name, sn)) => {
                suggestSchemaPropertyKeys(theType)
              }
              case _ => {
                suggestSchemaPropertyValues(schemaPropertyNode)
              }
            }
          }
          case None => {
            Array()
          }
        }
        SuggestionResult(result, ni.location().startPosition.index, ni.location().endPosition.index)
      }
      case Some(sv: SchemaNode) => {
        val maybeType = astNavigator.parentWithType(sv, classOf[WeaveTypeNodeWithSchema])
        val result: Array[Suggestion] = maybeType match {
          case Some(theType) => {
            suggestSchemaPropertyKeys(theType)
          }
          case None => {
            Array()
          }
        }
        SuggestionResult(result, cursorLocation, cursorLocation)
      }
      case Some(st: StringNode) if st.value == "" => {
        SuggestionResult(suggestLiterals(st, astNavigator).toArray, cursorLocation, cursorLocation)
      }
      case Some(_: WeaveTypeNode) => { //Type nodes there is not much to suggest
        emptySuggestion(cursorLocation)
      }
      case Some(astNode) => {
        if (isSimpleNode(astNode) || astNode.isInstanceOf[UnaryOpNode]) {
          val maybeTextPrefix: Option[String] = calculateTextPrefix(cursorLocation, astNode)
          suggestions(astNode, cursorLocation, astNavigator, editor.scopeGraph(), editor.typeGraph(), maybeTextPrefix)
        } else {
          val suggestionSeq = suggestOperationsAvailableFor(astNode)
          SuggestionResult(suggestionSeq.toArray, cursorLocation, cursorLocation)
        }
      }
      case None => emptySuggestion(cursorLocation)
    }
  }

  private def emptySuggestion(cursorLocation: Int) = {
    SuggestionResult(Array(), cursorLocation, cursorLocation)
  }

  private def calculateTextPrefix(cursorLocation: Int, astNode: AstNode): Option[String] = {
    astNode match {
      case node: StringNode if astNode.location().startPosition.index < cursorLocation && astNode.location().endPosition.index >= cursorLocation =>
        val currentWord = node.literalValue
        val begin = node.quotedBy().map((_) => 1).getOrElse(0)
        val end = if (cursorLocation >= astNode.location().startPosition.index + begin) {
          cursorLocation - (astNode.location().startPosition.index + begin)
        } else {
          cursorLocation
        }
        val prefix = if (currentWord.length > end) currentWord.substring(0, end) else currentWord
        Some(prefix)
      case node: NameIdentifier if node.parent().isEmpty && astNode.location().startPosition.index < cursorLocation => {
        if (node.name != NameIdentifier.INSERTED_FAKE_VARIABLE_NAME)
          node.nameElements().lastOption
        else
          None
      }
      case node: DirectiveOptionName => {
        val currentWord = node.name
        val prefix = currentWord.substring(0, cursorLocation - astNode.location().startPosition.index)
        Some(prefix)
      }
      case contentType: ContentType => {
        val currentWord = contentType.mime
        val prefix = currentWord.substring(0, cursorLocation - astNode.location().startPosition.index)
        Some(prefix)
      }
      case _ =>
        None
    }
  }

  private def suggestExpressionsBasedOnContextNode(err: MissingExpressionErrorAstNode, keyValuePairNode: AstNode): Seq[Suggestion] = {
    val typeGraph: Option[TypeGraph] = editor.reverseTypeGraph()
    typeGraph
      .flatMap((tg) => {
        val maybeTypeNodeNode: Option[TypeNode] = tg.findNode(keyValuePairNode)
        maybeTypeNodeNode
          .map((kv) => {
            val maybeKeyValuePairType: Option[WeaveType] = kv.outgoingEdges().headOption.flatMap(_.mayBeExpectedPropagatedType())
            maybeKeyValuePairType match {
              case Some(keyValuePairType: KeyValuePairType) => {
                keyValuePairType.key match {
                  case kt: KeyType => {
                    kt.name match {
                      case NameType(Some(QName(name, ns))) => {
                        suggestExpressionsBasedOnNameType(name, "", keyValuePairType.value, err)
                      }
                      case _ => Seq()
                    }
                  }
                  case _ => Seq()
                }
              }
              case Some(nvp: NameValuePairType) => {
                nvp.name match {
                  case NameType(Some(QName(name, ns))) => {
                    suggestExpressionsBasedOnNameType(name, "", nvp.value, err)
                  }
                  case _ => Seq()
                }
              }
              case _ => Seq()
            }
          })
      })
      .getOrElse(Seq())
  }

  private def suggestMimesAndIds(nodeOrCursor: Either[AstNode, Int]) = {
    val ctSuggestions = suggestContentTypes()
    val idSuggestions = suggestFormatIds()

    val suggs = (ctSuggestions ++ idSuggestions).toArray
    nodeOrCursor match {
      case Left(node)            => SuggestionResult(node, suggs)
      case Right(cursorLocation) => SuggestionResult(suggs, cursorLocation, cursorLocation)
    }
  }

  private def isSchemaPropertyNodeExpression(err: MissingExpressionErrorAstNode, astNavigator: AstNavigator) = {
    astNavigator.parentWithType(err, classOf[SchemaPropertyNode]).nonEmpty
  }

  private def isDirectiveOptionNodeExpression(err: MissingExpressionErrorAstNode, astNavigator: AstNavigator) = {
    astNavigator.parentWithType(err, classOf[DirectiveOption]).nonEmpty
  }

  private def suggestSchemaPropertyValues(schemaPropertyNode: SchemaPropertyNode): Array[Suggestion] = {
    schemaPropertyNode.name match {
      case StringNode(name, _) => {
        name match {
          case MetadataConstraint.UNIT_PROPERTY_NAME =>
            Array(
              Suggestion("\"nanos\"", Some("The Temporal Data Converted into Nanoseconds"), StringType()),
              Suggestion("\"milliseconds\"", Some("The Temporal Data Converted into Milliseconds"), StringType()),
              Suggestion("\"seconds\"", Some("The Temporal Data Converted into Seconds"), StringType()),
              Suggestion("\"hours\"", Some("The Temporal Data Converted into Hours"), StringType()),
              Suggestion("\"days\"", Some("The Temporal Data Converted into Days"), StringType()),
              Suggestion("\"months\"", Some("The Temporal Data Converted into Months"), StringType()),
              Suggestion("\"years\"", Some("The Temporal Data Converted into Years"), StringType()))
          case MetadataConstraint.BASE_PROPERTY_NAME =>
            Array(Suggestion("\"64\"", Some("For base 64"), StringType()), Suggestion("\"16\"", Some("For base 16"), StringType()))
          case MetadataConstraint.MODE_PROPERTY_NAME =>
            Array(
              Suggestion(
                "\"SMART\"",
                Some("Style to resolve dates and times in a smart, or intelligent, manner. Using smart resolution will perform the sensible default for each field, which may be the same as strict, the same as lenient, or a third behavior. Individual fields will interpret this differently."),
                StringType()),
              Suggestion(
                "\"LENIENT\"",
                Some(" Style to resolve dates and times leniently. Using lenient resolution will resolve the values in an appropriate lenient manner. Individual fields will interpret this differently."),
                StringType()),
              Suggestion(
                "\"STRICT\"",
                Some("Style to resolve dates and times strictly. Using strict resolution will ensure that all parsed values are within the outer range of valid values for the field. Individual fields may be further processed for strictness"),
                StringType()))
          case MetadataConstraint.ROUND_MODE_PROPERTY_NAME =>
            Array(
              Suggestion(
                "\"UP\"",
                Some("Rounding mode to round away from zero.  Always increments the digit prior to a non-zero discarded fraction.  Note that this rounding mode never decreases the magnitude of the calculated value."),
                StringType()),
              Suggestion(
                "\"DOWN\"",
                Some("Rounding mode to round towards zero.  Never increments the digit prior to a discarded fraction (i.e., truncates).  Note that this rounding mode never increases the magnitude of the calculated value."),
                StringType()),
              Suggestion(
                "\"CEILING\"",
                Some("Rounding mode to round towards positive infinity.  If the result is positive, behaves as for  UP; if negative, behaves as for  DOWN.  Note that this rounding mode never decreases the calculated value."),
                StringType()),
              Suggestion(
                "\"FLOOR\"",
                Some("Rounding mode to round towards negative infinity.  If the result is positive, behave as for  DOWN; if negative, behave as for  UP.  Note that this rounding mode never increases the calculated value."),
                StringType()),
              Suggestion(
                "\"HALF_DOWN\"",
                Some("Rounding mode to round towards  \"nearest neighbor\" unless both neighbors are equidistant, in which case round down. Behaves as for UP if the discarded  fraction is &gt; 0.5; otherwise, behaves as for   DOWN."),
                StringType()),
              Suggestion(
                "\"HALF_UP\"",
                Some(
                  "The default roundMode.Rounding mode to round towards \"nearest neighbor\" unless both neighbors are equidistant, in which case round up. Behaves as for \"UP\" if the discarded fraction is > 0.5 otherwise, behaves as for \"DOWN\".  Note that this is the rounding mode commonly taught at school."),
                StringType()))
          case _ => Array()
        }
      }
      case _ => Array()
    }
  }

  private def suggestSchemaPropertyKeys(theType: WeaveTypeNodeWithSchema): Array[Suggestion] = {
    val clazz = Suggestion("class", Some("The fully qualified name of the class name to be used by the java writer when writing this value. "), StringType())
    theType match {
      case rt: TypeReferenceNode if (rt.variable.name == "String") => {
        Array(
          Suggestion("base", Some("The `base` to use to convert a Binary Value to a String. Valid values are '64' '16'"), StringType()),
          Suggestion("locale", Some("The locale to be used for this conversion"), StringType()),
          Suggestion("encoding", Some("The `encoding` to be use to convert a Binary Value to a String"), StringType()),
          Suggestion(
            "format",
            Some(
              """The `format` to be used to convert a Temporal Type (Date, DateTime, Time, LocalDateTime, Period, ...) Value to a String.
                |The `format` to be used to convert a Number Value to a String""".stripMargin),
            StringType()),
          Suggestion(
            "roundMode",
            Some("Specifies a `rounding behavior` for numerical operations capable of discarding precision. Valid options are: UP, DOWN, CEILING, FLOOR, HALF_UP, HALF_DOWN, HALF_DOWN."),
            StringType()),
          Suggestion("mode", Some("Specifies a `parsing mode` for date types by default is `SMART`. Valid options are: SMART, STRICT, LENIENT."), StringType()),
          clazz)
      }
      case rt: TypeReferenceNode if (rt.variable.name == "Binary") => {
        Array(
          Suggestion("base", Some("The `base` to use to convert a Binary Value to a String. Valid values are '64' '16'"), StringType()),
          Suggestion("encoding", Some("The `encoding` to be use to convert a Binary Value to a String"), StringType()))
      }
      case rt: TypeReferenceNode if (rt.variable.name == "Number") => {
        Array(
          Suggestion(
            "unit",
            Some(
              """The `unit` to be used to convert  a Temporal (Date, DateTime, Time, LocalDateTime, Period, ...) Type to String.
                |Valid values are :
                |  - nanos
                |  - milliseconds
                |  - seconds
                |  - hours
                |  - days
                |  - months
                |  - years
                |  """.stripMargin),
            StringType()),
          Suggestion("locale", Some("The locale to be used for this conversion"), StringType()),
          Suggestion("format", Some("""The `format` to be used to convert a String Value to a Number""".stripMargin), StringType()),
          clazz)
      }
      case rt: TypeReferenceNode if (Seq("Date", "DateTime", "LocalDateTime", "LocalTime", "Time").contains(rt.variable.name)) => {
        Array(
          Suggestion(
            "format",
            Some("""The `format` to be use to convert a Temporal Type (Date, DateTime, Time, LocalDateTime, Period, ...) Value to a String."""),
            StringType()),
          clazz)
      }
      case _ => Array()
    }
  }

  private def suggestForDirectives(elements: Seq[DirectiveNode]): Seq[Suggestion] = {
    val cursorLineNumber = editor.cursorLineNumber() + 1
    elements
      .filterNot(_.isInstanceOf[VersionDirective])
      .find((p) => {
        p.location().endPosition.line == cursorLineNumber
      })
      .map((directive) => {
        suggestOperationsAvailableFor(directive)
      })
      .getOrElse(keyWords.toSeq)
  }

  private def suggestOperationsAvailableFor(astNode: AstNode): Seq[Suggestion] = {
    val suggestions = astNode match {
      case odn: OutputDirective => {
        suggestWriterOptions(odn)
      }
      case idn: InputDirective => {
        suggestReaderOptions(idn)
      }
      case fd: FunctionDirectiveNode => {
        suggestOperationsAvailableFor(fd.literal)
      }
      case vd: VarDirective => {
        suggestOperationsAvailableFor(vd.value)
      }
      case fd: FunctionNode => {
        suggestOperationsAvailableFor(fd.body)
      }
      case fd: DoBlockNode => {
        suggestOperationsAvailableFor(fd.body)
      }
      case fd: UsingNode => {
        suggestOperationsAvailableFor(fd.expr)
      }
      case hd: HeaderNode => {
        suggestForDirectives(hd.directives)
      }
      case _ => {
        val scopeResult = editor.scopeGraph()
        if (scopeResult.isDefined) {
          val scopeNavigator = scopeResult.get
          val maybeScope = scopeNavigator.scopeOf(astNode)
          maybeScope match {
            case Some(scope) => {
              val maybeTypeGraph = editor.typeGraph()
              val options = suggestVisibleVariables(scope, maybeTypeGraph)
              filterBasedOnLeftType(options, astNode, maybeTypeGraph) ++ binaryOperators
            }
            case None => Seq()
          }
        } else {
          Seq()
        }
      }
    }
    suggestions
  }

  private def previousCharAnyOf(weaveResource: WeaveResource, index: Int, chars: Seq[Char]): Boolean = {
    val maybeChar = weaveResource.previousNonWSChar(index)
    maybeChar match {
      case Some(previousChar) => chars.contains(previousChar)
      case None               => false
    }
  }

  private def filterBasedOnLeftType(options: Seq[Suggestion], astNode: AstNode, maybeTypeGraph: Option[TypeGraph]): Seq[Suggestion] = {
    val node = getLeafOnTheLeft(astNode, editor.cursorLocation())
    filterBasedOn(options, maybeTypeGraph, node)
  }

  private def filterBasedOn(options: Seq[Suggestion], maybeTypeGraph: Option[TypeGraph], node: Option[AstNode]): Seq[Suggestion] = {
    if (node.isDefined) {
      val mayBeLeftType = maybeTypeGraph.flatMap((typeGraph) => {
        typeGraph
          .findNode(node.get)
          .flatMap((node) => {
            node.resultType()
          })
      })
      if (mayBeLeftType.isDefined) {
        val filteredOptions = options.flatMap((suggestion) => {
          suggestion.wtype match {
            case Some(ft: FunctionType) => {
              if (ft.overloads.nonEmpty) {
                val options = ft.overloads.filter((overload) => {
                  canBeCalledAsBinary(overload) &&
                    //Specify null as ctx to avoid dynamicReturnType resolution as it can not be resolved here
                    TypeHelper.canBeSubstituted(mayBeLeftType.get, overload.params.head.wtype, null)
                })
                if (options.isEmpty) {
                  None
                } else {
                  val availableOverload = if (options.size == 1) options.head else ft.copy(overloads = options)
                  val secondParameter = availableOverload.params(1).wtype
                  val template: Template = createInfixTemplate(suggestion, secondParameter)
                  val documentation = collectDocumentation(availableOverload)
                  Some(suggestion.copy(template = template, mayBeDocumentation = Some(documentation), wtype = Some(availableOverload), functionCallType = Some(SuggestionFunctionCallType.PREFIX)))
                }
              } else if (canBeCalledAsBinary(ft)) {
                //Specify null as ctx to avoid dynamicReturnType resolution as it can not be resolved here
                val canBeSubstituted = TypeHelper.canBeSubstituted(mayBeLeftType.get, ft.params.head.wtype, null)
                if (canBeSubstituted) {
                  val functionType = suggestion.wtype.get.asInstanceOf[FunctionType]
                  val secondParameter = functionType.params(1).wtype
                  val template: Template = createInfixTemplate(suggestion, secondParameter)
                  Some(suggestion.copy(template = template, functionCallType = Some(SuggestionFunctionCallType.PREFIX)))
                } else {
                  None
                }
              } else {
                None
              }
            }
            case _ => None
          }
        })
        val additionalOptions: Seq[Suggestion] = suggestMatch(mayBeLeftType)
        filteredOptions ++ additionalOptions
      } else {
        options.flatMap((suggestion) => {
          suggestion.wtype match {
            case Some(ft: FunctionType) if canBeCalledAsBinary(ft) => {
              if (ft.overloads.nonEmpty) {
                val options = ft.overloads
                val availableOverload = if (options.size == 1) options.head else ft.copy(overloads = options)
                val secondParameter = availableOverload.params(1).wtype
                val template: Template = createInfixTemplate(suggestion, secondParameter)
                val documentation = collectDocumentation(availableOverload)
                Some(suggestion.copy(template = template, mayBeDocumentation = Some(documentation), wtype = Some(availableOverload), functionCallType = Some(SuggestionFunctionCallType.PREFIX)))
              } else if (canBeCalledAsBinary(ft)) {
                val functionType = suggestion.wtype.get.asInstanceOf[FunctionType]
                val secondParameter = functionType.params(1).wtype
                val template: Template = createInfixTemplate(suggestion, secondParameter)
                Some(suggestion.copy(template = template, functionCallType = Some(SuggestionFunctionCallType.PREFIX)))
              } else {
                None
              }
            }
            case _ => None
          }
        })
      }
    } else {
      options
    }
  }

  private def suggestMatch(mayBeLeftType: Option[WeaveType]): Seq[Suggestion] = {
    val leftType = mayBeLeftType.get
    val types = TypeHelper.collectAllTypeFormUnionType(leftType)
    if (types.isEmpty) {
      if (TypeHelper.isArrayType(leftType)) {
        val template = Template()
          .add(" match {")
          .add("\n\tcase [x ~ xs] -> ")
          .placeHolder("???")
          .add("\n\tcase [] -> ")
          .placeHolder("???")
          .add("\n}")
        Seq(Suggestion("match (deconstruct array)", template, Some("Pattern match the `Array` by deconstructing in head and tail.")))
      } else if (TypeHelper.isObjectType(leftType)) {
        val template = Template()
          .add(" match {")
          .add("\n\tcase {k: v ~ xs} -> ")
          .placeHolder("???")
          .add("\n\tcase {} -> ")
          .placeHolder("???")
          .add("\n}")
        Seq(Suggestion("match (deconstruct object)", template, Some("Pattern match the `Object` by deconstructing in head and tail.")))
      } else {
        Seq()
      }
    } else {
      val template = Template().add(" match {")
      types.foreach((wtype) => {
        template.add(s"\n\tcase is ${wtype.toString(prettyPrint = false, namesOnly = true)} -> ").placeHolder("???")
      })
      template.add("\n}")
      Seq(Suggestion("match (exhaustive)", template, Some("Pattern match with all the options.")))
    }

  }

  private def createInfixTemplate(option: Suggestion, secondParameter: WeaveType): Template = {
    val template = Template(option.name)
    template.space()
    addLambdaTemplate(template, secondParameter)
    template
  }

  private def canBeCalledAsBinary(ft: FunctionType): Boolean = {
    //It should have at least two parameters and required can not be more than 2
    ft.params.size >= 2 && ft.params.count(!_.optional) <= 2
  }

  @scala.annotation.tailrec
  private def addLambdaTemplate(template: Template, parameterType: WeaveType): Unit = {
    parameterType match {
      case ft: FunctionType => {
        template.add("((")
        ft.params.zipWithIndex.map((paramType) => {
          val fp = paramType._1
          if (paramType._2 > 0) {
            template.add(", ")
          }
          if (!fp.name.startsWith("$")) {
            template.placeHolder(fp.name)
          } else {
            template.placeHolder(s"param_${fp.name.length}")
          }
        })
        template.add(") -> ").placeHolder().add(")")
      }
      case ut: UnionType => {
        addLambdaTemplate(template, ut.of.head)
      }
      case _ => {
        template.space().placeHolder()
      }
    }

  }

  private def isSimpleNode(astNode: AstNode) = {
    astNode.children().isEmpty && !(astNode.isInstanceOf[ObjectNode] || astNode.isInstanceOf[ArrayNode])
  }

  def collectFieldSelectionSuggestions(wtype: WeaveType, astNode: AstNode, prefix: String = ""): Seq[Suggestion] = {
    wtype match {
      case ot: ObjectType => {
        suggestObjectFields(ot, astNode, prefix)
      }
      case array: ArrayType => {
        collectFieldSelectionSuggestions(array.of, astNode, prefix).map((suggestion) => suggestion.copy(wtype = suggestion.wtype.map(ArrayType)))
      }
      case ut: UnionType => {
        val weaveType = TypeHelper.resolveUnion(TypeHelper.simplifyUnions(ut))
        weaveType match {
          case nut: UnionType => {
            nut.of.flatMap((wtype) => {
              collectFieldSelectionSuggestions(wtype, astNode, prefix)
            })
          }
          case wt => {
            collectFieldSelectionSuggestions(wt, astNode, prefix)
          }
        }
      }
      case it: IntersectionType => {
        val weaveType = TypeHelper.resolveIntersection(it.of)
        weaveType match {
          case nit: IntersectionType => {
            nit.of.flatMap((wtype) => collectFieldSelectionSuggestions(wtype, astNode, prefix))
          }
          case _ => collectFieldSelectionSuggestions(weaveType, astNode, prefix)
        }
      }
      case rt: ReferenceType    => collectFieldSelectionSuggestions(rt.resolveType(), astNode, prefix)
      case _: LocalDateType     => temporalFields
      case _: DateTimeType      => temporalFields
      case _: LocalDateTimeType => temporalFields
      case _: TimeType          => temporalFields
      case _: LocalTimeType     => temporalFields
      case _: PeriodType        => periodFields
      case _: NamespaceType     => Seq(Suggestion("uri", Some("The Uri of this namespace"), StringType()), Suggestion("prefix", Some("The Prefix Value"), StringType()))
      case _                    => Seq()
    }
  }

  def suggestObjectFields(ot: ObjectType, astNode: AstNode, prefix: String = ""): Seq[Suggestion] = {
    val properties: Seq[KeyValuePairType] = ot.properties
    keyValuePairsToSuggestion(properties, astNode, true, prefix)
  }

  private def keyValuePairsToSuggestion(properties: Seq[KeyValuePairType], astNode: AstNode, isInSelection: Boolean, prefix: String = ""): Seq[Suggestion] = {
    properties.flatMap((property) => {
      val key: WeaveType = property.key
      key match {
        case kt: KeyType => {
          val name: WeaveType = kt.name
          name match {
            case NameType(Some(QName(propertyName, ns))) => {
              var suggestionAction: Option[QuickFixAction] = None
              val nsPrefix: String = if (ns.isDefined) {
                val documentNode: AstNode = editor.astNavigator().get.documentNode
                val namespaceDirectives: Seq[NamespaceDirective] = AstNodeHelper.getNamespaceDirectives(documentNode)
                val maybeDirective: Option[NamespaceDirective] = namespaceDirectives.find((nd) => nd.uri.literalValue == ns.get)
                maybeDirective match {
                  case Some(nd) => {
                    nd.prefix.name + "#"
                  }
                  case None => {
                    var i = 0
                    while (namespaceDirectives.exists((nd) => nd.prefix.name == s"ns${i}")) {
                      i = i + 1
                    }
                    suggestionAction = Some(new DeclareNamespaceDeclarationAction(editor, s"ns${i}", ns.get, namespaceDirectives.headOption.getOrElse(documentNode)))
                    s"ns${i}#"
                  }
                }
              } else {
                ""
              }

              val keyname: String = quoteKeyIfRequired(propertyName)
              val propertyQName: String = prefix + nsPrefix + keyname
              val propertyType: WeaveType = property.value
              if (isInSelection) {
                val suggestion: Suggestion = Suggestion(propertyQName, Template(propertyQName), property.getDocumentation(), propertyType, SuggestionType.Field)
                suggestionAction.foreach((sa) => suggestion.withInsertAction(sa))
                Seq(suggestion)
              } else {
                val keyExpression = propertyQName + ": "
                val simpleKeySuggestion: Suggestion = Suggestion(propertyQName, Template(keyExpression), property.getDocumentation(), propertyType, SuggestionType.Field)
                val smartSuggestions: Seq[Suggestion] = suggestExpressionsBasedOnNameType(propertyName, keyExpression, propertyType, astNode)
                val allSuggestions = smartSuggestions :+ simpleKeySuggestion
                //Apply quick fixes to all
                suggestionAction.foreach((sa) =>
                  allSuggestions.foreach((ss) => {
                    ss.withInsertAction(sa)
                  }))
                allSuggestions
              }
            }
            case _ => None
          }

        }
        case _ => None
      }
    })
  }

  private def suggestExpressionsBasedOnNameType(propertyName: String, suggestionPrefix: String, propertyType: WeaveType, location: AstNode) = {
    val maybeScope = editor.scopeGraph().flatMap((scopeGraph) => scopeGraph.scopeOf(location))
    val smartSuggestions: Seq[Suggestion] = if (maybeScope.isDefined) {
      val visibleVariables: Seq[Reference] = maybeScope.get.visibleVariables
      visibleVariables
        .filter((ref) => {
          if (ref.isLocalReference) {
            val maybeVarDirective: Option[VarDirective] = ref.scope.astNavigator().parentWithType(ref.referencedNode, classOf[VarDirective])
            maybeVarDirective match {
              case Some(varDirective) => {
                !AstNodeHelper.containsChild(varDirective.value, location)
              }
              case None => true
            }
          } else {
            true
          }
        })
        .flatMap((ref) => {
          val variableWithType: Option[(String, WeaveType)] = editor
            .typeGraph()
            .flatMap((typeGraph) => {
              val foundRef: Option[TypeNode] = typeGraph.findNode(ref.referencedNode)
              val variableName: String = ref.referencedNode.name
              val resultType = foundRef.flatMap(_.resultType())
              resultType.map((wt) => (variableName, wt))
            })
          variableWithType
        })
        .flatMap((wt) => {
          suggestBasedOnPropertyAndType(wt._1, wt._2, propertyName, suggestionPrefix, propertyType)
        })
    } else {
      Seq()
    }
    smartSuggestions
  }

  private def quoteKeyIfRequired(name: String): String = {
    if (StringEscapeHelper.keyRequiresQuotes(name)) {
      "\"" + name + "\""
    } else {
      name
    }
  }

  private def suggestBasedOnPropertyAndType(expression: String, actualType: WeaveType, expectedPropertyName: String, suggestionPrefix: String, expectedType: WeaveType, depth: Int = 0): Seq[Suggestion] = {
    actualType match {
      case ot: ObjectType => {
        ot.properties.flatMap((prop) => {
          prop.key match {
            case kt: KeyType => {
              kt.name match {
                case NameType(Some(QName(name, ns))) if (name.equalsIgnoreCase(expectedPropertyName) && TypeHelper.canBeAssignedTo(prop.value, expectedType, null)) => {
                  Seq(Suggestion(suggestionPrefix + expression + "." + quoteKeyIfRequired(name), prop.value.getDocumentation(), prop.value))
                }
                case NameType(Some(QName(name, ns))) if (depth < configuration.smartCompletionMaxDepth) => {
                  suggestBasedOnPropertyAndType(expression + "." + name, prop.value, expectedPropertyName, suggestionPrefix, expectedType, depth + 1)
                }
                case _ => Seq()
              }
            }
            case _ => Seq()
          }
        })
      }
      case rt: ReferenceType => {
        val weaveType: WeaveType = rt.resolveType()
        suggestBasedOnPropertyAndType(expression, weaveType, expectedPropertyName, suggestionPrefix, expectedType, depth)
      }
      case ut: UnionType => {
        ut.of.flatMap((wt) => suggestBasedOnPropertyAndType(expression, wt, expectedPropertyName, suggestionPrefix, expectedType, depth))
      }
      case _ => Seq()
    }
  }

  def suggestFullQualifiedNameIdentifier(ni: NameIdentifier, cursorLocation: Int): SuggestionResult = {
    val moduleName = ni.parent().get
    val maybeSuggestions: Option[SuggestionResult] = suggestModuleElements(moduleName).map((suggestions) => {
      val startIndex = ni.location().startPosition.index + moduleName.name.length + NameIdentifier.SEPARATOR.length + ni.loader.map(_.length + 1).getOrElse(0)
      SuggestionResult(suggestions.toArray, startIndex, ni.location().endPosition.index)
    })
    maybeSuggestions.getOrElse(suggestVisibleModules(moduleName, cursorLocation))
  }

  private def suggestModuleElements(ni: NameIdentifier): Option[Seq[Suggestion]] = {
    editor
      .resolveFullQualifiedName(ni)
      .flatMap((moduleName) => {
        val value = editor.getScopeGraphForModule(moduleName)
        val maybeSuggestions: Option[Seq[Suggestion]] = value.mayBeResult.map((scopeGraphResult) => {
          val currentVariableScope = scopeGraphResult.scope.rootScope
          val typeCheckingResult = editor.getTypeCheckingForModule(moduleName).mayBeResult.map(_.typeGraph)
          currentVariableScope
            .declarations()
            .sortBy(_.name.toLowerCase)
            .map((ni) => {
              toSuggestion(ni, currentVariableScope, typeCheckingResult, Some(moduleName))
            })
        })
        maybeSuggestions
      })

  }

  private def suggestVisibleModules(ni: NameIdentifier, cursorLocation: Int, appendSeparator: Boolean = true): SuggestionResult = {
    val isFakeName = isInsertedName(ni.localName().name)
    val name = if (isFakeName) {
      ni.parent().map(_.name).getOrElse("")
    } else {
      ni.name
    }
    val suggestions = suggestModulesWith(name, appendSeparator)
    val startIndex: Int = resolveStartIndex(ni, cursorLocation)
    val endIndex: Int = resolveEndIndex(ni, cursorLocation)
    SuggestionResult(addWhitespaceIfNeeded(ni, suggestions, cursorLocation), startIndex, endIndex)
  }

  private def resolveStartIndex(ni: NameIdentifier, cursorLocation: Int): Int = {
    if (ni.isInstanceOf[MissingNameIdentifierError]) {
      cursorLocation
    } else {
      ni.location().startPosition.index
    }
  }

  private def addWhitespaceIfNeeded(ni: NameIdentifier, suggestions: Array[Suggestion], cursorLocation: Int): Array[Suggestion] = {
    var result: Array[Suggestion] = suggestions
    if (ni.isInstanceOf[MissingNameIdentifierError] && cursorLocation == ni.location().startPosition.index) {
      result = suggestions.map(suggestion => suggestion.copy(template = suggestion.template.prepend(" ")))
    }
    result
  }

  private def resolveEndIndex(ni: NameIdentifier, cursorLocation: Int) = {
    val isFakeName = isInsertedName(ni.localName().name)
    val originalEndIndex = ni.location().endPosition.index
    if (isFakeName && !ni.isInstanceOf[ErrorAstNode]) {
      originalEndIndex - AutoCompletionService.INSERTED_TEXT.length
    } else if (ni.isInstanceOf[MissingNameIdentifierError] && editor.positionOf(cursorLocation).line != ni.location().endPosition.line) {
      cursorLocation
    } else {
      originalEndIndex
    }
  }

  private def suggestModulesWith(filterExpression: String = "", appendSeparator: Boolean = true): Array[Suggestion] = {
    val files: util.Iterator[LocatedResult[WeaveDocument]] = indexService.searchDocumentContainingName(filterExpression)
    val suggestions =
      files.asScala
        .filter(_.value.kind == DocumentKind.MODULE)
        .map((vf) => {
          val nameIdentifier: NameIdentifier = vf.moduleName
          val template: Template = if (appendSeparator) Template(nameIdentifier.name).add("::").placeHolder() else Template(nameIdentifier.name)
          Suggestion(nameIdentifier.localName().name, template, Some(vf.value.documentation), SuggestionType.Module)
        })
    //Limit the size of the elements to suggest

    val suggestionsArray = suggestions.toArray
    suggestionsArray.sortBy(_.name.toLowerCase)
  }

  def selectionOptions(toUpdate: WeaveType, astNode: AstNode, use: UpdateSelectorNode): Seq[Suggestion] = {
    use.child match {
      case Some(usn: UpdateSelectorNode) => {
        dispatchSelectType(toUpdate, use) match {
          case Some(selectedValue) => selectionOptions(selectedValue, astNode, usn)
          case None                => Seq()
        }
      }
      case _ => {
        use match {
          case FieldNameUpdateSelectorNode(NameNode(StringNode(name, _), _, _), _) => {
            suggestObjectFieldsForSelectionWithName(toUpdate, astNode, name)
          }
          case MultiFieldNameUpdateSelectorNode(NameNode(StringNode(name, _), _, _), _) => {
            suggestObjectFieldsForSelectionWithName(toUpdate, astNode, name)
          }
          case AttributeNameUpdateSelectorNode(NameNode(StringNode(name, _), _, _), _) => {
            toUpdate.parentKey
              .map((key) => {
                attributesToSuggestions(key, astNode).filter(_.name.startsWith(name))
              })
              .getOrElse(Seq())
          }
          case ArrayIndexUpdateSelectorNode(_, _) => {
            Seq()
          }
          case _ => {
            Seq()
          }
        }
      }
    }

  }

  private def suggestObjectFieldsForSelectionWithName(toUpdate: WeaveType, astNode: AstNode, name: String): Seq[Suggestion] = {
    toUpdate match {
      case ot: ObjectType =>
        val unfiltered = suggestObjectFields(ot, astNode)
        if (isInsertedName(name)) {
          unfiltered
        } else {
          unfiltered.filter(_.name.startsWith(name))
        }
      case _ => Seq()
    }
  }

  def dispatchSelectType(toUpdate: WeaveType, use: UpdateSelectorNode): Option[WeaveType] = {
    toUpdate match {
      case rt: ReferenceType => {
        dispatchSelectType(rt.resolveType(), use)
      }
      case ut: UnionType => {
        val weaveTypes = ut.of.flatMap((wt) => dispatchSelectType(wt, use))
        if (weaveTypes.isEmpty) {
          None
        } else {
          val unified = TypeHelper.unify(weaveTypes)
          WeaveTypeCloneHelper.copyAdditionalTypeInformation(ut, unified)
          Some(unified)
        }
      }
      case it: IntersectionType => {
        val weaveTypes = it.of.flatMap((wt) => dispatchSelectType(wt, use))
        if (weaveTypes.isEmpty) {
          None
        } else {
          val copied = it.copy(of = weaveTypes)
          WeaveTypeCloneHelper.copyAdditionalTypeInformation(it, copied)
          Some(copied)
        }
      }
      case tp: TypeParameter if (tp.top.isDefined) => {
        dispatchSelectType(tp.top.get, use)
      }
      case _ => {
        selectType(toUpdate, use)
      }
    }
  }

  private def selectType(toUpdate: WeaveType, use: UpdateSelectorNode): Option[WeaveType] = {
    use match {
      case FieldNameUpdateSelectorNode(NameNode(StringNode(name, _), _, _), _) => {
        toUpdate match {
          case ObjectType(kvps, _, _) =>
            kvps
              .find((kvp) => {
                kvp.key match {
                  case KeyType(NameType(Some(keyName)), _) if (keyName.name.equals(name)) => {
                    true
                  }
                  case _ => false
                }
              })
              .map(_.value)
          case _ => None
        }
      }
      case MultiFieldNameUpdateSelectorNode(NameNode(StringNode(name, _), _, _), _) => {
        toUpdate match {
          case ObjectType(kvps, _, _) =>
            kvps
              .find((kvp) => {
                kvp.key match {
                  case KeyType(NameType(Some(keyName)), _) if (keyName.name.equals(name)) => {
                    true
                  }
                  case _ => false
                }
              })
              .map(_.value)
          case _ => None
        }
      }
      case AttributeNameUpdateSelectorNode(NameNode(StringNode(name, _), _, _), _) => {
        toUpdate.parentKey.flatMap((key) => {
          key.attrs
            .find((attr) => {
              attr.name match {
                case NameType(Some(attrName)) if (attrName.name.equals(name)) => true
                case _ => false
              }
            })
            .map(_.value)
        })
      }
      case ArrayIndexUpdateSelectorNode(_, _) => {
        toUpdate match {
          case arrayType: ArrayType => Some(arrayType.of)
          case _                    => None
        }
      }
      case _ => None
    }
  }

  private def suggestions(astNode: AstNode, cursorLocation: Int, navigator: AstNavigator, scopesNavigator: Option[ScopesNavigator], typeGraph: Option[TypeGraph], prefix: Option[String]): SuggestionResult = {
    val result: SuggestionResult =
      astNode match {
        case fcpn: FunctionCallParametersNode => {
          val suggestionOfLiterals: Seq[Suggestion] = suggestLiterals(fcpn, navigator)
          val suggestionOfExpressions: Seq[Suggestion] = suggestParametersExpression(fcpn, navigator)
          val visibleVariables: Seq[Suggestion] =
            if (scopesNavigator.isDefined) {
              val maybeScope: Option[VariableScope] = scopesNavigator.get.scopeOf(fcpn)
              maybeScope match {
                case Some(scope) => {
                  suggestVisibleVariables(scope, typeGraph)
                }
                case None => Seq()
              }
            } else {
              Seq()
            }
          SuggestionResult(astNode, suggestionOfLiterals ++ suggestionOfExpressions ++ visibleVariables)
        }
        case vr: VariableReferenceNode => {
          val suggestionOfLiterals = suggestLiterals(vr, navigator)
          val suggestionOfExpressions: Seq[Suggestion] = {
            navigator.parentOf(vr) match {
              case Some(fcpn: FunctionCallParametersNode) => {
                suggestParametersExpression(fcpn, navigator)
              }
              case _ => Seq()
            }
          }

          if (scopesNavigator.isDefined) {
            val maybeScope = scopesNavigator.get.scopeOf(vr)
            maybeScope match {
              case Some(scope) => {
                val visibleVariables: Seq[Suggestion] = suggestVisibleVariables(scope, typeGraph)
                val options: Seq[Suggestion] = filterTypesAndNamespaces(visibleVariables)
                val suggestions = navigator.parentOf(vr) match {
                  case Some(fcn: FunctionCallNode) if AstNodeHelper.isInfixFunctionCall(fcn) => {
                    val availableFunctionCalls = filterBasedOn(options, typeGraph, fcn.args.args.headOption)
                    val isRecoveryCase = fcn.args.args(1) match {
                      case _: ErrorAstNode => true
                      case _               => false
                    }
                    if (isRecoveryCase) {
                      availableFunctionCalls ++ binaryOperators
                    } else {
                      availableFunctionCalls
                    }
                  }
                  case _ => {
                    val result = suggestVisibleModules(vr.variable, cursorLocation)
                    options ++ result.suggestions ++ templates
                  }
                }
                SuggestionResult(astNode, suggestionOfLiterals ++ suggestionOfExpressions ++ suggestions)
              }
              case None => {
                SuggestionResult(astNode, suggestionOfLiterals ++ suggestionOfExpressions)
              }
            }
          } else {
            SuggestionResult(astNode, suggestionOfLiterals ++ suggestionOfExpressions)
          }
        }
        case tr: TypeSelectorNode => {
          suggestForTypeSelectionNode(scopesNavigator, tr, typeGraph).getOrElse(emptySuggestion(cursorLocation))
        }
        case tr: TypeReferenceNode => {
          if (scopesNavigator.isDefined) {
            val maybeScope = scopesNavigator.get.scopeOf(tr)
            maybeScope match {
              case Some(scope) => {
                val suggestions = suggestVisibleVariables(scope, typeGraph).filter((s) => {
                  s.wtype match {
                    case Some(_: TypeType)      => true
                    case Some(_: NamespaceType) => false
                    case _                      => false
                  }
                })
                SuggestionResult(astNode, suggestions)
              }
              case None => SuggestionResult(astNode)
            }
          } else {
            SuggestionResult(astNode)
          }
        }
        case tr: AnnotationNode => {
          if (scopesNavigator.isDefined) {
            val maybeScope = scopesNavigator.get.scopeOf(tr)
            maybeScope match {
              case Some(vs) => {
                val identifiers: Seq[Reference] = vs.collectVisibleVariables((vs) => {
                  vs.scope.astNavigator().parentWithType(vs.referencedNode, classOf[AnnotationDirectiveNode]).isDefined
                })
                val suggestions = identifiers.map((ref) => {
                  val maybeNode = ref.scope.astNavigator().parentWithType(ref.referencedNode, classOf[AnnotationDirectiveNode])
                  val description = maybeNode.get.weaveDoc.map(_.literalValue)
                  Suggestion(ref.referencedNode.name, Template(ref.referencedNode.name).add("(").placeHolder().add(")"), description, AnyType(), Some(ref))
                })
                SuggestionResult(tr.name, suggestions)
              }
              case _ => SuggestionResult(astNode)
            }
          } else {
            SuggestionResult(astNode)
          }
        }
        case _: ContentType => {
          val suggestions = suggestContentTypes()
          SuggestionResult(astNode, suggestions)
        }
        case tr: NamespaceNode => {
          if (scopesNavigator.isDefined) {
            val maybeScope = scopesNavigator.get.scopeOf(tr)
            maybeScope match {
              case Some(scope) => {
                val suggestions = suggestVisibleVariables(scope, typeGraph).filter((s) => {
                  s.wtype match {
                    case Some(_: TypeType)      => false
                    case Some(_: NamespaceType) => true
                    case _                      => false
                  }
                })
                SuggestionResult(astNode, suggestions)
              }
              case None => SuggestionResult(astNode)
            }
          } else {
            SuggestionResult(astNode)
          }
        }
        case anan: AnnotationArgumentsNode => {
          val suggestions = if (scopesNavigator.isDefined) {
            val maybeNode = navigator.parentWithType(anan, classOf[AnnotationNode])
            maybeNode match {
              case Some(annotationNode) => {
                val maybeReference = scopesNavigator.get.resolveVariable(annotationNode.name)
                maybeReference match {
                  case Some(reference) => {
                    val maybeAnnotationDirectiveNode = reference.scope.astNavigator().parentWithType(reference.referencedNode, classOf[AnnotationDirectiveNode])
                    maybeAnnotationDirectiveNode match {
                      case Some(annotationDirectiveNode) => {
                        val args = anan.args.map(_.name.name)
                        val parameterNodes = annotationDirectiveNode.params.paramList.filter((param) => !args.contains(param.nameIdentifier.name))
                        parameterNodes.map((param) => {
                          Suggestion(param.nameIdentifier.name, Template(param.nameIdentifier.name).add(" = ").placeHolder(), None)
                        })

                      }
                      case None => Seq()
                    }
                  }
                  case None => Seq()
                }

              }
              case None => Seq()
            }
          } else {
            Seq()
          }

          SuggestionResult(anan, suggestions)
        }
        case odn: OutputDirective => {
          val suggestions = suggestWriterOptions(odn)
          SuggestionResult(astNode, suggestions)
        }
        case idn: InputDirective => {
          val suggestions = suggestReaderOptions(idn)
          SuggestionResult(astNode, suggestions)
        }
        case dn: DirectiveOptionName => {
          val suggestions = navigator
            .parentOf(dn)
            .flatMap(navigator.parentOf)
            .map({
              case odn: OutputDirective => {
                suggestWriterOptions(odn)
              }
              case idn: InputDirective => {
                suggestReaderOptions(idn)
              }
              case _ => Seq()
            })
            .getOrElse(Seq())
          SuggestionResult(astNode, suggestions)
        }
        case me: MissingSelectorExpressionErrorAstNode => {
          val parentUpdateNode = navigator.parentWithType(me, classOf[UpdateNode]).get
          val options = typeGraph
            .flatMap((typeGraph) => {
              typeGraph
                .findNode(parentUpdateNode.expression)
                .flatMap((node) => {
                  node.resultType()
                })
            })
            .map((weaveType) => {
              collectFieldSelectionSuggestions(weaveType, astNode, ".")
            })
            .getOrElse(Seq())
          SuggestionResult(me, options)
        }
        case use: UpdateSelectorNode => {
          val parentUpdateNode = navigator.parentWithType(use, classOf[UpdateNode]).get
          val parentExpressionNode = navigator.parentWithType(use, classOf[UpdateExpressionNode]).get

          val options: Seq[Suggestion] = typeGraph
            .flatMap((typeGraph) => {
              typeGraph
                .findNode(parentUpdateNode.expression)
                .flatMap((node) => {
                  node.resultType()
                })
            })
            .map((weaveType) => {

              val selector = parentExpressionNode.selector
              selector match {
                case updateSelectorNode: UpdateSelectorNode => selectionOptions(weaveType, astNode, updateSelectorNode)
                case _ => {
                  weaveType match {
                    case ot: ObjectType => suggestObjectFields(ot, astNode)
                    case _              => Seq()
                  }
                }
              }
            })
            .getOrElse(Seq())
          SuggestionResult(use, options)
        }
        case dOption: DirectiveOption => {
          val suggestions = navigator.parentOf(dOption) match {
            case Some(idn: InputDirective) => {
              findDataFormat(idn.mime.orElse(idn.dataFormat))
                .flatMap((dataFormat) => {
                  val properties = dataFormat.readerProperties
                  propertyValueSuggestions(properties, dOption)
                })
                .getOrElse(Seq())
            }
            case Some(odn: OutputDirective) => {
              findDataFormat(odn.mime.orElse(odn.dataFormat))
                .flatMap((dataFormat) => {
                  val properties = dataFormat.writerProperties
                  propertyValueSuggestions(properties, dOption)
                })
                .getOrElse(Seq())
            }
            case _ => Seq()
          }
          SuggestionResult(dOption.value, suggestions)

        }
        case un: UnaryOpNode if unarySelector(un) => {
          val lhs = un.rhs
          if (typeGraph.isDefined) {
            val suggestions = typeGraph.get.findNode(lhs) match {
              case Some(node) => {
                node
                  .resultType()
                  .map((wtype) => {
                    un.opId match {
                      case AllSchemaSelectorOpId => {
                        wtype.metadataConstraints().map((a) => Suggestion(a.name, None, StringType()))
                      }
                      case DescendantsSelectorOpId => {
                        collectFieldSelectionSuggestions(wtype, astNode)
                      }
                      case AllAttributesSelectorOpId => {
                        collectAttributeSuggestions(wtype)
                      }
                      case _ => Seq()
                    }
                  })
                  .getOrElse(Seq())

              }
              case None => Seq()
            }
            SuggestionResult(un.rhs, suggestions)
          } else {
            SuggestionResult(astNode)
          }
        }
        case bon: BinaryOpNode => {
          val lhs = bon.lhs
          if (typeGraph.isDefined) {
            val suggestions = typeGraph.get.findNode(lhs) match {
              case Some(node) => {
                node
                  .resultType()
                  .map((wtype) => {
                    bon.opId match {
                      case AttributeValueSelectorOpId => {
                        collectAttributeSuggestions(wtype)
                      }
                      case MultiValueSelectorOpId => {
                        collectFieldSelectionSuggestions(wtype, astNode).map((s) => {
                          Suggestion(s.name, ArrayType(s.wtype.get), s.reference)
                        })
                      }
                      case MultiAttributeValueSelectorOpId => {
                        collectAttributeSuggestions(wtype).map((s) => {
                          Suggestion(s.name, ArrayType(s.wtype.get), s.reference)
                        })
                      }
                      case SchemaValueSelectorOpId => {
                        wtype.metadataConstraints().map((a) => Suggestion(a.name, None, StringType()))
                      }
                      case ValueSelectorOpId => {
                        collectFieldSelectionSuggestions(wtype, astNode)
                      }
                      case ObjectKeyValueSelectorOpId => {
                        collectFieldSelectionSuggestions(wtype, astNode).map((s) => {
                          val objectType = ObjectType(Seq(KeyValuePairType(KeyType(NameType(Some(QName(s.name)))), s.wtype.get)))
                          Suggestion(s.name, s.mayBeDocumentation, objectType, s.reference)
                        })
                      }
                      case DynamicSelectorOpId => {
                        collectFieldSelectionSuggestions(wtype, astNode)
                      }
                      case _ => Seq()
                    }

                  })
                  .getOrElse(Seq())
              }
              case None => Seq()
            }
            SuggestionResult(bon.rhs, suggestions)
          } else {
            SuggestionResult(astNode)
          }
        }
        case ni: NameIdentifier if ni.parent().isDefined => {
          suggestFullQualifiedNameIdentifier(ni, cursorLocation)
        }
        case node => {
          navigator.parentOf(node) match {
            case Some(value) => suggestions(value, cursorLocation, navigator, scopesNavigator, typeGraph, prefix)
            case None        => SuggestionResult(astNode)
          }
        }
      }

    if (prefix.isDefined) {
      val suggestions = result.suggestions.filter((s) => {
        var name = s.name
        if (name.startsWith("\"")) {
          name = name.substring(1)
        }
        name.startsWith(prefix.get)
      })
      result.copy(suggestions)
    } else {
      result
    }
  }

  private def suggestForTypeSelectionNode(scopesNavigator: Option[ScopesNavigator], tr: TypeSelectorNode, typeGraph: Option[TypeGraph]): Option[SuggestionResult] = {
    val mayBeType: Option[WeaveType] = typeGraph.flatMap(tpg => tpg.findNode(tr).flatMap(typeNode => typeNode.resultType() match {
      case Some(TypeType(typeToResolve: TypeSelectorType)) => Some(typeToResolve.referencedType.resolveType())
      case _ => None
    })).orElse(scopesNavigator.map(scn =>
      WeaveType(tr.weaveTypeNode, scn.referenceResolver)))
    mayBeType match {
      case Some(weaveType) => {
        val startPosition = tr.selector.location().startPosition
        var endPosition = tr.selector.location().endPosition
        val finalPrefix: String = (tr.selector match {
          case NameNode(StringNode(keyName, _), ns, _) => Some(ns.map {
            case NamespaceNode(prefix) => prefix.name + "#"
            case _                     => None
          }.getOrElse("") + (if (keyName == INSERTED_TEXT) {
            endPosition = tr.selector.location().startPosition
            ""
          } else keyName))
          case _ => None
        }).getOrElse("")
        val suggestions = collectFieldSelectionSuggestions(weaveType, tr)
        Some(SuggestionResult(suggestions.filter(suggestion => {
          suggestion.name.startsWith(finalPrefix)
        }).toArray, startPosition.index, endPosition.index))
      }
      case _ => None
    }
  }

  private def propertyValueSuggestions(properties: Array[DataFormatProperty], dOption: DirectiveOption) = {
    val maybeProperty: Option[DataFormatProperty] = properties
      .find((rp) => {
        rp.name.equals(dOption.name.name)
      })

    maybeProperty.map(toPropertyValueSuggestion)
  }

  private def toPropertyValueSuggestion(prop: DataFormatProperty): Seq[Suggestion] = {
    if (isStringProperty(prop)) {
      prop.values.toSeq.map((value) => {
        Suggestion(value, Template("\"" + value + "\""), Some(prop.description), StringType())
      })
    } else if (isBooleanProperty(prop)) {
      Seq(Suggestion("true", Some(prop.description), BooleanType()), Suggestion("false", Some(prop.description), BooleanType()))
    } else {
      prop.values.toSeq.map((value) => {
        Suggestion(value, Some(prop.description), AnyType())
      })
    }

  }

  private def isStringProperty(prop: DataFormatProperty) = {
    prop.wtype.equalsIgnoreCase("String")
  }

  private def filterTypesAndNamespaces(visibleVariables: Seq[Suggestion]) = {
    val options = visibleVariables.filter((s) => {
      s.wtype match {
        case Some(_: TypeType)      => false
        case Some(_: NamespaceType) => false
        case _                      => true
      }
    })
    options
  }

  private def suggestWriterOptions(odn: OutputDirective): Seq[Suggestion] = {
    findDataFormat(odn.mime.orElse(odn.dataFormat))
      .map((dataFormat) => {
        dataFormat.writerProperties.toSeq.map((property) => {
          toSuggestion(property)
        })
      })
      .getOrElse(Seq())
  }

  private def toSuggestion(property: DataFormatProperty) = {
    val wtype: Option[WeaveType] = WeaveTypeParser.parse(property.wtype, editor.buildParsingContext())
    val options = property.values
    val template = Template(property.name).add("=")
    if (options.isEmpty) {
      if (isBooleanProperty(property)) {
        template.choice(Seq("true", "false"))
      } else {
        template.placeHolder()
      }
    } else if (isStringProperty(property)) {
      template.add("\"").choice(options).add("\"")
    } else {
      template.choice(options)
    }
    Suggestion(property.name, template, Some(property.description), wtype.getOrElse(AnyType()))
  }

  private def isBooleanProperty(property: DataFormatProperty) = {
    property.wtype.equalsIgnoreCase("Boolean")
  }

  private def suggestReaderOptions(idn: InputDirective): Seq[Suggestion] = {
    val mime = idn.mime.orElse(idn.dataFormat)
    findDataFormat(mime)
      .map((dataFormat) => {
        dataFormat.readerProperties.toSeq.map((property) => toSuggestion(property))
      })
      .getOrElse(Seq())
  }

  private def findDataFormat(format: Option[FormatExpression]): Option[DataFormatDescriptor] = {
    val toSearch = format match {
      case Some(ContentType(mime)) => mime
      case Some(DataFormatId(id))  => id
      case _                       => return None
    }
    provider
      .dataFormats()
      .find((df) => {
        df.mimeType.equals(toSearch) || df.id.exists(id => id.equals(toSearch))
      })
  }

  private def suggestContentTypes() = {
    provider.dataFormats().toSeq.map((format) => Suggestion(format.mimeType, Template(format.mimeType), None, StringType()))
  }

  private def suggestFormatIds() = {
    provider
      .dataFormats()
      .toSeq
      .filter(format => format.id.isDefined)
      .map(format => Suggestion(format.id.get, Template(format.id.get), None, StringType()))
  }

  private def unarySelector(un: UnaryOpNode) = {
    un.opId == AllAttributesSelectorOpId || un.opId == AllSchemaSelectorOpId || un.opId == DescendantsSelectorOpId
  }

  private def collectAttributeSuggestions(wtype: WeaveType) = {
    wtype.parentKey match {
      case Some(key) =>
        key.attrs.map((attr) => {
          Suggestion(attr.name.toString, attr.getDocumentation(), attr.value)
        })
      case None => Seq()
    }
  }

  private def suggestVisibleVariables(scope: VariableScope, maybeTypeGraph: Option[TypeGraph]): Seq[Suggestion] = {
    collectVisibleVariables(scope, maybeTypeGraph)
      .filterNot((s) => {
        isInsertedName(s.name)
      })
  }

  private def isInsertedName(name: String) = {
    name == AutoCompletionService.INSERTED_TEXT
  }

  private def collectVisibleVariables(scope: VariableScope, maybeTypeGraph: Option[TypeGraph], moduleName: Option[NameIdentifier] = None): Seq[Suggestion] = {
    var variables = ArrayBuffer[Suggestion]()
    var currentMaybeScope: Option[VariableScope] = Some(scope)
    while (currentMaybeScope.isDefined) {
      val currentVariableScope = currentMaybeScope.get
      val scopeVars = currentVariableScope
        .declarations()
        .map((ni) => {
          toSuggestion(ni, currentVariableScope, maybeTypeGraph, moduleName)
        }).sortBy(_.name.toLowerCase)
      variables.++=(scopeVars)
      currentMaybeScope = currentVariableScope.parentScope
    }
    variables.++=(
      scope
        .rootScope()
        .importedModules()
        .flatMap((directiveScopePair) => {
          val importDirective: ImportDirective = directiveScopePair._1
          val importedElements: ImportedElements = importDirective.subElements
          val importedModuleScope: VariableScope = directiveScopePair._2
          if (importDirective.isImportStart()) {
            val typeCheck: PhaseResult[TypeCheckingResult[ModuleNode]] = editor.getTypeCheckingForModule(importDirective.importedModule.elementName)
            val typeGraph: Option[TypeGraph] = if (typeCheck.hasResult()) {
              Some(typeCheck.getResult().typeGraph)
            } else {
              None
            }
            importedModuleScope.localVariables
              .map((r) => {
                toSuggestion(r.referencedNode, importedModuleScope, typeGraph, Some(importDirective.importedModule.elementName))
              })
              .sortBy(_.name.toLowerCase)
          } else {
            importedElements.elements
              .flatMap((element) => {
                val typeCheck: PhaseResult[TypeCheckingResult[ModuleNode]] = editor.getTypeCheckingForModule(importDirective.importedModule.elementName)
                val typeGraph: Option[TypeGraph] = if (typeCheck.hasResult()) {
                  Some(typeCheck.getResult().typeGraph)
                } else {
                  None
                }
                val maybeReference: Option[Reference] = directiveScopePair._2.rootScope().resolveVariable(element.elementName)
                if (maybeReference.isDefined) {
                  Some(toSuggestion(maybeReference.get.referencedNode, importedModuleScope, typeGraph, Some(importDirective.importedModule.elementName), element.alias))
                } else {
                  None
                }
              })
              .sortBy(_.name.toLowerCase)
          }
        }))
    variables
  }

  private def toSuggestion(ni: NameIdentifier, currentVariableScope: VariableScope, maybeTypeGraph: Option[TypeGraph], moduleName: Option[NameIdentifier] = None, alias: Option[NameIdentifier] = None): Suggestion = {
    val suggestionName = alias.getOrElse(ni).name
    val weaveType = maybeTypeGraph
      .map((typeGraph) => {
        val importedElementNode = typeGraph
          .findNode(ni)
        importedElementNode
          .flatMap(_.resultType())
          .getOrElse(AnyType())
      })
      .getOrElse(AnyType())
    weaveType match {
      case ft: FunctionType => {
        val documentation: String = collectDocumentation(ft)
        Suggestion(
          suggestionName,
          Template(suggestionName).add("(").placeHolder().add(")"),
          Some(documentation),
          ft,
          Some(Reference(ni, currentVariableScope, moduleName)),
          Some(SuggestionFunctionCallType.PREFIX))
      }
      case _ => Suggestion(suggestionName, weaveType.getDocumentation(), weaveType, Some(Reference(ni, currentVariableScope, moduleName)))
    }
  }

  private def collectDocumentation(ft: FunctionType): String = {
    val overloads =
      if (ft.overloads.size > 1) {
        ft.overloads
      } else {
        Seq(ft)
      }

    val documentation = overloads
      .map((overloadedFunction) => {
        s"\n*${getTypeText(overloadedFunction)}*\n" +
          overloadedFunction.getDocumentation().getOrElse("")
      })
      .mkString("\n------------------------\n")
    documentation
  }

  private def getTypeText(weaveType: WeaveType) = {
    val emitter = WeaveTypeEmitter.DEFAULT_MESSAGE_CONFIG_BUILDER.withPrintFunctionConstrains(true).build()
    weaveType.toString(emitter)
  }
}

object AutoCompletionService {
  val INSERTED_TEXT: String = DocumentParser.FAKE_VARIABLE_NAME
}

case class SuggestionResult(suggestions: Array[Suggestion], replacementStart: Int, replacementEnd: Int)

object SuggestionResult {
  def apply(node: AstNode, suggestions: Seq[Suggestion] = Seq()): SuggestionResult = {
    SuggestionResult(suggestions.toArray, node.location().startPosition.index, node.location().endPosition.index)
  }
}

case class Suggestion(name: String, template: Template, mayBeDocumentation: Option[String], wtype: Option[WeaveType], itemType: Int, reference: Option[Reference] = None, functionCallType: Option[Int] = None) {
  private val actions: ArrayBuffer[QuickFixAction] = ArrayBuffer()

  def documentation(): Option[String] = {
    reference
      .flatMap((ref) => {
        val maybeNode = ref.scope.astNavigator().parentOf(ref.referencedNode)
        maybeNode.flatMap(_.comments.find(_.commentType == CommentType.DocComment)).map(_.literalValue)
      })
      .orElse(mayBeDocumentation)
  }

  def insertText: String = template.toVSCodeString

  def markdownDocumentation(): Option[String] = {
    documentation().map(AsciiDocMigrator.toMarkDown)
  }

  def withInsertAction(action: QuickFixAction): Suggestion = {
    actions.+=(action)
    this
  }

  def insertAction(): Array[QuickFixAction] = actions.toArray

  def weavedoc(parsingContext: ParsingContext): Option[WeaveDocumentation] = {
    documentation().map((doc) => {
      val weavedoc = WeaveDocParser.parseDocumentation(doc)
      if (functionCallType.isDefined) {
        functionCallType.get match {
          case SuggestionFunctionCallType.PREFIX => weavedoc.asPrefix(parsingContext)
          case SuggestionFunctionCallType.INFIX  => weavedoc.asInfix(parsingContext)
          case _                                 => weavedoc
        }
      } else {
        weavedoc
      }
    })
  }

}

object SuggestionFunctionCallType {
  val INFIX: Int = 0
  val PREFIX: Int = 1
}

object Suggestion {

  def apply(name: String, wtype: WeaveType, reference: Option[Reference]): Suggestion = {
    Suggestion(name, Template(name), None, wtype, reference)
  }

  def apply(name: String, description: Option[String], wtype: WeaveType, reference: Option[Reference]): Suggestion = {
    Suggestion(name, Template(name), description, wtype, reference)
  }

  def apply(name: String, description: Option[String], wtype: WeaveType): Suggestion = {
    Suggestion(name, Template(name), description, wtype, None)
  }

  def apply(name: String, description: Option[String], wtype: WeaveType, suggestionType: Int): Suggestion = {
    new Suggestion(name, Template(name), description, Some(wtype), suggestionType, None)
  }

  def apply(name: String, template: Template, description: Option[String], wtype: WeaveType, suggestionType: Int): Suggestion = {
    new Suggestion(name, template, description, Some(wtype), suggestionType, None)
  }

  def apply(name: String, template: Template, description: Option[String], wtype: WeaveType): Suggestion = {
    Suggestion(name, template, description, wtype, None)
  }

  def apply(name: String, template: Template, description: Option[String]): Suggestion = {
    new Suggestion(name, template, description, None, SuggestionType.Keyword)
  }

  def apply(name: String, description: Option[String]): Suggestion = {
    new Suggestion(name, Template(name), description, None, SuggestionType.Property)
  }

  def apply(name: String, template: Template, description: Option[String], suggestionType: Int): Suggestion = {
    new Suggestion(name, template, description, None, suggestionType)
  }

  def apply(name: String, template: Template, description: Option[String], wtype: WeaveType, reference: Option[Reference]): Suggestion = {
    Suggestion(name, template, description, wtype, reference, None)
  }

  def apply(name: String, template: Template, description: Option[String], wtype: WeaveType, reference: Option[Reference], functionCallType: Option[Int]): Suggestion = {
    val itemType = wtype match {
      case _: FunctionType => SuggestionType.Function
      case _: TypeType     => SuggestionType.Class
      case _               => SuggestionType.Variable
    }
    new Suggestion(name, template, description, Some(wtype), itemType, reference, functionCallType)
  }
}

case class AutoCompletionConfiguration(smartCompletionMaxDepth: Int = 2)

//This list is taken from the LSP to make it easier
object SuggestionType {
  val Text = 1
  val Method = 2
  val Function = 3
  val Constructor = 4
  val Field = 5
  val Variable = 6
  val Class = 7
  val Interface = 8
  val Module = 9
  val Property = 10
  val Unit = 11
  val Value = 12
  val Enum = 13
  val Keyword = 14
  val Snippet = 15
  val Color = 16
  val File = 17
  val Reference = 18
}