package org.mule.weave.v2.metadata.api

import org.mule.metadata.api.ListableTypeLoader
import org.mule.metadata.api.model.MetadataFormat
import org.mule.metadata.api.model.MetadataType
import org.mule.weave.v2.parser.Message
import org.mule.weave.v2.parser.ModuleParser
import org.mule.weave.v2.parser.ast.AstNodeHelper
import org.mule.weave.v2.parser.ast.header.directives.TypeDirective
import org.mule.weave.v2.parser.location.WeaveLocation
import org.mule.weave.v2.parser.phase.ParsingContext
import org.mule.weave.v2.sdk.ParsingContextFactory
import org.mule.weave.v2.sdk.WeaveResourceFactory
import org.mule.weave.v2.ts.WeaveType

import java.util
import java.util.Optional

class WeaveTypeLoader(content: String, targetFormat: MetadataFormat = new MetadataFormat("Weave", "Weave", "application/dw"), parsingContext: ParsingContext = ParsingContextFactory.createParsingContext()) extends ListableTypeLoader {

  private lazy val parseResult = ModuleParser.parse(ModuleParser.scopePhase(), WeaveResourceFactory.fromContent(content), parsingContext)

  override def load(typeName: String): Optional[MetadataType] = {
    val weaveTypeName = WeaveTypesConverter.toValidWeaveId(typeName)

    parseResult.errorMessages().foreach((messages) => {
      println(messages._2.message + "\nat \n" + messages._1.locationString)
    })
    if (parseResult.hasResult()) {
      val result = parseResult.getResult()
      val node = result.astNode
      val typeDirectives = AstNodeHelper.collectChildrenWith(node, classOf[TypeDirective])
      val typeDefinition = typeDirectives.find((typeDirective) => typeDirective.variable.name.equals(weaveTypeName))
      typeDefinition match {
        case Some(value) =>
          val weaveType = WeaveType(value.typeExpression, result.scope.referenceResolver)
          Optional.of(MuleTypesConverter.toMuleType(weaveType, targetFormat))
        case None => Optional.empty()
      }
    } else {
      Optional.empty()
    }
  }

  def load(typeName: String, errorHandling: (Seq[(WeaveLocation, Message)]) => Unit): Optional[MetadataType] = {
    if (parseResult.hasErrors()) {
      errorHandling(parseResult.errorMessages())
    }
    load(typeName)
  }

  /**
    * Returns the list of types declared in the DW script
    *
    * @return
    */
  def typeNames(): Array[String] = {
    if (parseResult.hasResult()) {
      val result = parseResult.getResult()
      val node = result.astNode
      val typeDirectives = AstNodeHelper.collectChildrenWith(node, classOf[TypeDirective])
      typeDirectives.map(_.variable.name).toArray
    } else {
      Array()
    }
  }

  override def getAllTypes(): util.Map[String, MetadataType] = {
    val list = new util.HashMap[String, MetadataType]()
    if (parseResult.hasResult()) {
      val result = parseResult.getResult()
      val node = result.astNode
      val typeDirectives = AstNodeHelper.collectChildrenWith(node, classOf[TypeDirective])
      typeDirectives.foreach((typeDirective) => {
        val weaveType = WeaveType(typeDirective.typeExpression, result.scope.referenceResolver)
        list.put(typeDirective.variable.name, MuleTypesConverter.toMuleType(weaveType, targetFormat))
      })
    }
    list
  }
}
