package org.mule.weave.v2.module.pojo.writer.entry

import org.mule.weave.v2.cache.service.Cache
import org.mule.weave.v2.core.RuntimeConfigProperties
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.module.commons.java.writer.converter.BaseJavaDataConverter
import org.mule.weave.v2.module.commons.java.writer.entry.PropertyAwareEntry
import org.mule.weave.v2.module.commons.java.writer.entry.WriterEntry
import org.mule.weave.v2.module.pojo.exception.BuildMethodCallFailException
import org.mule.weave.v2.module.pojo.exception.BuilderMethodNotFoundException
import org.mule.weave.v2.module.pojo.exception.InvalidBuilderReturnTypeException
import org.mule.weave.v2.module.pojo.exception.MissingBuildMethodOnBuilderException
import org.mule.weave.v2.module.pojo.writer.converter.JavaDataConverter
import org.mule.weave.v2.parser.location.LocationCapable

import java.lang.reflect.InvocationTargetException
import java.lang.reflect.Method
import scala.collection.mutable
import scala.util.Failure
import scala.util.Success
import scala.util.Try

class BuilderContainerEntry(override val location: LocationCapable, element: Any, expectedClass: Class[_]) extends WriterEntry with PropertyAwareEntry {

  override implicit val converter: BaseJavaDataConverter = JavaDataConverter

  private val definition = BuilderContainerEntry.getBuilderProxy(element.getClass)

  override def write(value: Any): Unit = {}

  override def entryType(): Class[_] = element.getClass

  override def resolveEntryValue(): Any = {
    definition.builderMethod match {
      case Failure(_) => {
        throw new MissingBuildMethodOnBuilderException(location.location(), element.getClass, expectedClass)
      }
      case Success(value) => {
        Try(value.invoke(element)) match {
          case Failure(ite: InvocationTargetException) => {
            val message = if (ite.getCause != null) ite.getCause.getMessage else ite.getMessage
            throw new BuildMethodCallFailException(location.location(), element.getClass, message)
          }
          case Failure(_) => {
            throw new MissingBuildMethodOnBuilderException(location.location(), element.getClass, expectedClass)
          }
          case Success(result) => {
            if (expectedClass.isInstance(result)) {
              result
            } else {
              throw new InvalidBuilderReturnTypeException(location.location(), element.getClass, expectedClass, result.getClass)
            }
          }
        }

      }
    }
  }

  override def createPropertyEntry(location: LocationCapable, qname: QualifiedName, schema: Option[Schema]): WriterEntry = {
    new BuilderPropertyEntry(location, qname.name, element, definition)
  }
}

object BuilderContainerEntry {
  val BUILD_METHOD_NAME = "build"

  private val cache = Cache
    .builder()
    .maximumSize(RuntimeConfigProperties.JAVA_BEAN_CACHE_SIZE)
    //Is should be weak keys to avoid classloader leaks
    .weakKeys()
    .build[Class[_], BuilderProxyDefinition]()

  def getBuilderProxy(clazz: Class[_]): BuilderProxyDefinition = {
    cache.get(clazz, (_) => new BuilderProxyDefinition(clazz))
  }
}

class BuilderProxyDefinition(clazz: Class[_]) {

  lazy val builderMethod: Try[Method] = Try(clazz.getMethod(BuilderContainerEntry.BUILD_METHOD_NAME))
  private val properties = mutable.Map[String, Method]()

  def getPropertyMethod(name: String, location: LocationCapable): Method = {
    properties.getOrElseUpdate(
      name, {
      clazz.getMethods
        .find(p => p.getName.equals(name) && p.getParameterCount == 1)
        .getOrElse(throw new BuilderMethodNotFoundException(location.location(), name, clazz))
    })
  }

}
