package org.mule.weave.v2.module.pojo.writer.converter

import org.mule.weave.v2.core.env.WeaveRuntime
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema

import scala.collection.mutable

/**
  * Converts to a given type. Are used by the Java Writer to transform a value to a specific java class.
  */
trait JavaCustomConverter {

  /**
    * Returns true if it can handle the transformation from this source and target
    *
    * @param sourceClass The source of the transformation
    * @param targetClass The targetType
    * @return True if it can handle otherwise false
    */
  def handles(sourceClass: Class[_], targetClass: Class[_]): Boolean

  /**
    * Tries to convert the source to a given target.
    *
    * @param source The value to transform
    * @param schema The schema of that value
    * @param targetClass The target class
    * @return The transformation if it was possible
    */
  def convert(source: Any, schema: Option[Schema], targetClass: Class[_])(implicit ctx: EvaluationContext): Option[Any]
}

object JavaCustomConverter {

  lazy val converters: Seq[JavaCustomConverter] = {
    WeaveRuntime.getServiceProvider().serviceImplementations(classOf[JavaCustomConverter])
  }

  private val cache = mutable.Map[Class[_], mutable.Map[Class[_], Option[JavaCustomConverter]]]()

  def convert(source: Any, schema: Option[Schema], clazz: Class[_])(implicit ctx: EvaluationContext): Option[Any] = {
    if (converters.isEmpty || source == null) {
      return None
    }

    val sourceClass = source.getClass
    val targetCache = cache.getOrElseUpdate(sourceClass, mutable.Map[Class[_], Option[JavaCustomConverter]]())
    val maybeConverter = targetCache.getOrElseUpdate(clazz, converters.find((converter) => converter.handles(sourceClass, clazz)))
    if (maybeConverter.isDefined) {
      try {
        maybeConverter.get.convert(source, schema, clazz)
      } catch {
        case e: Exception => None
      }
    } else {
      None
    }
  }

}
