package org.mule.weave.v2.module.pojo.reader

import org.mule.weave.v2.cache.service.Cache
import org.mule.weave.v2.core.RuntimeConfigProperties.MAX_JAVA_MAPPER_CACHE_SIZE
import org.mule.weave.v2.core.env.WeaveRuntime
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode

/**
  * Maps a java value to a Data Weave Value
  */
trait JavaValueMapper {

  def javaType: Class[_]

  def map(value: Any, loc: () => String): Value[_]

  def typeNode(value: Class[_]): WeaveTypeNode
}

object JavaValueMapper {

  private lazy val valueMappers: Seq[JavaValueMapper] = {
    WeaveRuntime.getServiceProvider().serviceImplementations(classOf[JavaValueMapper])
  }

  private lazy val valueMapperResolver = Cache
    .builder()
    .maximumSize(MAX_JAVA_MAPPER_CACHE_SIZE)
    .weakKeys()
    .build[Class[_], Option[JavaValueMapper]]((key: Class[_]) => valueMappers.find(typeMapper => typeMapper.javaType.isAssignableFrom(key)))

  def map(value: Any, loc: () => String): Option[Value[_]] = {
    if (valueMappers.isEmpty) {
      None
    } else {
      val maybeMapper: Option[JavaValueMapper] = valueMapperResolver.get(value.getClass)
      maybeMapper.map(_.map(value, loc))
    }
  }

  def typeNode(clazz: Class[_]): Option[WeaveTypeNode] = {
    if (valueMappers.isEmpty) {
      None
    } else {
      val maybeMapper: Option[JavaValueMapper] = valueMapperResolver.get(clazz)
      maybeMapper.map(_.typeNode(clazz))
    }
  }
}
