package org.mule.weave.v2.module.java

import org.mule.weave.v2.grammar.literals.TypeLiteral
import org.mule.weave.v2.module.commons.java.JavaTypesHelper
import org.mule.weave.v2.module.pojo.reader.JavaValueMapper
import org.mule.weave.v2.parser.ast.types.KeyTypeNode
import org.mule.weave.v2.parser.ast.types.KeyValueTypeNode
import org.mule.weave.v2.parser.ast.types.NameTypeNode
import org.mule.weave.v2.parser.ast.types.ObjectTypeNode
import org.mule.weave.v2.parser.ast.types.TypeReferenceNode
import org.mule.weave.v2.parser.ast.types.UnionTypeNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.mule.weave.v2.parser.ast.variables.NameIdentifier

import java.beans.Introspector.getBeanInfo
import java.beans.PropertyDescriptor
import java.lang.reflect.GenericArrayType
import java.lang.reflect.ParameterizedType
import java.lang.reflect.Type
import java.lang.reflect.TypeVariable
import java.lang.reflect.WildcardType
import java.util.Optional
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

object JavaClassHelper {

  val INVALID_TYPE_CHARACTER_REGEX = "[^a-zA-Z0-9_]+";

  /**
    * @param theType The Java Type to be matched to a DataWeaveType (WeaveTypeNode)
    * @param stack If a Type is being already processed to generate its WeaveTypeNode, we should avoid processing it again, and just add a TypeReference.
    * @param expanded If set to true and theType is a JavaBean, it will generate an ObjectType and expand that ObjectType matching its fields to the corresponding DW type for each property in the Bean.
    * @param typeDirectiveIdSuffix The suffix used in the declaration of the Type. Generally is the suffix "Class", and it is used in the type declarations using the JavaModuleLoader.
    * @param rootClassName The name of the root class. Needed to create a TypeReference to the typeDirectiveIdSuffix when a property of a Bean points to the rootClass.
    * @param typesReferences The typeReferences being generated so their declarations can be added.
    * @return A WeaveTypeNode representing theType passed as input.
    */
  //This mapping should be in synch with the one in JavaValue
  def toWeaveType(theType: Type, stack: ArrayBuffer[Type] = ArrayBuffer[Type](), expanded: Boolean = false, typeDirectiveIdSuffix: Option[String] = None, rootClassName: Option[String] = None, typesReferences: mutable.Map[String, WeaveTypeNode] = mutable.Map()): WeaveTypeNode = {
    if (stack.exists((actual) => actual eq theType) && typeDirectiveIdSuffix.isDefined) {
      if (rootClassName.isDefined && theType.getTypeName == rootClassName.get) {
        TypeReferenceNode(NameIdentifier(typeDirectiveIdSuffix.get))
      } else {
        TypeReferenceNode(NameIdentifier(referenceTypeName(theType.getTypeName, typeDirectiveIdSuffix)))
      }
    } else if (typesReferences.nonEmpty && typeDirectiveIdSuffix.isDefined && typesReferences.contains(referenceTypeName(theType.getTypeName, typeDirectiveIdSuffix))) {
      TypeReferenceNode(NameIdentifier(referenceTypeName(theType.getTypeName, typeDirectiveIdSuffix)))
    } else {
      stack.+=(theType)
      val result = theType match {
        case clazz: Class[_] => {
          val mappedTypeNode = JavaValueMapper.typeNode(clazz)
          if (mappedTypeNode.isDefined) {
            mappedTypeNode.get
          } else if (JavaTypesHelper.isByteArray(clazz) || JavaTypesHelper.isInputStream(clazz) || JavaTypesHelper.isByte(clazz)
            || JavaTypesHelper.isByteBuffer(clazz) || JavaTypesHelper.isBlob(clazz) || JavaTypesHelper.isFile(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.BINARY_TYPE_NAME))
          } else if ((JavaTypesHelper.isIterableType(clazz)
            || JavaTypesHelper.isIteratorType(clazz)
            || JavaTypesHelper.isArray(clazz)) && !JavaTypesHelper.isSQLException(clazz)) {
            val componentType = clazz.getComponentType
            val typeArguments = if (componentType != null) {
              Some(Seq(toWeaveType(componentType, stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences)))
            } else {
              None
            }
            TypeReferenceNode(NameIdentifier(TypeLiteral.ARRAY_TYPE_NAME), typeArguments)
          } else if (JavaTypesHelper.isBoolean(clazz) || JavaTypesHelper.isAtomicBoolean(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.BOOLEAN_TYPE_NAME))
          } else if (JavaTypesHelper.isString(clazz) || JavaTypesHelper.isChar(clazz) || JavaTypesHelper.isCharSequence(clazz)
            || JavaTypesHelper.isEnum(clazz) || JavaTypesHelper.isClass(clazz) || JavaTypesHelper.isReader(clazz)
            || JavaTypesHelper.isClob(clazz) || JavaTypesHelper.isUUID(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.STRING_TYPE_NAME), None)
          } else if (JavaTypesHelper.isInt(clazz) || JavaTypesHelper.isShort(clazz) || JavaTypesHelper.isLong(clazz)
            || JavaTypesHelper.isBigInteger(clazz) || JavaTypesHelper.isBigDecimal(clazz) || JavaTypesHelper.isFloat(clazz)
            || JavaTypesHelper.isDouble(clazz) || JavaTypesHelper.isAtomicInteger(clazz) || JavaTypesHelper.isAtomicLong(clazz)
            || JavaTypesHelper.isNumber(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.NUMBER_TYPE_NAME), None)
          } else if (JavaTypesHelper.isLocalTime(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.LOCALTIME_TYPE_NAME), None)
          } else if (JavaTypesHelper.isOffsetTime(clazz) ||
            JavaTypesHelper.isSqlTime(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.TIME_TYPE_NAME), None)
          } else if (JavaTypesHelper.isCalendar(clazz) || JavaTypesHelper.isXmlCalendar(clazz) || JavaTypesHelper.isZonedDateTime(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.DATETIME_TYPE_NAME), None)
          } else if (JavaTypesHelper.isDate(clazz) || JavaTypesHelper.isLocalDateTime(clazz)
            || JavaTypesHelper.isSqlTimestamp(clazz) || JavaTypesHelper.isInstant(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.LOCALDATETIME_TYPE_NAME), None)
          } else if (JavaTypesHelper.isSqlDate(clazz) || JavaTypesHelper.isLocalDate(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.DATE_TYPE_NAME), None)
          } else if (JavaTypesHelper.isTimeZone(clazz)
            || JavaTypesHelper.isZoneId(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.TIMEZONE_TYPE_NAME), None)
          } else if (clazz.equals(classOf[Object])) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME), None)
          } else if (JavaTypesHelper.isFunctionValue(clazz)) {
            //There is no way to model a Function with n amount of arguments so we should use Any // :(
            TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME), None)
          } else if (JavaTypesHelper.isOptionalInt(clazz) || JavaTypesHelper.isOptionalLong(clazz) || JavaTypesHelper.isOptionalDouble(clazz)) {
            UnionTypeNode(Seq(TypeReferenceNode(NameIdentifier(TypeLiteral.NUMBER_TYPE_NAME)), TypeReferenceNode(NameIdentifier(TypeLiteral.NULL_TYPE_NAME))))
          } else if (JavaTypesHelper.isVoid(clazz)) {
            TypeReferenceNode(NameIdentifier(TypeLiteral.NULL_TYPE_NAME))
          } else {
            if (expanded) {
              val propertyDescriptors = loadProperties(clazz)
              val properties = propertyDescriptors.filter(_.getPropertyType != null).map((pd) => {
                val value = toWeaveType(getPropertyType(pd), stack, expanded = false, typeDirectiveIdSuffix, rootClassName, typesReferences)
                value match {
                  case UnionTypeNode(elems, _, _, _) if elems.exists(isNullType) =>
                    KeyValueTypeNode(KeyTypeNode(NameTypeNode(Some(pd.getName))), value, repeated = false, optional = true)
                  case TypeReferenceNode(nameIdentifier, _, _, _, _) if nameIdentifier.name == TypeLiteral.NULL_TYPE_NAME =>
                    KeyValueTypeNode(KeyTypeNode(NameTypeNode(Some(pd.getName))), value, repeated = false, optional = true)
                  case _ => KeyValueTypeNode(KeyTypeNode(NameTypeNode(Some(pd.getName))), UnionTypeNode(Seq(value, TypeReferenceNode(NameIdentifier(TypeLiteral.NULL_TYPE_NAME)))), repeated = false, optional = true)
                }
              })
              ObjectTypeNode(properties)
            } else {
              if (typeDirectiveIdSuffix.isDefined) {
                if (rootClassName.isDefined && clazz.getName == rootClassName.get) {
                  TypeReferenceNode(NameIdentifier(typeDirectiveIdSuffix.get))
                } else {
                  val typeRefName = referenceTypeName(clazz.getName, typeDirectiveIdSuffix)
                  typesReferences.put(typeRefName, toWeaveType(clazz, stack.take(stack.size - 1), expanded = true, typeDirectiveIdSuffix, rootClassName, typesReferences))
                  TypeReferenceNode(NameIdentifier(typeRefName))
                }
              } else {
                ObjectTypeNode(Seq())
              }
            }
          }
        }
        case pt: ParameterizedType => {
          val rawType = pt.getRawType
          rawType match {
            case cl: Class[_] => {
              val mappedTypeNode = JavaValueMapper.typeNode(cl)
              if (mappedTypeNode.isDefined) {
                mappedTypeNode.get
              } else if (classOf[Optional[_]].isAssignableFrom(cl) && pt.getActualTypeArguments.length == 1) {
                UnionTypeNode(Seq(
                  toWeaveType(pt.getActualTypeArguments()(0), stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences),
                  TypeReferenceNode(NameIdentifier(TypeLiteral.NULL_TYPE_NAME))))
              } else if ((JavaTypesHelper.isIterableType(cl) || JavaTypesHelper.isIteratorType(cl) || JavaTypesHelper.isArray(cl)) && pt.getActualTypeArguments.length == 1) {
                TypeReferenceNode(NameIdentifier(TypeLiteral.ARRAY_TYPE_NAME), Some(Seq(toWeaveType(pt.getActualTypeArguments()(0), stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences))))
              } else if (JavaTypesHelper.isMap(cl)) {
                if (pt.getActualTypeArguments.length == 2) {
                  ObjectTypeNode(Seq(KeyValueTypeNode(KeyTypeNode(NameTypeNode(None)), toWeaveType(pt.getActualTypeArguments()(1), stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences), repeated = false, optional = true)))
                } else {
                  ObjectTypeNode(Seq())
                }
              } else {
                toWeaveType(rawType, stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences)
              }
            }
            case _ => toWeaveType(rawType, stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences)
          }
        }
        case gat: GenericArrayType => {
          TypeReferenceNode(NameIdentifier(TypeLiteral.ARRAY_TYPE_NAME), Some(Seq(toWeaveType(gat.getGenericComponentType, stack, expanded, typeDirectiveIdSuffix, rootClassName, typesReferences))))
        }
        case wt: WildcardType => {
          TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME), None)
        }
        case tv: TypeVariable[_] => {
          TypeReferenceNode(NameIdentifier(TypeLiteral.ANY_TYPE_NAME), None)
        }
      }
      stack.remove(stack.size - 1)
      result
    }
  }

  private def referenceTypeName(fullTypeName: String, typeDirectiveIdSuffix: Option[String]) = {
    fullTypeName.replaceAll(INVALID_TYPE_CHARACTER_REGEX, "_") ++ typeDirectiveIdSuffix.get
  }

  private def loadProperties(clazz: Class[_]): Seq[PropertyDescriptor] = {
    val propertyDescriptors: Seq[PropertyDescriptor] = if (clazz.isInterface) {
      getBeanInfo(clazz).getPropertyDescriptors.toSeq ++ getSuperInterfaces(clazz).flatMap((c) => getBeanInfo(c).getPropertyDescriptors.toSeq)
    } else {
      getBeanInfo(clazz, classOf[Any]).getPropertyDescriptors.toSeq
    }
    propertyDescriptors
      .filter((pd) => pd.getReadMethod != null || pd.getWriteMethod != null)
  }

  def getSuperInterfaces(clazz: Class[_]): ArrayBuffer[Class[_]] = {
    val superInterfaces = new ArrayBuffer[Class[_]]();
    collectSuperInterfaces(clazz, superInterfaces)
    superInterfaces
  }

  private def collectSuperInterfaces(clazz: Class[_], superInterfaces: ArrayBuffer[Class[_]]): Unit = {
    for (superInterface <- clazz.getInterfaces) {
      superInterfaces.+=(superInterface)
      collectSuperInterfaces(superInterface, superInterfaces)
    }
  }

  private def getPropertyType(pd: PropertyDescriptor): Type = {
    if (pd.getReadMethod != null) pd.getReadMethod.getGenericReturnType
    else pd.getWriteMethod.getParameterTypes.head
  }

  private def isNullType(wt: WeaveTypeNode): Boolean = {
    wt match {
      case tr: TypeReferenceNode => tr.variable.name == TypeLiteral.NULL_TYPE_NAME
      case _                     => false
    }
  }
}

