package org.mule.weave.v2.module.commons.java.writer.converter

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.types.StringType

import java.time._
import java.time.temporal.ChronoField
import java.util.Calendar
import java.util.Date
import java.util.TimeZone
import scala.util.Try

class CalendarDataConverter extends DataConverter[Calendar] {
  override def convert(source: Any, schema: Option[Schema])(implicit ctx: EvaluationContext): Option[Calendar] = {
    val calendar: Calendar = source match {
      case time: LocalDateTime => {
        val instance: Calendar = Calendar.getInstance()
        instance.set(time.getYear, time.getMonthValue - 1, time.getDayOfMonth, time.getHour, time.getMinute, time.getSecond)
        instance.set(Calendar.MILLISECOND, time.get(ChronoField.MILLI_OF_SECOND))
        instance
      }
      case time: ZonedDateTime => {
        asCalendar(time)
      }
      case time: LocalTime => {
        val instance: Calendar = Calendar.getInstance()
        instance.set(Calendar.HOUR_OF_DAY, time.getHour)
        instance.set(Calendar.MINUTE, time.getMinute)
        instance.set(Calendar.SECOND, time.getSecond)
        instance.set(Calendar.MILLISECOND, time.get(ChronoField.MILLI_OF_SECOND))
        instance
      }
      case time: OffsetTime => {
        val instance: Calendar = Calendar.getInstance(TimeZone.getTimeZone(time.getOffset.getId))
        instance.set(Calendar.HOUR_OF_DAY, time.getHour)
        instance.set(Calendar.MINUTE, time.getMinute)
        instance.set(Calendar.SECOND, time.getSecond)
        instance.set(Calendar.MILLISECOND, time.get(ChronoField.MILLI_OF_SECOND))
        instance
      }
      case time: LocalDate => {
        val instance: Calendar = Calendar.getInstance()
        instance.set(Calendar.MONTH, time.getMonthValue - 1)
        instance.set(Calendar.DAY_OF_MONTH, time.getDayOfMonth)
        instance.set(Calendar.YEAR, time.getYear)
        instance.set(Calendar.HOUR_OF_DAY, 0)
        instance.set(Calendar.MINUTE, 0)
        instance.set(Calendar.SECOND, 0)
        instance.set(Calendar.MILLISECOND, 0)
        instance
      }
      case long: Long => {
        val instance: Calendar = Calendar.getInstance()
        instance.setTime(new Date(long))
        instance
      }
      case str: StringType.T => {
        val zonedDateTime: ZonedDateTime = fallbackStringCoercion(str)
        asCalendar(zonedDateTime)
      }
      case cal: Calendar => cal
      case _             => null
    }
    Option(calendar)
  }

  def asCalendar(time: ZonedDateTime): Calendar = {
    val instance: Calendar = Calendar.getInstance(getTimeZone(time))
    instance.set(time.getYear, time.getMonthValue - 1, time.getDayOfMonth, time.getHour, time.getMinute, time.getSecond)
    instance.set(Calendar.MILLISECOND, time.get(ChronoField.MILLI_OF_SECOND))
    instance
  }

  def getTimeZone(time: ZonedDateTime): TimeZone = {
    time.getZone.getId match {
      case a: String if a.matches("[+-][0-9]{2}:[0-9]{2}") => TimeZone.getTimeZone("GMT" + time.getZone.getId)
      case _ => TimeZone.getTimeZone(time.getZone.getId)
    }
  }

  def fallbackStringCoercion(str: StringType#T): ZonedDateTime = {
    Try(ZonedDateTime.parse(str))
      .orElse(Try(LocalDateTime.parse(str).atZone(ZoneOffset.UTC)))
      .get
  }
}
