/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.domain.message.request;


import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.util.MultiMap;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.http.HttpConstants.Method;
import org.mule.sdk.api.http.HttpService;
import org.mule.sdk.api.http.domain.HttpProtocolVersion;
import org.mule.sdk.api.http.domain.message.HttpMessageBuilder;

import java.net.URI;

/**
 * Builder of {@link HttpRequest}s. Instances can only be obtained using {@link HttpService#requestBuilder()}. At the very least,
 * the request URI needs to be provided via {@link #uri(String)}. By default, GET is used as method with empty headers, query
 * params and entity.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface HttpRequestBuilder extends HttpMessageBuilder<HttpRequestBuilder, HttpRequest> {

  /**
   * Declares the URI where this {@link HttpRequest} will be sent. Minimum required configuration.
   *
   * @param uri the URI (as a String) of the {@link HttpRequest} desired. Non null.
   * @return this builder
   */
  HttpRequestBuilder uri(String uri);

  /**
   * Declares the URI where this {@link HttpRequest} will be sent. Minimum required configuration.
   *
   * @param uri the URI of the {@link HttpRequest} desired. Non null.
   * @return this builder
   */
  HttpRequestBuilder uri(URI uri);

  /**
   * Allows for using extension methods, as defined in the rfc. In general, {@link #method(Method)} should be used.
   *
   * @param method the HTTP method of the {@link HttpRequest} desired. Non null.
   * @return this builder
   */
  HttpRequestBuilder method(String method);

  /**
   * @param method the HTTP method of the {@link HttpRequest} desired. Non null.
   * @return this builder
   */
  HttpRequestBuilder method(Method method);

  /**
   * @param protocol the HTTP protocol of the {@link HttpRequest} desired. Non null.
   * @return this builder
   */
  HttpRequestBuilder protocol(HttpProtocolVersion protocol);

  /**
   * @param queryParams a {@link MultiMap} representing the HTTP query params of the {@link HttpRequest} desired. Non null.
   * @return this builder
   */
  HttpRequestBuilder queryParams(MultiMap<String, String> queryParams);

  /**
   * Includes a new queryParam to be sent in the desired {@link HttpRequest}.
   *
   * @param name  the name of the HTTP queryParam
   * @param value the value of the HTTP queryParam
   * @return this builder
   */
  HttpRequestBuilder addQueryParam(String name, String value);

  /**
   * @return the current URI configured in the builder.
   */
  URI getUri();

  /**
   * @return the current HTTP method configured in the builder.
   */
  String getMethod();

  /**
   * @return an immutable version of the current query parameters in the builder.
   */
  MultiMap<String, String> getQueryParams();
}
