/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.http.client;

import org.mule.api.annotation.Experimental;
import org.mule.api.annotation.NoImplement;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.sdk.api.annotation.MinMuleVersion;
import org.mule.sdk.api.http.client.proxy.ProxyConfigurer;
import org.mule.sdk.api.http.protocol.Http1ProtocolConfigurer;
import org.mule.sdk.api.http.protocol.Http2ProtocolConfigurer;
import org.mule.sdk.api.http.tcp.TcpSocketPropertiesConfigurer;

import java.util.function.Consumer;

/**
 * Allows configuring a {@link HttpClient}.
 *
 * @since 0.12.0
 */
@Experimental
@NoImplement
@MinMuleVersion("4.10.0")
public interface HttpClientConfigurer {

  /**
   * Required exclusively for HTTPS, this defines through a {@link TlsContextFactory} all the TLS related data to establish such
   * connections. Set to {@code null} by default.
   *
   * @param tlsContextFactory a {@link TlsContextFactory} with the required data.
   * @return this configurer
   */
  HttpClientConfigurer setTlsContextFactory(TlsContextFactory tlsContextFactory);

  /**
   * Defines the maximum number of outbound connections that will be kept open at the same time. Unlimited by default.
   *
   * @param maxConnections number of connections to allow
   * @return this configurer
   */
  HttpClientConfigurer setMaxConnections(int maxConnections);

  /**
   * Defines if connections should be kept after a request is completed or closed. Default value is {@code true}.
   *
   * @param usePersistentConnections {@code boolean} specifying the decision
   * @return this configurer
   */
  HttpClientConfigurer setUsePersistentConnections(boolean usePersistentConnections);

  /**
   * Defines the number of milliseconds that a connection can remain idle before being closed. Only relevant if persistent
   * connections are used, the default value is 30 seconds.
   *
   * @param connectionIdleTimeout timeout value (in milliseconds)
   * @return this configurer
   */
  HttpClientConfigurer setConnectionIdleTimeout(int connectionIdleTimeout);

  /**
   * Defines whether the HTTP response should be streamed, meaning processing will continue as soon as all headers are parsed and
   * the body populated as it arrives. Default is {@code false}.
   * <p/>
   * When streaming is enabled, because of the internal buffer used to hold the arriving body chunks, the response MUST be
   * eventually read or the worker threads handling the chunks will block waiting to allocate them. Likewise, read/write speed
   * differences could cause issues. Buffer size can be customized for these reasons. Additionally, to avoid deadlocks, a hand off
   * to another thread MUST be performed before consuming the response.
   *
   * @param streaming whether or not to enable streaming
   * @return this configurer
   */
  HttpClientConfigurer setStreaming(boolean streaming);

  /**
   * Defines the size of the buffer in bytes used to store the HTTP response, the default value is 10KB.
   *
   * @param responseBufferSize buffer size (in bytes)
   * @return this configurer
   */
  HttpClientConfigurer setResponseBufferSize(int responseBufferSize);

  /**
   * Defines the name of the {@link HttpClient}. Must be specified.
   *
   * @param name a {@link String} representing the prefix
   * @return this configurer
   */
  HttpClientConfigurer setName(String name);

  /**
   * Defines whether responses should be decompressed automatically by the {@link HttpClient}, {@code false} by default. Note that
   * only GZIP encoding is supported.
   *
   * @param decompress whether responses should be decompressed
   * @return this configurer
   */
  HttpClientConfigurer setDecompress(Boolean decompress);

  /**
   * Optionally defines TCP specific properties like the socket connection timeout, for example, via a
   * {@link TcpSocketPropertiesConfigurer}.
   *
   * @param configCallback a callback to set the tcp properties.
   * @return this configurer
   */
  HttpClientConfigurer configClientSocketProperties(Consumer<TcpSocketPropertiesConfigurer> configCallback);

  /**
   * Required when connecting through a proxy, this defines the relevant data to do so using a {@link ProxyConfigurer}.
   *
   * @param configCallback a callback to configure the {@link ProxyConfigurer}
   * @return this configurer
   */
  HttpClientConfigurer configProxy(Consumer<ProxyConfigurer> configCallback);

  /**
   * Allows enabling/disabling and configuring HTTP/1 specific parameters.
   *
   * @param configCallback a callback to configure the {@link Http1ProtocolConfigurer}
   * @return this configurer
   */
  HttpClientConfigurer configHttp1(Consumer<Http1ProtocolConfigurer> configCallback);

  /**
   * Allows enabling/disabling and configuring HTTP/2 specific parameters.
   *
   * @param configCallback a callback to configure the {@link Http2ProtocolConfigurer}
   * @return this configurer
   */
  HttpClientConfigurer configHttp2(Consumer<Http2ProtocolConfigurer> configCallback);
}
