/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.sdk.api.connectivity;

import org.mule.runtime.api.config.PoolingProfile;
import org.mule.sdk.api.annotation.MinMuleVersion;

import javax.transaction.xa.XAResource;

/**
 * Specialization of {@link ConnectionProvider} for connections that may participate in a XA Transaction.
 * <p>
 * The <b>TransactionManager</b> within the Mule Runtime will handle the {@link XAResource} for the provided connections (ref:
 * {@link XATransactionalConnection#getXAResource()}).
 * <p>
 * This interface will mark a {@link ConnectionProvider} as its connections <i>may</i> participate in a XA transaction. It will be
 * up to the implementation/user config to actually participate in an XA transaction, a local transaction or no transaction at
 * all.
 *
 * @param <C> the generic type of the connections that may support XA to be handled
 * @since 1.0
 */
@MinMuleVersion("4.10")
public interface XATransactionalConnectionProvider<C extends TransactionalConnection> extends ConnectionProvider<C> {

  /**
   * Allows to configure the connection pool to be used exclusively for XA. This is useful for when a connection pool is not
   * required for local transactions or non-transactional cases.
   * <p>
   * XA support may require connections to be pooled so that:
   * <ul>
   * <li>If more than one session cannot be used within the same connection, getting a new connection is not an expensive
   * operation.</li>
   * <li>Transaction management tasks that may need a connection (i.e.: transaction recovery) have a connection ready for
   * use.</li>
   * </ul>
   * <p>
   * This configuration will be added to the connection management for this {@link ConnectionProvider} <b>only</b> for
   * {@link XATransactionalConnection}s. Other connections will use the default connection management determined for this
   * {@link ConnectionProvider}.
   * <p>
   * If this connectionProvider already implements {@link PoolingConnectionProvider}, the pooling connection configured in the
   * application will be used instead of calling this method.
   *
   * @return the pooling configuration to use for XA connections provided by this {@link ConnectionProvider}.
   */
  PoolingProfile getXaPoolingProfile();

}
