/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.module.apikit.validation.body.schema.v1;

import static org.mule.module.apikit.ApikitErrorTypes.throwErrorType;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.dom.DOMSource;
import javax.xml.validation.Schema;
import javax.xml.validation.Validator;

import org.mule.module.apikit.api.exception.BadRequestException;
import org.mule.module.apikit.validation.body.schema.IRestSchemaValidatorStrategy;
import org.mule.runtime.api.exception.ErrorTypeRepository;
import org.mule.runtime.core.api.util.xmlsecurity.XMLSecureFactories;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class RestXmlSchemaValidator implements IRestSchemaValidatorStrategy {

  protected static final Logger logger = LoggerFactory.getLogger(RestXmlSchemaValidator.class);

  private Schema schema;
  private ErrorTypeRepository errorTypeRepository;

  public RestXmlSchemaValidator(Schema schemaCache) {
    this(schemaCache, null);
  }

  public RestXmlSchemaValidator(Schema schemaCache, ErrorTypeRepository errorTypeRepository) {
    this.schema = schemaCache;
    this.errorTypeRepository = errorTypeRepository;
  }

  @Override
  public void validate(String payload) throws BadRequestException {
    try {
      Document data = loadDocument(new StringReader(payload));

      Validator validator = schema.newValidator();
      validator.validate(new DOMSource(data.getDocumentElement()));

    } catch (IOException | SAXException e) {
      logger.info("Schema validation failed: " + e.getMessage());
      throw throwErrorType(new BadRequestException(e), errorTypeRepository);
    }

  }

  private static Document loadDocument(Reader reader) throws IOException {
    return loadDocument(new InputSource(reader));
  }

  /**
   * Loads the document from the <code>content</code>.
   *
   * @param source the content to load
   * @return the {@link Document} represents the DOM of the content
   * @throws IOException
   */
  private static Document loadDocument(InputSource source) throws IOException {
    DocumentBuilderFactory factory = XMLSecureFactories.createDefault().getDocumentBuilderFactory();
    factory.setNamespaceAware(true);
    try {
      DocumentBuilder builder = factory.newDocumentBuilder();
      // Setting error handler to null to avoid logs generated by the parser.
      builder.setErrorHandler(null);
      return builder.parse(source);
    } catch (ParserConfigurationException e) {
      throw new IOException("An internal operation failed.", e);
    } catch (SAXException e) {
      throw new IOException("An internal operation failed.", e);
    }
  }


}
