/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.module.apikit.uri;

import java.util.Map;
import java.util.regex.Pattern;

/**
 * A URI token wrapping a variable.
 * <p/>
 * <p>
 * Variables follow the following expression:
 * 
 * <pre>
 * var         = varname [ &quot;=&quot; vardefault ]
 * varname     = (ALPHA / DIGIT)*(ALPHA / DIGIT / &quot;.&quot; / &quot;_&quot; / &quot;-&quot; )
 * vardefault  = *(unreserved / pct-encoded)
 * </pre>
 *
 * @author Christophe Lauret
 * @version 30 December 2008
 */
public class TokenVariable extends TokenBase implements Token, Matchable {

  /**
   * The variable for this token.
   */
  private Variable _var;

  /**
   * Creates a new variable token.
   *
   * @param var The variable this token corresponds to.
   * @throws NullPointerException If the specified text is <code>null</code>.
   */
  public TokenVariable(Variable var) throws NullPointerException {
    super('{' + var.toString() + "}");
    this._var = var;
  }

  /**
   * Returns the variable wrapped by this token.
   *
   * @return The variable wrapped by this token.
   */
  public Variable getVariable() {
    return _var;
  }

  /**
   * {@inheritDoc}
   */
  public boolean match(String value) {
    return Variable.isValidValue(value);
  }

  /**
   * {@inheritDoc}
   */
  public Pattern pattern() {
    return Variable.VALID_VALUE;
  }

  /**
   * {@inheritDoc}
   */
  public boolean resolve(String expanded, Map<Variable, Object> values) {
    if (expanded.isEmpty()) {
      return false;
    }
    values.put(this._var, URICoder.decode(expanded));
    return true;
  }

}
