/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security.configuration;

import static org.apache.wss4j.common.ConfigurationConstants.SIGNATURE_PARTS;
import static org.apache.wss4j.common.ConfigurationConstants.SIG_ALGO;
import static org.apache.wss4j.common.ConfigurationConstants.SIG_C14N_ALGO;
import static org.apache.wss4j.common.ConfigurationConstants.SIG_DIGEST_ALGO;
import static org.apache.wss4j.common.ConfigurationConstants.SIG_KEY_ID;

import java.util.List;
import java.util.Properties;

/**
 * Security configuration regarding signing algorithm.
 *
 * @since 1.1
 */
public class WssSignConfiguration {

  private String signatureKeyIdentifier;
  private String signatureAlgorithm;
  private String signatureDigestAlgorithm;
  private String signatureC14nAlgorithm;
  private List<WssPart> wssParts;
  private Boolean expandXopInclude; // nullable for backward compatibility

  public WssSignConfiguration(String signatureKeyIdentifier,
                              String signatureAlgorithm,
                              String signatureDigestAlgorithm,
                              String signatureC14nAlgorithm,
                              List<WssPart> wssParts) {
    this(signatureKeyIdentifier, signatureAlgorithm, signatureDigestAlgorithm, signatureC14nAlgorithm, wssParts, null);
  }

  public WssSignConfiguration(String signatureKeyIdentifier,
                              String signatureAlgorithm,
                              String signatureDigestAlgorithm,
                              String signatureC14nAlgorithm,
                              List<WssPart> wssParts,
                              Boolean expandXopInclude) {
    this.signatureKeyIdentifier = signatureKeyIdentifier;
    this.signatureAlgorithm = signatureAlgorithm;
    this.signatureDigestAlgorithm = signatureDigestAlgorithm;
    this.signatureC14nAlgorithm = signatureC14nAlgorithm;
    this.wssParts = wssParts;
    this.expandXopInclude = expandXopInclude;
  }

  public String getSignatureKeyIdentifier() {
    return signatureKeyIdentifier;
  }

  public String getSignatureAlgorithm() {
    return signatureAlgorithm;
  }

  public String getSignatureDigestAlgorithm() {
    return signatureDigestAlgorithm;
  }

  public String getSignatureC14nAlgorithm() {
    return signatureC14nAlgorithm;
  }

  public List<WssPart> getWssParts() {
    return wssParts;
  }

  public Boolean getExpandXopInclude() {
    return expandXopInclude;
  }

  public Properties getConfigurationProperties() {
    Properties properties = new Properties();

    if (signatureKeyIdentifier != null) {
      properties.setProperty(SIG_KEY_ID, signatureKeyIdentifier);
    }
    if (signatureAlgorithm != null) {
      properties.setProperty(SIG_ALGO, signatureAlgorithm);
    }
    if (signatureDigestAlgorithm != null) {
      properties.setProperty(SIG_DIGEST_ALGO, signatureDigestAlgorithm);
    }
    if (signatureC14nAlgorithm != null) {
      properties.setProperty(SIG_C14N_ALGO, signatureC14nAlgorithm);
    }
    if (wssParts != null) {
      StringBuilder sb = new StringBuilder();
      for (WssPart wssPart : wssParts) {
        sb.append(wssPart.toString());
      }
      properties.setProperty(SIGNATURE_PARTS, sb.toString());
    }

    return properties;
  }

}
