/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.message;

import java.util.Optional;

/**
 * Represents a request that aims to be sent to a Soap Web Service.
 *
 * @since 1.0
 */
public interface SoapRequest extends SoapMessage {

  /**
   * @return The name of the operation that is requested.
   */
  String getOperation();

  /**
   * @return True if the XML initial declaration must be appended to the request's body, otherwise returns False.
   */
  boolean isUseXMLInitialDeclaration();

  /**
   * @return the {@link AddressingProperties} that describes the WS-A properties of the request.
   *
   * @since 1.6
   */
  default Optional<AddressingProperties> getAddressingProperties() {
    return Optional.empty();
  }

  /**
   * @return the {@link ReliableMessagingProperties} that describes the WS-A properties of the request.
   *
   * @since 1.6
   */
  default Optional<ReliableMessagingProperties> getReliableMessagingProperties() {
    return Optional.empty();
  }

  /**
   * @return a {@link SoapRequestBuilder} instance to create a new {@link SoapRequest}.
   */
  static SoapRequestBuilder builder() {
    return new SoapRequestBuilder();
  }

  /**
   * @param operation the operation that is going to be executed.
   * @return an empty Soap Request with no content for the requested operation.
   */
  static SoapRequest empty(String operation) {
    return builder().operation(operation).build();
  }
}
