/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core.collectors.system.info;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import lombok.NonNull;
import org.apache.commons.lang3.SystemUtils;
import org.jfrog.common.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * An enum of info groups that writes all the information to the log
 *
 * @author Noam Tenne
 */
public class CommonInfoWriter {

    private static final Logger log = LoggerFactory.getLogger(CommonInfoWriter.class);
    private boolean aol;

    public CommonInfoWriter(boolean isAol) {
        this.aol = isAol;
    }

    /**
     * Info group class
     */
    private List<Class<? extends BasePropInfoGroup>> infoGroup;
    /**
     * The format of the list to be printed
     */
    public static final String listFormat = "   %1$-70s| %2$s%n";

    /**
     * Dumps the info from all the groups in the enum to the log
     *
     * @throws IllegalAccessException
     * @throws InstantiationException
     */
    public void writeInfo() throws IllegalAccessException, InstantiationException {
        if (log.isInfoEnabled()) {
            log.info(getInfo());
        }
    }

    /**
     * Dumps the info from all the groups in the enum to the log
     *
     * @throws IllegalAccessException
     * @throws InstantiationException
     */
    public String getInfo() throws IllegalAccessException, InstantiationException {
        return getInfoString();
    }

    public String getInfoString() throws InstantiationException, IllegalAccessException {
        StringBuilder sb = new StringBuilder();
        sb.append(String.format("%n%n SYSTEM INFORMATION DUMP%n"));
        sb.append(String.format(" =======================%n"));
        for (Class<? extends BasePropInfoGroup> clazz : getGroups()) {
            BasePropInfoGroup group = clazz.newInstance();
            if (!group.isInUse()) {
                continue;
            }
            //Create group title
            sb.append(String.format("%n ")).append(group.getGroupName()).append(String.format("%n"));
            sb.append(String.format(" ========================%n"));
            //Iterate over all info objects
            for (InfoObject infoObject : group.getInfo(false)) {
                String propertyName = infoObject.getPropertyName();
                String value = infoObject.getPropertyValue();
                if (shouldMaskValue(propertyName)) {
                    value = Strings.mask(value);
                } else if (clazz.equals(JavaSysPropInfo.class) && shouldMaskValue(value)) {
                    value = Strings.maskKeyValue(value);
                }
                if (propertyName.matches(".*class\\.?path.*")) {
                    splitLine(sb, propertyName, value);
                } else {
                    sb.append(String.format(listFormat, propertyName, value));
                }
            }
        }

        //Dump the info to the log
        return sb.toString();
    }

    public Map<String, Multimap<String, String>> getInfoMap() throws InstantiationException, IllegalAccessException {
        Map<String, Multimap<String, String>> info = new HashMap<>();
        for (Class<? extends BasePropInfoGroup> clazz : getGroups()) {
            BasePropInfoGroup group = clazz.newInstance();
            if (!group.isInUse()) {
                continue;
            }
            ArrayListMultimap<String, String> currentMap = ArrayListMultimap.create();
            info.put(group.getGroupName(), currentMap);
            //Iterate over all info objects
            for (InfoObject infoObject : group.getInfo(true)) {
                String propertyName = infoObject.getPropertyName();
                String value = infoObject.getPropertyValue();
                if (shouldMaskValue(propertyName)) {
                    value = Strings.mask(value);
                } else if (clazz.equals(JavaSysPropInfo.class) && shouldMaskValue(value)) {
                    value = Strings.maskKeyValue(value);
                }
                if (propertyName.matches(".*class\\.?path.*")) {
                    splitLine(currentMap, propertyName, value);
                } else {
                    currentMap.put(propertyName, value);
                }
            }
        }
        //Dump the info to the log
        return info;
    }

    private void splitLine(StringBuilder sb, String propertyName, String value) {
        String multiValueSeparator = SystemUtils.IS_OS_WINDOWS ? ";" : ":";
        String[] separateValues = value.split(multiValueSeparator);
        for (int i = 0; i < separateValues.length; i++) {
            String separateValue = separateValues[i];
            sb.append(String.format(listFormat, (i == 0) ? propertyName : "", separateValue));
        }
    }

    private void splitLine(Multimap<String, String> map, String propertyName, String value) {
        String multiValueSeparator = SystemUtils.IS_OS_WINDOWS ? ";" : ":";
        String[] separateValues = value.split(multiValueSeparator);
        for (int i = 0; i < separateValues.length; i++) {
            String separateValue = separateValues[i];
            map.put(propertyName, separateValue);
        }
    }

    public static boolean shouldMaskValue(String propertyKey) {
        String propKeyLower = propertyKey.toLowerCase();
        return propKeyLower.contains("password")
                || propKeyLower.contains("secret")
                || propKeyLower.contains("key")
                || getMaskedKeys().contains(propertyKey);
    }

    protected static Set<String> getMaskedKeys() {
        return Sets.newHashSet();
    }

    protected @NonNull List<Class<? extends BasePropInfoGroup>> getGroups() {
        if (isAol()) {
            return List.of(ClassPathPropInfo.class, JavaSysPropInfo.class, UserPropInfo.class);
        }
        return List.of(ClassPathPropInfo.class, JavaSysPropInfo.class, UserPropInfo.class, HostPropInfo.class);
    }

    protected boolean isAol() {
        return aol;
    }
}
