/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jfrog.support.common.CollectConfiguration;
import org.jfrog.support.common.core.exceptions.TempDirAccessException;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;

/**
 * Provides content specific services where every
 * item in the context requires unique treatment
 *
 * @author Michael Pasternak
 */
public abstract class AbstractSpecificContentCollector<T extends CollectConfiguration>
        extends AbstractContentCollector<T> {

    private static final Logger log = LoggerFactory.getLogger(AbstractSpecificContentCollector.class);

    /**
     * @param contentName a name for specific sub-folder
     */
    protected AbstractSpecificContentCollector(String contentName) {
        super(contentName);
    }

    @Override
    public Logger getLog() {
        return log;
    }

    /**
     * Performs copy to destination directory
     *
     * @param file source
     * @param tmpDir target
     */
    protected void copyToTempDir(Path file, File tmpDir) {
        log.debug("Initiating copy of file '{}'", file.getFileName());
        try {
            Files.copy(
                    file,
                    Paths.get(
                            tmpDir.getPath() + File.separator + file.getFileName()
                    ),
                    StandardCopyOption.COPY_ATTRIBUTES
            );
        } catch (IOException e) {
            String message = (e instanceof FileAlreadyExistsException) ? "a file with that name already exists there" : e.getMessage();
            throw new TempDirAccessException("Cannot copy file '" + file.getFileName()
                    + "' to '" + tmpDir.getPath() + "' because: " + message, e);
        }
    }

    /**
     * Prints failure cause by exception
     *
     * @param e the cause
     */
    protected boolean failure(Exception e) {
        getLog().warn("Collecting content has failed - '" + e.getMessage() + "'");
        getLog().debug("Cause: {}", e);
        getLog().info("Collection of " + getContentName() +
                " was not properly accomplished, see logs for more details");
        return false;
    }
}
