/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.support.common.core.collectors;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.jfrog.support.common.LogsFilesMatcher;
import org.jfrog.support.common.config.SystemLogsConfiguration;
import org.jfrog.support.common.core.AbstractSpecificContentCollector;
import org.jfrog.support.common.core.exceptions.BundleConfigurationException;
import org.jfrog.support.common.core.exceptions.TempDirAccessException;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.FileTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.Date;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * System logs collector
 *
 * @author Michael Pasternak
 */
public abstract class LogsCollector extends AbstractSpecificContentCollector<SystemLogsConfiguration> {
    private static final Logger log = LoggerFactory.getLogger(LogsCollector.class);

    private LogsFilesMatcher filesMatcher;

    public LogsCollector() {
        super("logs");
        filesMatcher = new DefaultFilesMatcher();
    }

    //TODO [by tamir]: ctor that gets filematcher and log dir

    protected void setFilesMatcher(LogsFilesMatcher matcher) {
        this.filesMatcher = matcher;
    }

    /**
     * Collects SystemLogs
     *
     * @param configuration {@link SystemLogsConfiguration}
     * @param tmpDir        output directory for produced content
     * @return operation result
     */
    @Override
    protected boolean doCollect(SystemLogsConfiguration configuration, File tmpDir) {
        try {
            collectLogs(configuration, tmpDir);
            getLog().info("Collection of {} was successfully accomplished", getContentName());
            return true;
        } catch (IOException | TempDirAccessException e) {
            return failure(e);
        }
    }

    private void collectLogs(SystemLogsConfiguration configuration, File tmpDir) throws IOException {
        try (Stream<Path> walk = Files.walk(getLogsDirectory().toPath(), FileVisitOption.FOLLOW_LINKS)) {
            walk.filter(Files::isRegularFile)
                    .filter(f -> isCollectible(f, configuration))
                    .forEach(f -> {
                        String subDir = f.getParent().toString().replace(getLogsDirectory().toPath().toString(), "");
                        File targetLocation = new File(tmpDir.getPath() + subDir);
                        try {
                            FileUtils.forceMkdir(targetLocation);
                        } catch (IOException e) {
                            getLog().error("Failed to create a directory", e);
                        }
                        copyToTempDir(f, targetLocation);
                    });
        }
    }

    /**
     * Checks whether given file is eligible for collections
     *
     * @param configuration {@link SystemLogsConfiguration}
     * @return boolean
     */
    private boolean isCollectible(Path filePath, SystemLogsConfiguration configuration) {
        getLog().debug("Initiating collect eligibility check for file '{}'", filePath.getFileName());
        return this.filesMatcher.isMatch(filePath, configuration);
    }

    @Override
    public Logger getLog() {
        return log;
    }

    /**
     * Makes sure configuration is valid
     *
     * @param configuration configuration to check
     * @throws BundleConfigurationException if configuration is invalid
     */
    @Override
    protected void doEnsureConfiguration(SystemLogsConfiguration configuration)
            throws BundleConfigurationException {
        if (configuration.getDaysCount() != null) {
            ensureDateRange(
                    configuration.getDaysCount()
            );
        } else if (configuration.getStartDate() != null &&
                configuration.getEndDate() != null) {
            ensureDateRange(
                    configuration.getStartDate(),
                    configuration.getEndDate()
            );
        } else {
            throw new BundleConfigurationException(
                    "SystemLogsConfiguration is incomplete, either DaysCount or StartDate+EndDate must be specified"
            );
        }
    }

    /**
     * Makes sure date range is legal
     *
     * @throws BundleConfigurationException thrown when illegal configuration is found
     */
    private void ensureDateRange(Date startDate, Date endDate)
            throws BundleConfigurationException {
        if (startDate == null || endDate == null) {
            throw new BundleConfigurationException(
                    "Date range is illegal, " +
                            "startDate/endDate cannot be empty"
            );
        }

        if (startDate.getTime() > endDate.getTime()) {
            throw new BundleConfigurationException(
                    "Date range is illegal, " +
                            "startDate cannot be greater than endDate"
            );
        }
    }

    /**
     * Makes sure date range is legal
     *
     * @throws BundleConfigurationException thrown when illegal configuration is found
     */
    private void ensureDateRange(Integer daysCount)
            throws BundleConfigurationException {
        if (daysCount == null) {
            throw new BundleConfigurationException("Date range is illegal, daysCount cannot be empty");
        }

        if (daysCount <= 0) {
            throw new BundleConfigurationException(
                    "Date range is illegal, daysCount cannot be negative number or zero");
        }
    }

    /**
     * @return logs folder location
     */
    protected abstract File getLogsDirectory();


    /**
     * By default collects all .log and .zip files in the date range specified  at the {@link SystemLogsConfiguration}
     */
    private class DefaultFilesMatcher implements LogsFilesMatcher {
        private final Logger log = LoggerFactory.getLogger(DefaultFilesMatcher.class);

        @Override
        public boolean isMatch(Path filePath, SystemLogsConfiguration configuration) {
            String logsFilesPattern = "glob:**.log";
            String zippedLogsPattern = "glob:**.zip";
            log.debug("Matching file {}", filePath);
            PathMatcher pathMatcher = FileSystems.getDefault().getPathMatcher(logsFilesPattern);
            PathMatcher zippedPathMatcher = FileSystems.getDefault().getPathMatcher(zippedLogsPattern);
            if (pathMatcher.matches(filePath) || zippedPathMatcher.matches(filePath)) {
                long fileModifiedTime = getFileModifiedTimeAsEpoch(filePath);
                long minTime = getMinTimeAsEpoch(configuration);
                long maxTime = getMaxTimeAsEpoch(configuration);
                return isDateBetween(fileModifiedTime, minTime, maxTime);
            }

            log.debug("{} didn't matched any pattern by default collect the file as well.", filePath);
            return true;
        }

        private boolean isDateBetween(long fileModifiedTime, long minTime, long maxTime) {
            return fileModifiedTime >= minTime && fileModifiedTime <= maxTime;
        }

        private long getMaxTimeAsEpoch(SystemLogsConfiguration logsSettings) {
            return logsSettings.getEndDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate().atTime(23, 59, 59)
                    .toEpochSecond(ZoneOffset.UTC) * 1000;
        }

        private long getMinTimeAsEpoch(SystemLogsConfiguration logsSettings) {
            return logsSettings.getStartDate().toInstant().atZone(ZoneId.systemDefault()).toLocalDate().atTime(0, 0, 0).toEpochSecond(ZoneOffset.UTC) * 1000;
        }

        private Long getFileModifiedTimeAsEpoch(Path path) {
            try {
                FileTime lastModifiedTime = Files.getLastModifiedTime(path);
                log.debug("File's last modified time: {}", lastModifiedTime);
                return Optional.ofNullable(lastModifiedTime)
                        .map(FileTime::toMillis)
                        .orElse(0L);
            } catch (IOException e) {
                log.warn("Cannot get file's last modified time: {}", e.getMessage());
                log.debug("Exception:", e);
                return 0L;
            }
        }
    }

}
