/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.access.key.master;

import com.jfrog.sysconf.SysConfig;
import java.io.File;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nonnull;
import org.jfrog.access.key.master.MasterKeyStorage;
import org.jfrog.access.key.master.exception.MasterKeyStorageException;
import org.jfrog.access.key.master.status.MasterKeyStatus;
import org.jfrog.security.common.KeyUtils;
import org.jfrog.security.crypto.EncryptionWrapper;
import org.jfrog.security.crypto.EncryptionWrapperFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class MasterKeyBootstrapper {
    protected static final Logger log = LoggerFactory.getLogger(MasterKeyBootstrapper.class);
    protected static final long DEFAULT_WAIT_FOR_KEY_TIMEOUT = 60000L;
    private final SysConfig sysConf;
    protected MasterKeyStorage masterKeyService;

    public MasterKeyBootstrapper(SysConfig sysConf) {
        this.sysConf = Objects.requireNonNull(sysConf, "System configuration cannot be null");
    }

    protected void handleMasterKey() {
        log.debug("Searching for Master key under home directory.");
        EncryptionWrapper masterKeyWrapper = this.getOrGenerateMasterKeyWrapper();
        log.debug("Master key found.");
        MasterKeyStatus keyStatus = this.getKeyDetails(masterKeyWrapper);
        if (this.validateOrInsertKeyIfNeeded(keyStatus)) {
            log.debug("Master key validation/insertion succeeded.");
            this.saveAndSecureMasterKeyIfWasGenerated(masterKeyWrapper);
            this.setMasterKeyEncryptionWrapper(masterKeyWrapper);
        }
    }

    EncryptionWrapper getOrGenerateMasterKeyWrapper() {
        EncryptionWrapper masterEncryptionWrapper = null;
        try {
            Optional<String> masterKeyValue = this.getMasterKeyValue();
            if (masterKeyValue.isPresent()) {
                masterEncryptionWrapper = EncryptionWrapperFactory.aesKeyWrapperFromString((String)masterKeyValue.get());
            }
            if (masterEncryptionWrapper == null) {
                log.info("No master.key supplied. Generating master.key");
                masterEncryptionWrapper = this.generateNewMasterKey();
            }
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
        return masterEncryptionWrapper;
    }

    private MasterKeyStatus getKeyDetails(EncryptionWrapper masterKeyWrapper) {
        String serviceName = this.getServiceName();
        return new MasterKeyStatus(masterKeyWrapper.getFingerprint(), MasterKeyStatus.MasterKeyStatusEnum.on, serviceName, 0L);
    }

    private boolean validateOrInsertKeyIfNeeded(@Nonnull MasterKeyStatus fsKeyDetails) {
        MasterKeyStatus dbKeyInfo = this.masterKeyService.getMasterKeyInfo();
        boolean shouldVerifyAgainstDb = false;
        if (dbKeyInfo == null) {
            try {
                log.debug("Attempting to insert key fingerprint into the DB.");
                this.masterKeyService.insertMasterKey(fsKeyDetails);
            }
            catch (MasterKeyStorageException e) {
                log.debug("Could not insert key fingerprint into the DB.", (Throwable)e);
                shouldVerifyAgainstDb = true;
            }
        } else {
            log.debug("DB already has master key associated.");
            shouldVerifyAgainstDb = true;
        }
        return !shouldVerifyAgainstDb || this.verifyKeyAgainstDb(fsKeyDetails);
    }

    private boolean verifyKeyAgainstDb(MasterKeyStatus fsKeyDetails) {
        log.trace("Validating master key against the DB.");
        boolean sameKey = this.masterKeyService.isKeyExists(fsKeyDetails.getKid());
        if (!sameKey) {
            File masterKeyFile = this.getMasterKeyFile();
            if (masterKeyFile.exists() || this.masterKeyProvidedByParam()) {
                throw new IllegalStateException("Master key mismatch. The provided master.key file does't match the DB fingerprint. Make sure your db.properties configurations are valid and the master key matches the DB you are trying to connect to.");
            }
            log.error("Found existing master key fingerprint in the DB, without master.key file. Please provide a master key file manually in '{}'.", (Object)this.getMasterKeyFile().getAbsolutePath());
            log.info("Waiting for 1 minute until the key is supplied manually...");
            KeyUtils.waitForKey((String)"master", (long)this.getWaitForKeyTimeoutValue(), (File)this.getMasterKeyFile());
            this.handleMasterKey();
            return false;
        }
        return true;
    }

    private void saveAndSecureMasterKeyIfWasGenerated(EncryptionWrapper masterKeyWrapper) {
        File targetKeyFile = this.getMasterKeyFile();
        boolean keyProvidedByParam = this.masterKeyProvidedByParam();
        if (!targetKeyFile.exists() && !keyProvidedByParam) {
            log.info("Saving master.key to {}", (Object)targetKeyFile);
            KeyUtils.saveKeyToFile((File)targetKeyFile, (EncryptionWrapper)masterKeyWrapper);
        }
    }

    private boolean masterKeyProvidedByParam() {
        return this.getMasterKeyValue().isPresent();
    }

    Optional<String> getMasterKeyValue() {
        return this.sysConf.helper().getMasterKey();
    }

    private File getMasterKeyFile() {
        return new File(this.sysConf.helper().getMasterKeyFile());
    }

    protected abstract long getWaitForKeyTimeoutValue();

    protected abstract void setMasterKeyEncryptionWrapper(EncryptionWrapper var1);

    protected abstract EncryptionWrapper generateNewMasterKey();

    protected abstract String getServiceName();
}

