/*
 * Decompiled with CFR 0.152.
 */
package hudson.plugins.swarm;

import hudson.plugins.swarm.ConfigurationException;
import hudson.plugins.swarm.LabelFileWatcher;
import hudson.plugins.swarm.Options;
import hudson.plugins.swarm.RetryException;
import hudson.plugins.swarm.SwarmClient;
import hudson.plugins.swarm.YamlConfig;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.NamedOptionDef;
import org.kohsuke.args4j.spi.FieldSetter;
import org.kohsuke.args4j.spi.OptionHandler;

public class Client {
    private static final Logger logger = Logger.getLogger(Client.class.getName());
    private static final String NON_FATAL_JNLP_AGENT_ENDPOINT_RESOLUTION_EXCEPTIONS = "hudson.remoting.Engine.nonFatalJnlpAgentEndpointResolutionExceptions";

    public static void main(String ... args) throws InterruptedException {
        Options options = new Options();
        CmdLineParser parser = new CmdLineParser(options);
        try {
            parser.parseArgument(args);
        }
        catch (CmdLineException e) {
            Client.fail(e.getMessage());
        }
        Client.logArguments(parser);
        if (options.help) {
            parser.printUsage(System.out);
            System.exit(0);
        }
        if (options.config != null) {
            if (Client.hasConflictingOptions(parser)) {
                Client.fail("'-config' can not be used with other options.");
            }
            logger.log(Level.INFO, "Load configuration from {0}", options.config.getPath());
            try (InputStream is = Files.newInputStream(options.config.toPath(), new OpenOption[0]);){
                options = new YamlConfig().loadOptions(is);
            }
            catch (ConfigurationException | IOException | InvalidPathException e) {
                Client.fail(e.getMessage());
            }
        }
        try {
            Client.validateOptions(options);
        }
        catch (RuntimeException e) {
            Client.fail(e.getMessage());
        }
        Client.run(new SwarmClient(options), options, args);
    }

    private static boolean hasConflictingOptions(CmdLineParser parser) {
        return parser.getOptions().stream().anyMatch(oh -> !Client.getKey(oh).equals("-config") && !Client.isDefaultOption(Client.getKey(oh), Client.getValue(oh), new CmdLineParser(new Options())));
    }

    private static void validateOptions(Options options) {
        if (options.url == null) {
            throw new IllegalArgumentException("Missing 'url' option.");
        }
        if (options.pidFile != null) {
            ProcessHandle current = ProcessHandle.current();
            Path pidFile = Paths.get(options.pidFile, new String[0]);
            if (Files.exists(pidFile, new LinkOption[0])) {
                long oldPid;
                try {
                    oldPid = Long.parseLong(Files.readString(pidFile, StandardCharsets.US_ASCII));
                }
                catch (NumberFormatException e) {
                    oldPid = 0L;
                }
                catch (IOException e) {
                    throw new UncheckedIOException("Failed to read PID file " + String.valueOf(pidFile), e);
                }
                if (oldPid > 0L) {
                    Optional<ProcessHandle> oldProcess = ProcessHandle.of(oldPid);
                    if (oldProcess.isPresent() && oldProcess.get().isAlive()) {
                        String curCommand = current.info().command().orElse(null);
                        String oldCommand = oldProcess.get().info().command().orElse(null);
                        if (curCommand != null && curCommand.equals(oldCommand)) {
                            throw new IllegalStateException(String.format("Refusing to start because PID file '%s' already exists and the previous process %d (%s) is still running.", pidFile.toAbsolutePath(), oldPid, oldProcess.get().info().commandLine().orElse("unknown")));
                        }
                        logger.warning(String.format("Ignoring stale PID file '%s' because the process %d (%s) is not a Swarm Client.", pidFile.toAbsolutePath(), oldPid, oldProcess.get().info().commandLine().orElse("unknown")));
                    } else {
                        logger.fine(String.format("Ignoring PID file '%s' because the previous process %d is no longer running.", pidFile.toAbsolutePath(), oldPid));
                    }
                }
            }
            pidFile.toFile().deleteOnExit();
            try {
                Files.writeString(pidFile, (CharSequence)Long.toString(current.pid()), StandardCharsets.US_ASCII, new OpenOption[0]);
            }
            catch (IOException e) {
                throw new UncheckedIOException("Failed to write PID file " + options.pidFile, e);
            }
        }
        if (options.passwordEnvVariable != null) {
            options.password = System.getenv(options.passwordEnvVariable);
        }
        if (options.password == null && options.passwordFile != null) {
            try {
                options.password = Files.readString(Paths.get(options.passwordFile, new String[0]), StandardCharsets.UTF_8).trim();
            }
            catch (IOException e) {
                throw new UncheckedIOException("Failed to read password from file", e);
            }
        }
        if (options.name == null) {
            try {
                options.name = InetAddress.getLocalHost().getCanonicalHostName();
            }
            catch (UnknownHostException e) {
                logger.severe("Failed to look up the canonical hostname of this agent. Check the system DNS settings.");
                logger.severe("If it is not possible to resolve this host, specify a name using the \"-name\" option.");
                throw new UncheckedIOException("Failed to set hostname", e);
            }
        }
    }

    static void run(SwarmClient swarmClient, Options options, String ... args) throws InterruptedException {
        logger.info("Connecting to Jenkins controller");
        URL url = swarmClient.getUrl();
        int retry = 0;
        while (true) {
            try {
                logger.info("Attempting to connect to " + String.valueOf(url));
                swarmClient.createSwarmAgent(url);
                if (options.labelsFile != null) {
                    logger.info("Setting up LabelFileWatcher");
                    LabelFileWatcher l = new LabelFileWatcher(url, options, swarmClient.getName(), args);
                    Thread labelFileWatcherThread = new Thread((Runnable)l, "LabelFileWatcher");
                    labelFileWatcherThread.setDaemon(true);
                    labelFileWatcherThread.start();
                }
                if (System.getProperty(NON_FATAL_JNLP_AGENT_ENDPOINT_RESOLUTION_EXCEPTIONS) == null) {
                    System.setProperty(NON_FATAL_JNLP_AGENT_ENDPOINT_RESOLUTION_EXCEPTIONS, "true");
                }
                swarmClient.connect(url);
                if (options.noRetryAfterConnected) {
                    logger.warning("Connection closed, exiting...");
                    swarmClient.exitWithStatus(0);
                }
            }
            catch (RetryException | IOException | InterruptedException e) {
                logger.log(Level.SEVERE, "An error occurred", e);
            }
            int waitTime = options.retryBackOffStrategy.waitForRetry(retry++, options.retryInterval, options.maxRetryInterval);
            if (options.retry >= 0) {
                if (retry >= options.retry) {
                    logger.severe("Retry limit reached, exiting...");
                    swarmClient.exitWithStatus(1);
                } else {
                    logger.warning("Remaining retries: " + (options.retry - retry));
                }
            }
            logger.info("Retrying in " + waitTime + " seconds");
            swarmClient.sleepSeconds(waitTime);
        }
    }

    private static void logArguments(CmdLineParser parser) {
        Options defaultOptions = new Options();
        CmdLineParser defaultParser = new CmdLineParser(defaultOptions);
        StringBuilder sb = new StringBuilder("Client invoked with: ");
        for (OptionHandler argument : parser.getArguments()) {
            Client.logValue(sb, argument, null);
        }
        for (OptionHandler option : parser.getOptions()) {
            Client.logValue(sb, option, defaultParser);
        }
        logger.info(sb.toString());
    }

    private static void logValue(StringBuilder sb, OptionHandler<?> handler, CmdLineParser defaultParser) {
        String key = Client.getKey(handler);
        Object value = Client.getValue(handler);
        if (handler.option.help()) {
            return;
        }
        if (defaultParser != null && Client.isDefaultOption(key, value, defaultParser)) {
            return;
        }
        sb.append(key);
        sb.append(' ');
        if (key.equals("-username") || key.startsWith("-password")) {
            sb.append("*****");
        } else {
            sb.append(value);
        }
        sb.append(' ');
    }

    private static String getKey(OptionHandler<?> optionHandler) {
        if (optionHandler.option instanceof NamedOptionDef) {
            NamedOptionDef namedOptionDef = (NamedOptionDef)optionHandler.option;
            return namedOptionDef.name();
        }
        return optionHandler.option.toString();
    }

    private static Object getValue(OptionHandler<?> optionHandler) {
        FieldSetter setter = optionHandler.setter.asFieldSetter();
        return setter == null ? null : setter.getValue();
    }

    private static boolean isDefaultOption(String key, Object value, CmdLineParser defaultParser) {
        for (OptionHandler defaultOption : defaultParser.getOptions()) {
            String defaultKey = Client.getKey(defaultOption);
            if (!defaultKey.equals(key)) continue;
            Object defaultValue = Client.getValue(defaultOption);
            if (defaultValue == null && value == null) {
                return true;
            }
            return defaultValue != null && defaultValue.equals(value);
        }
        return false;
    }

    private static void fail(String message) {
        System.err.println(message);
        System.exit(1);
    }
}

