/*
 * Decompiled with CFR 0.152.
 */
package jenkins.plugins.git;

import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.EnvVars;
import hudson.ExtensionList;
import hudson.ExtensionPoint;
import hudson.model.Item;
import hudson.model.Node;
import hudson.model.TaskListener;
import hudson.plugins.git.GitTool;
import hudson.plugins.git.util.GitUtils;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import jenkins.model.Jenkins;
import jenkins.plugins.git.AbstractGitSCMSource;
import org.apache.commons.io.FileUtils;
import org.jenkinsci.plugins.gitclient.Git;
import org.jenkinsci.plugins.gitclient.GitClient;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

public class GitToolChooser {
    private long sizeOfRepo = 0L;
    private String implementation;
    private String gitTool;
    private TaskListener listener;
    private Node currentNode;
    private static final int SIZE_TO_SWITCH = 5000;
    private boolean JGIT_SUPPORTED = false;
    private static ConcurrentHashMap<String, Long> repositorySizeCache = new ConcurrentHashMap();
    private static Pattern gitProtocolPattern = Pattern.compile("^git://([^/]+)/(.+?)/*$");
    private static Pattern httpProtocolPattern = Pattern.compile("^https?://([^/]+)/(.+?)/*$");
    private static Pattern sshAltProtocolPattern = Pattern.compile("^[\\w]+@(.+):(.+?)/*$");
    private static Pattern sshProtocolPattern = Pattern.compile("^ssh://[\\w]+@([^/]+)/(.+?)/*$");
    private static final Logger LOGGER = Logger.getLogger(GitToolChooser.class.getName());

    public GitToolChooser(String remoteName, Item projectContext, String credentialsId, GitTool gitExe, Node n, TaskListener listener, Boolean useJGit) throws IOException, InterruptedException {
        boolean useCache = false;
        if (useJGit != null) {
            this.JGIT_SUPPORTED = useJGit;
        }
        this.currentNode = n;
        this.listener = listener;
        this.implementation = "NONE";
        useCache = this.decideAndUseCache(remoteName);
        if (useCache) {
            this.implementation = this.determineSwitchOnSize(this.sizeOfRepo, gitExe);
        } else {
            this.decideAndUseAPI(remoteName, projectContext, credentialsId, gitExe);
        }
        this.gitTool = this.implementation;
    }

    private boolean decideAndUseCache(String remoteName) throws IOException, InterruptedException {
        boolean useCache = false;
        if (this.setSizeFromInternalCache(remoteName)) {
            LOGGER.log(Level.FINE, "Found cache key for {0} with size {1}", new Object[]{remoteName, this.sizeOfRepo});
            useCache = true;
            return useCache;
        }
        for (String repoUrl : this.remoteAlternatives(remoteName)) {
            String cacheEntry = AbstractGitSCMSource.getCacheEntry(repoUrl);
            File cacheDir = AbstractGitSCMSource.getCacheDir(cacheEntry, false);
            if (cacheDir == null) continue;
            Git git = Git.with((TaskListener)TaskListener.NULL, (EnvVars)new EnvVars(EnvVars.masterEnvVars)).in(cacheDir).using("git");
            GitClient client = git.getClient();
            if (client.hasGitRepo()) {
                long clientRepoSize = FileUtils.sizeOfDirectory((File)cacheDir) / 1024L;
                if (clientRepoSize > this.sizeOfRepo) {
                    if (this.sizeOfRepo > 0L) {
                        LOGGER.log(Level.FINE, "Replacing prior size estimate {0} with new size estimate {1} for remote {2} from cache {3}", new Object[]{this.sizeOfRepo, clientRepoSize, remoteName, cacheDir});
                    }
                    this.sizeOfRepo = clientRepoSize;
                    this.assignSizeToInternalCache(remoteName, this.sizeOfRepo);
                }
                useCache = true;
                if (remoteName.equals(repoUrl)) {
                    LOGGER.log(Level.FINE, "Remote URL {0} found cache {1} with size {2}", new Object[]{remoteName, cacheDir, this.sizeOfRepo});
                    continue;
                }
                LOGGER.log(Level.FINE, "Remote URL {0} found cache {1} with size {2}, alternative URL {3}", new Object[]{remoteName, cacheDir, this.sizeOfRepo, repoUrl});
                continue;
            }
            LOGGER.log(Level.FINE, "Remote URL {0} cache {1} has no git dir", new Object[]{remoteName, cacheDir});
        }
        if (!useCache) {
            LOGGER.log(Level.FINE, "Remote URL {0} cache not found", remoteName);
        }
        return useCache;
    }

    private void decideAndUseAPI(String remoteName, Item context, String credentialsId, GitTool gitExe) {
        if (this.setSizeFromAPI(remoteName, context, credentialsId)) {
            this.implementation = this.determineSwitchOnSize(this.sizeOfRepo, gitExe);
        }
    }

    private void addSuffixVariants(@NonNull String remoteURL, @NonNull Set<String> alternatives) {
        alternatives.add(remoteURL);
        String suffix = ".git";
        if (remoteURL.endsWith(suffix)) {
            alternatives.add(remoteURL.substring(0, remoteURL.length() - suffix.length()));
        } else {
            alternatives.add(remoteURL + suffix);
        }
    }

    private String addSuffix(@NonNull String canonicalURL) {
        String suffix = ".git";
        if (!canonicalURL.endsWith(suffix)) {
            canonicalURL = canonicalURL + suffix;
        }
        return canonicalURL;
    }

    @NonNull
    String convertToCanonicalURL(String remoteURL) {
        if (remoteURL == null || remoteURL.isEmpty()) {
            LOGGER.log(Level.FINE, "Null or empty remote URL not cached");
            return "";
        }
        Pattern[] protocolPatterns = new Pattern[]{sshAltProtocolPattern, sshProtocolPattern, gitProtocolPattern};
        String matcherReplacement = "https://$1/$2";
        String canonicalURL = remoteURL = this.addSuffix(remoteURL);
        if (httpProtocolPattern.matcher(remoteURL).matches()) {
            canonicalURL = remoteURL;
        } else {
            for (Pattern protocolPattern : protocolPatterns) {
                Matcher protocolMatcher = protocolPattern.matcher(remoteURL);
                if (!protocolMatcher.matches()) continue;
                canonicalURL = protocolMatcher.replaceAll(matcherReplacement);
                break;
            }
        }
        LOGGER.log(Level.FINE, "Cache repo URL: {0}", canonicalURL);
        return canonicalURL;
    }

    private boolean setSizeFromInternalCache(String repoURL) {
        if (repositorySizeCache.containsKey(repoURL = this.convertToCanonicalURL(repoURL))) {
            this.sizeOfRepo = repositorySizeCache.get(repoURL);
            return true;
        }
        return false;
    }

    @NonNull
    Set<String> remoteAlternatives(String remoteURL) {
        LinkedHashSet<String> alternatives = new LinkedHashSet<String>();
        if (remoteURL == null || remoteURL.isEmpty()) {
            LOGGER.log(Level.FINE, "Null or empty remote URL not cached");
            return alternatives;
        }
        Pattern[] protocolPatterns = new Pattern[]{gitProtocolPattern, httpProtocolPattern, sshAltProtocolPattern, sshProtocolPattern};
        String[] matcherReplacements = new String[]{"git://$1/$2", "git@$1:$2", "https://$1/$2", "ssh://git@$1/$2"};
        boolean matched = false;
        for (Pattern protocolPattern : protocolPatterns) {
            Matcher protocolMatcher = protocolPattern.matcher(remoteURL);
            if (!protocolMatcher.matches()) continue;
            for (String replacement : matcherReplacements) {
                String alternativeURL = protocolMatcher.replaceAll(replacement);
                this.addSuffixVariants(alternativeURL, alternatives);
            }
            matched = true;
        }
        if (!matched) {
            this.addSuffixVariants(remoteURL, alternatives);
        }
        LOGGER.log(Level.FINE, "Cache repo alternative URLs: {0}", alternatives);
        return alternatives;
    }

    private void assignSizeToInternalCache(String repoURL, long repoSize) {
        if (repositorySizeCache.containsKey(repoURL = this.convertToCanonicalURL(repoURL))) {
            long oldSize = repositorySizeCache.get(repoURL);
            if (oldSize < repoSize) {
                LOGGER.log(Level.FINE, "Replacing old repo size {0} with new size {1} for repo {2}", new Object[]{oldSize, repoSize, repoURL});
                repositorySizeCache.put(repoURL, repoSize);
            } else if (oldSize > repoSize) {
                LOGGER.log(Level.FINE, "Ignoring new size {1} in favor of old size {0} for repo {2}", new Object[]{oldSize, repoSize, repoURL});
            }
        } else {
            LOGGER.log(Level.FINE, "Caching repo size {0} for repo {1}", new Object[]{repoSize, repoURL});
            repositorySizeCache.put(repoURL, repoSize);
        }
    }

    private boolean setSizeFromAPI(String repoUrl, Item context, String credentialsId) {
        List acceptedRepository = Objects.requireNonNull(RepositorySizeAPI.all()).stream().filter(r -> r.isApplicableTo(repoUrl, context, credentialsId)).collect(Collectors.toList());
        if (acceptedRepository.size() > 0) {
            try {
                for (RepositorySizeAPI repo : acceptedRepository) {
                    long size = repo.getSizeOfRepository(repoUrl, context, credentialsId);
                    if (size == 0L) continue;
                    this.sizeOfRepo = size;
                    this.assignSizeToInternalCache(repoUrl, size);
                }
            }
            catch (Exception e) {
                LOGGER.log(Level.INFO, "Not using performance improvement from REST API: {0}", e.getMessage());
                return false;
            }
            return this.sizeOfRepo != 0L;
        }
        return false;
    }

    String determineSwitchOnSize(Long sizeOfRepo, GitTool tool) {
        if (sizeOfRepo != 0L) {
            if (sizeOfRepo < 5000L) {
                if (!this.JGIT_SUPPORTED) {
                    return "NONE";
                }
                GitTool rTool = this.resolveGitToolForRecommendation(tool, "jgit");
                if (rTool == null) {
                    return "NONE";
                }
                return rTool.getGitExe();
            }
            GitTool rTool = this.resolveGitToolForRecommendation(tool, "git");
            return rTool.getGitExe();
        }
        return "NONE";
    }

    private GitTool resolveGitToolForRecommendation(GitTool userChoice, String recommendation) {
        if (recommendation.equals("jgit")) {
            GitTool tool;
            if (userChoice.getGitExe().equals("jgitapache")) {
                recommendation = "jgitapache";
            }
            if ((tool = this.getResolvedGitTool(recommendation)).getName().equals(recommendation)) {
                return tool;
            }
            return null;
        }
        if (!userChoice.getName().equals("jgit") && !userChoice.getName().equals("jgitapache")) {
            return userChoice;
        }
        return this.recommendGitToolOnAgent(userChoice);
    }

    public GitTool recommendGitToolOnAgent(GitTool userChoice) {
        ArrayList<GitTool> preferredToolList = new ArrayList<GitTool>();
        GitTool correctTool = GitTool.getDefaultInstallation();
        String toolName = userChoice.getName();
        if (toolName.equals("jgit") || toolName.equals("jgitapache")) {
            GitTool[] toolList = (GitTool[])((GitTool.DescriptorImpl)Jenkins.get().getDescriptorByType(GitTool.DescriptorImpl.class)).getInstallations();
            for (GitTool tool : toolList) {
                if (tool.getProperties().isEmpty()) continue;
                preferredToolList.add(tool);
            }
            for (GitTool tool : preferredToolList) {
                if (!tool.getName().equals(this.getResolvedGitTool(tool.getName()).getName())) continue;
                correctTool = this.getResolvedGitTool(tool.getName());
            }
        }
        return correctTool;
    }

    private GitTool getResolvedGitTool(String recommendation) {
        if (this.currentNode == null) {
            this.currentNode = Jenkins.get();
        }
        return GitUtils.resolveGitTool(recommendation, this.currentNode, null, this.listener);
    }

    public String getGitTool() {
        return this.gitTool;
    }

    public static void clearRepositorySizeCache() {
        repositorySizeCache = new ConcurrentHashMap();
    }

    @Restricted(value={NoExternalUse.class})
    public static void putRepositorySizeCache(String repoURL, long repoSize) {
        if (!repoURL.endsWith(".git")) {
            repoURL = repoURL + ".git";
        }
        repositorySizeCache.put(repoURL, repoSize);
    }

    public static abstract class RepositorySizeAPI
    implements ExtensionPoint {
        public abstract boolean isApplicableTo(String var1, Item var2, String var3);

        public abstract Long getSizeOfRepository(String var1, Item var2, String var3) throws Exception;

        public static ExtensionList<RepositorySizeAPI> all() {
            return Jenkins.get().getExtensionList(RepositorySizeAPI.class);
        }
    }
}

