package org.jenkinsci.plugins.displayurlapi;

import edu.umd.cs.findbugs.annotations.CheckForNull;
import edu.umd.cs.findbugs.annotations.NonNull;
import hudson.ExtensionList;
import hudson.ExtensionPoint;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;

/**
 * An extension point for decorating the URLs presented to users by the {@link DisplayURLProvider}. Use cases include
 * decorating URLs with {@code utm_source} etc tracking parameters in order to measure the usage rate of different URLs
 * generated by Jenkins.
 */
public abstract class DisplayURLDecorator implements ExtensionPoint {

    /**
     * Returns a map of query parameters to decorate the URL with. The keys and values will be URL encoded for you. A
     * {@code null} value will translate as a query parameter without a value.
     *
     * @param context the context within which the URL is being generated.
     * @return the map of parameters to append to the URL.
     */
    @NonNull
    protected abstract Map<String, String> parameters(@NonNull DisplayURLContext context);

    /**
     * Decorates the URL for the provided context.
     *
     * @param context the context.
     * @param url the URL to decorate.
     * @return the decorated URL.
     */
    @NonNull
    public static String decorate(@NonNull DisplayURLContext context, @NonNull String url) {
        ExtensionList<DisplayURLDecorator> extensionList = ExtensionList
            .lookup(DisplayURLDecorator.class);
        if (extensionList.isEmpty()) {
            return url;
        }
        Map<String, String> parameters = new TreeMap<>();
        // the extension with the highest ordinal wins for duplicate query parameters
        for (DisplayURLDecorator decorator : extensionList.reverseView()) {
            parameters.putAll(fixNull(decorator.parameters(context)));
        }
        if (parameters.isEmpty()) {
            return url;
        }
        Map<String, String> encodedParameters = new TreeMap<>();
        for (Map.Entry<String, String> p : parameters.entrySet()) {
            encodedParameters
                .put(encode(p.getKey()), p.getValue() == null ? null : encode(p.getValue()));
        }
        StringBuilder result = new StringBuilder(2083); // maximum URL length
        char sep = '?';
        int queryStart = url.indexOf(sep);
        if (queryStart == -1) {
            // quick win!
            result.append(url);
        } else {
            // ok this is the hard one, we need to build up the url and strip any duplicate query parameters that we
            // are decorating with
            result.append(url, 0, queryStart);
            for (String pair : url.substring(queryStart + 1).split("&")) {
                int index = pair.indexOf('=');
                if (index == -1 && encodedParameters.containsKey(pair)) {
                    // query parameter without value that we are decorating with
                    continue;
                }
                if (index > 0 && encodedParameters.containsKey(pair.substring(0, index))) {
                    // query parameter with value that we are decorating with
                    continue;
                }
                result.append(sep);
                result.append(pair);
                sep = '&';
            }
        }
        for (Map.Entry<String, String> p : encodedParameters.entrySet()) {
            result.append(sep).append(p.getKey());
            if (p.getValue() != null) {
                result.append('=').append(p.getValue());
            }
            sep = '&';
        }
        return result.toString();
    }

    /**
     * Trust but verify for implementations of this extension point.
     *
     * @param map the map that is supposed to be non-null.
     * @param <K> the key type.
     * @param <V> the value type.
     * @return most certainly a non-null map.
     */
    @NonNull
    private static <K, V> Map<K, V> fixNull(@CheckForNull Map<K, V> map) {
        return map == null ? Collections.emptyMap() : map;
    }

    /**
     * Encodes a query parameter name or value.
     *
     * @param value the name or value to encode.
     * @return the encoded name or value.
     */
    private static String encode(String value) {
        try {
            return URLEncoder.encode(value, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new AssertionError("UTF-8 encoding is mandated by the JLS", e);
        }
    }

}
