/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server.integration.jbpm3;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jboss.bpm.console.client.model.jbpm3.ActiveNodeInfo;
import org.jboss.bpm.console.client.model.jbpm3.DiagramInfo;
import org.jboss.bpm.console.client.model.jbpm3.DiagramNodeInfo;
import org.jboss.bpm.console.server.integration.ExtensionManagement;
import org.jbpm.file.def.FileDefinition;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.taskmgmt.exe.TaskInstance;
import org.jbpm.util.XmlUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class JBPM3ManagementExtension implements ExtensionManagement
{
   private JBPM3CommandDelegate delegate;

   private static final String PROCESSIMAGE_FILENAME = "processimage.jpg";
   private static final String GPD_XML_FILENAME = "gpd.xml";

   public JBPM3ManagementExtension()
   {
      this.delegate = new JBPM3CommandDelegate();
   }

   public byte[] getProcessImage(long processId)
   {
      ProcessDefinition p0 = delegate.getActualDefinition(processId);
      return p0.getFileDefinition().getBytes(PROCESSIMAGE_FILENAME);
   }

   public DiagramInfo getDiagramInfo(long processId)
   {
      ProcessDefinition p0 = delegate.getActualDefinition(processId);
      InputStream in = p0.getFileDefinition().getInputStream(GPD_XML_FILENAME);
      return DiagramInfoParser.parse(in);
   }

   public ActiveNodeInfo getActivNodeInfo(long instanceId)
   {
      ProcessInstance instance = delegate.getActualInstance(instanceId);
      String currentNodeName = instance.getRootToken().getNode().getName();

      DiagramInfo diagram = getDiagramInfo(instance.getProcessDefinition().getId());
      List<DiagramNodeInfo> nodes = diagram.getNodeList();

      DiagramNodeInfo activeNode = null;
      int i = 0;
      for(DiagramNodeInfo n : nodes)
      {
         if(currentNodeName.equals(n.getName()))
         {
            activeNode = n;
            break;
         }
      }

      if(null==activeNode)
         throw new RuntimeException("Failed to retrieve activeNodeInfo for instance " + instanceId);

      return new ActiveNodeInfo(diagram.getWidth(), diagram.getHeight(), activeNode);
   }

   public Map<String, String> getAvailableTaskForms(long processId)
   {
      ProcessDefinition processDefinition = delegate.getActualDefinition(processId);

      return getTask2FileMapping(processDefinition);
   }

   private Map<String, String> getTask2FileMapping(ProcessDefinition processDefinition)
   {
      Map<String, String> result = new HashMap<String, String>();

      final FileDefinition fileDefinition = processDefinition.getFileDefinition();
      if (! fileDefinition.hasFile("forms.xml"))
      {
         // exit with empty list
         // TODO: maybe an exception is better?
         return result;
      }

      final InputStream inputStream = fileDefinition.getInputStream("forms.xml");
      if (inputStream == null)
      {
         throw new RuntimeException("Failed to open stream on forms.xml");
      }

      final Document document = XmlUtil.parseXmlInputStream(inputStream);
      final Element documentElement = document.getDocumentElement();
      final NodeList nodeList = documentElement.getElementsByTagName("form");
      final int length = nodeList.getLength();
      for (int i = 0; i < length; i ++)
      {
         final Element element = (Element) nodeList.item(i);
         final String itemTaskName = element.getAttribute("task");
         final String itemFormName = element.getAttribute("form");
         if (itemTaskName == null || itemFormName == null) {
            continue;
         }
         result.put(itemTaskName, itemFormName);
      }
      return result;
   }

   public byte[] getTaskFormByTaskName(long processId, long taskId)
   {
      ProcessDefinition processDefinition = delegate.getActualDefinition(processId);
      TaskInstance t0 = delegate.getTaskById(taskId);

      Map<String,String> fileMapping = getTask2FileMapping(processDefinition);
      String fileName = fileMapping.get(t0.getName());
      if(fileName !=null)
         return processDefinition.getFileDefinition().getBytes(fileName);
      else
         throw new RuntimeException("No form definition for taskId '"+taskId+"'");

   }

   public org.jboss.bpm.console.client.model.ProcessDefinitionRef deployNewDefinition(byte[] data)
   {
      ProcessDefinition p0 = delegate.deploy(data);
      return Transform.processDefinition(p0);
   }
}
