/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.bpm.console.server;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.bpm.console.client.model.jbpm3.ActiveNodeInfo;
import org.jboss.bpm.console.client.model.jbpm3.DiagramInfo;
import org.jboss.bpm.console.client.model.ProcessDefinitionRef;
import org.jboss.bpm.console.server.integration.ExtensionManagement;
import org.jboss.bpm.console.server.integration.ManagementFactory;
import org.jboss.bpm.console.server.integration.ProcessManagement;
import org.jboss.bpm.console.server.gson.GsonFactory;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.util.Iterator;
import java.util.List;

import com.google.gson.Gson;

/**
 * REST server module for accessing proprietary jbpm data.
 * 
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
@Path("jbpm3")
public class JBPM3MgmtFacade
{
   private static final Log log = LogFactory.getLog(JBPM3MgmtFacade.class);

   private ExtensionManagement extensionManagement = null;
   private ProcessManagement processManagement = null;

   private ProcessManagement getProcessManagement()
   {
      if(null==this.processManagement)
      {
         ManagementFactory factory = ManagementFactory.newInstance();
         this.processManagement = factory.createProcessManagement();
         log.debug("Using ManagementFactory impl:" + factory.getClass().getName());
      }

      return this.processManagement;

   }

   private ExtensionManagement getManagementExtension()
   {
      if(null==this.extensionManagement)
      {
         ManagementFactory factory = ManagementFactory.newInstance();
         this.extensionManagement = factory.createExtensionManagement();
      }

      return this.extensionManagement;

   }

   @POST
   @Path("definitions/new")
   @Produces("application/json")
   @Consumes("multipart/form-data")
   public Response postNewDefinition(
         @Context
         HttpServletRequest request
   )
   {
      ProcessDefinitionRef processRef = null;

      try
      {
         FileItemFactory factory = new DiskFileItemFactory();
         ServletFileUpload upload = new ServletFileUpload(factory);
         List items = upload.parseRequest(request);

         Iterator iter = items.iterator();
         while (iter.hasNext())
         {
            FileItem item = (FileItem) iter.next();

            if (item.isFormField())
            {
               // ignore
               log.debug("Caught form field on file upload: " + item.getName());
            }
            else
            {
               String fieldName = item.getFieldName();
               String fileName = item.getName();
               String contentType = item.getContentType();
               boolean isInMemory = item.isInMemory();
               long sizeInBytes = item.getSize();               

               // Process a file upload in memory
               byte[] data = item.get();

               processRef = getManagementExtension().deployNewDefinition(data);

               // Process stream
               //InputStream uploadedStream = item.getInputStream();
               //uploadedStream.close();

            }


         }
      } catch (Throwable e)
      {
         throw new WebApplicationException(e);
      }

      return Response.ok().build();

   }

   @GET
   @Path("definitions/{id}/image")
   @Produces("image/jpeg")
   public Response getProcessImage(
         @PathParam("id")
         long id
   )
   {
      byte[] image = getManagementExtension().getProcessImage(id);
      if(null==image)
         return Response.serverError().build();
      else
         return Response.ok(image).type("image/jpeg").build();
   }

   @GET
   @Path("definitions/{id}/diagramInfo")
   @Produces("application/json")
   public DiagramInfo getDiagramInfo(
         @PathParam("id")
         long id
   )
   {
      return getManagementExtension().getDiagramInfo(id);
   }

   @GET
   @Path("instances/{id}/activeNodeInfo")
   @Produces("application/json")
   public ActiveNodeInfo getActiveNodeInfo(
         @PathParam("id")
         long id
   )
   {
      ActiveNodeInfo activNodeInfo = getManagementExtension().getActivNodeInfo(id);
      return activNodeInfo;
   }

   @POST
   @Path("tokens/{id}/transition")
   @Produces("application/json")
   public Response signalToken(
         @PathParam("id")
         long id,
         @QueryParam("signal")
         String signalName)
   {
      log.debug("Signal token " + id + " -> " + signalName);

      if ("default transition".equals(signalName))
        signalName = null;
      
      getProcessManagement().signalToken(id, signalName);
      return Response.ok().type("application/json").build();
   }

   @POST
   @Path("tokens/{id}/transition/default")
   @Produces("application/json")
   public Response signalTokenDefault(
         @PathParam("id")
         long id)
   {
      log.debug("Signal token " + id);

      getProcessManagement().signalToken(id, null);
      return Response.ok().type("application/json").build();
   }

   private Response createJsonResponse(Object wrapper)
   {
      Gson gson = GsonFactory.createInstance();
      String json = gson.toJson(wrapper);
      return Response.ok(json).type("application/json").build();
   }

}
