/*
* JBoss, Home of Professional Open Source.
* Copyright 2006, Red Hat Middleware LLC, and individual contributors
* as indicated by the @author tags. See the copyright.txt file in the
* distribution for a full listing of individual contributors. 
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/ 
package org.jboss.dependency.plugins;

import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.jboss.dependency.spi.ControllerContext;
import org.jboss.dependency.spi.ControllerState;
import org.jboss.dependency.spi.ControllerStateModel;

/**
 * Map based controller state model.
 *
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @version $Revision: 1.1 $
 */
public class MapControllerStateModel extends AbstractControllerStateModel
{
   private volatile ControllerStateWrapper first;
   private volatile ControllerStateWrapper last;
   //Not likely to be more than one thread updating states at the same time
   private final Map<ControllerState, ControllerStateWrapper> states = new ConcurrentHashMap<ControllerState, ControllerStateWrapper>(16, .75f, 1);
   
   /** Thread-safe list for obtaining the iterators */
   private final List<ControllerState> stateList = new CopyOnWriteArrayList<ControllerState>();

   
   /**
    * Add a state.
    * Synchronized so that only one state can be added at a time.
    * 
    * @param state The state to add
    * @param before The state to add the new state before, or null to add the new state at the end
    * @return true if the state was added, false if the state already existed and therefore was not added
    * @throws IllegalArgumentException if before is not null and does not exist
    */
   public synchronized boolean addState(ControllerState state, ControllerState before)
   {
      if (states.containsKey(state))
         return false;

      //Work out the values for the new state and create it
      ControllerStateWrapper previous;
      ControllerStateWrapper next;
      int index;

      if (before == null)
      {
         next = null;
         previous = last;
         index = last == null ? 0 : last.getIndex() + 1;
      }
      else
      {
          next = getState(before);
          previous = next.getBefore();
          index = next.getIndex();
      }      

      ControllerStateWrapper newState = new ControllerStateWrapper(this, state, index, previous, next);

      //Add the state so that calls to isValid() can return successfully
      states.put(state, newState);
      stateList.add(index, newState);
      if (previous == null)
         first = newState;
      if (next == null)
         last = newState;
      
      if (next != null)
      {
         //Increment the subsequent state indices in reverse order so that isBeforeState() and isAfterState() are still valid
         //(apart from newState and next, which will be valid once the loop completes)
         ControllerStateWrapper current = last;
         while (current != null)
         {
            current.incrementIndex();
            if (current == next)
               break;
            current = current.getBefore();
         }
      }
      
      //Update the previous and next states to point to the new state so that getNextState() and getPreviousState()
      //pick up the new state. Do next first so that we don't have the possibility of installing to the new state
      //without uninstalling from it
      if (next != null)
         next.setBefore(newState);
      if (previous != null)
         previous.setAfter(newState);
      return true;
   }
   
   @Override
   public ControllerState indexState(ControllerState state)
   {
      return getState(state);
   }

   protected ControllerStateWrapper getState(ControllerState state)
   {
      return getState(state, false);
   }

   protected ControllerStateWrapper getState(ControllerState state, boolean allowNotFound)
   {
      if (state == null)
         throw new IllegalArgumentException("Null state");
      if (state instanceof ControllerStateWrapper)
      {
         ControllerStateWrapper wrapper = (ControllerStateWrapper)state;
         if (wrapper.owner == this)
            return wrapper;
      }

      ControllerStateWrapper found = states.get(state);
      if (found == null && !allowNotFound)
         throw new IllegalArgumentException("No such state " + state + " in states " + states);

      return found;
   }

   protected int getStateIndex(ControllerState state)
   {
      return getStateIndex(state, false);
   }

   protected int getStateIndex(ControllerState state, boolean allowNotFound)
   {
      ControllerStateWrapper stateWrapper = getState(state, allowNotFound);
      return stateWrapper == null  ? -1 : stateWrapper.getIndex(); 
   }

   public ControllerState getPreviousState(ControllerState state)
   {
      return getState(state).getBefore();
   }

   public ControllerState getNextState(ControllerState state)
   {
      return getState(state).getAfter();
   }

   public boolean isBeforeState(ControllerState state, ControllerState reference)
   {
      int stateIndex = getStateIndex(state, true);
      int referenceIndex = getStateIndex(reference, true);
      return stateIndex < referenceIndex;
   }

   public boolean isAfterState(ControllerState state, ControllerState reference)
   {
      int stateIndex = getStateIndex(state, true);
      int referenceIndex = getStateIndex(reference, true);
      return stateIndex > referenceIndex;
   }

   public Iterator<ControllerState> iterator()
   {
      return stateList.iterator();
   }
   
   public ListIterator<ControllerState> listIteraror()
   {
      return stateList.listIterator(states.size() - 1);
   }
   
   public boolean isValidState(ControllerState state)
   {
      return state != null && states.containsKey(state);
   }

   public ControllerState getInitialState()
   {
      ControllerStateWrapper result = first;
      if (result == null)
         throw new IllegalStateException("No initial state");
      return result;
   }
   
   private static class ControllerStateWrapper extends ControllerState
   {
      private static final long serialVersionUID = 1L;
      
      final MapControllerStateModel owner;
      volatile int index;
      volatile ControllerStateWrapper before;
      volatile ControllerStateWrapper after;
      
      public ControllerStateWrapper(MapControllerStateModel owner, ControllerState state, int index, ControllerStateWrapper before, ControllerStateWrapper after)
      {
         super(state.getStateString(), false);
         this.owner = owner;
         this.index = index;
         this.before = before;
         this.after = after;
      }

      int getIndex()
      {
         return index;
      }

      @SuppressWarnings("unused")
      void setIndex(int index)
      {
         this.index = index;
      }
      
      void incrementIndex()
      {
         this.index++;
      }

      ControllerStateWrapper getBefore()
      {
         return before;
      }

      void setBefore(ControllerStateWrapper before)
      {
         this.before = before;
      }

      ControllerStateWrapper getAfter()
      {
         return after;
      }

      void setAfter(ControllerStateWrapper after)
      {
         this.after = after;
      }
   }
}

