/*
 * Decompiled with CFR 0.152.
 */
package org.jivesoftware.openfire.sasl;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.security.sasl.SaslException;
import javax.security.sasl.SaslServer;
import javax.xml.bind.DatatypeConverter;
import org.jivesoftware.openfire.auth.AuthFactory;
import org.jivesoftware.openfire.auth.ConnectionException;
import org.jivesoftware.openfire.auth.InternalUnauthenticatedException;
import org.jivesoftware.openfire.auth.ScramUtils;
import org.jivesoftware.openfire.user.UserNotFoundException;
import org.jivesoftware.util.JiveGlobals;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScramSha1SaslServer
implements SaslServer {
    private static final Logger Log = LoggerFactory.getLogger(ScramSha1SaslServer.class);
    private static final Pattern CLIENT_FIRST_MESSAGE = Pattern.compile("^(([pny])=?([^,]*),([^,]*),)(m?=?[^,]*,?n=([^,]*),r=([^,]*),?.*)$");
    private static final Pattern CLIENT_FINAL_MESSAGE = Pattern.compile("(c=([^,]*),r=([^,]*)),p=(.*)$");
    private String username;
    private State state = State.INITIAL;
    private String nonce;
    private String serverFirstMessage;
    private String clientFirstMessageBare;
    private SecureRandom random = new SecureRandom();

    @Override
    public String getMechanismName() {
        return "SCRAM-SHA-1";
    }

    @Override
    public byte[] evaluateResponse(byte[] response) throws SaslException {
        try {
            byte[] challenge;
            switch (this.state) {
                case INITIAL: {
                    challenge = this.generateServerFirstMessage(response);
                    this.state = State.IN_PROGRESS;
                    break;
                }
                case IN_PROGRESS: {
                    challenge = this.generateServerFinalMessage(response);
                    this.state = State.COMPLETE;
                    break;
                }
                case COMPLETE: {
                    if (response == null || response.length == 0) {
                        challenge = new byte[]{};
                        break;
                    }
                }
                default: {
                    throw new SaslException("No response expected in state " + (Object)((Object)this.state));
                }
            }
            return challenge;
        }
        catch (RuntimeException ex) {
            throw new SaslException("Unexpected exception while evaluating SASL response.", ex);
        }
    }

    private byte[] generateServerFirstMessage(byte[] response) throws SaslException {
        String clientFirstMessage = new String(response, StandardCharsets.UTF_8);
        Matcher m = CLIENT_FIRST_MESSAGE.matcher(clientFirstMessage);
        if (!m.matches()) {
            throw new SaslException("Invalid first client message");
        }
        this.clientFirstMessageBare = m.group(5);
        this.username = m.group(6);
        String clientNonce = m.group(7);
        this.nonce = clientNonce + UUID.randomUUID().toString();
        this.serverFirstMessage = String.format("r=%s,s=%s,i=%d", this.nonce, DatatypeConverter.printBase64Binary((byte[])this.getSalt(this.username)), this.getIterations(this.username));
        return this.serverFirstMessage.getBytes(StandardCharsets.UTF_8);
    }

    private byte[] generateServerFinalMessage(byte[] response) throws SaslException {
        String clientFinalMessage = new String(response, StandardCharsets.UTF_8);
        Matcher m = CLIENT_FINAL_MESSAGE.matcher(clientFinalMessage);
        if (!m.matches()) {
            throw new SaslException("Invalid client final message");
        }
        String clientFinalMessageWithoutProof = m.group(1);
        String clientNonce = m.group(3);
        String proof = m.group(4);
        if (!this.nonce.equals(clientNonce)) {
            throw new SaslException("Client final message has incorrect nonce value");
        }
        try {
            String authMessage = this.clientFirstMessageBare + "," + this.serverFirstMessage + "," + clientFinalMessageWithoutProof;
            byte[] storedKey = this.getStoredKey(this.username);
            if (storedKey == null) {
                throw new SaslException("No stored key for user '" + this.username + "'");
            }
            byte[] serverKey = this.getServerKey(this.username);
            if (serverKey == null) {
                throw new SaslException("No server key for user '" + this.username + "'");
            }
            byte[] clientSignature = ScramUtils.computeHmac(storedKey, authMessage);
            byte[] serverSignature = ScramUtils.computeHmac(serverKey, authMessage);
            byte[] clientKey = (byte[])clientSignature.clone();
            byte[] decodedProof = DatatypeConverter.parseBase64Binary((String)proof);
            for (int i = 0; i < clientKey.length; ++i) {
                int n = i;
                clientKey[n] = (byte)(clientKey[n] ^ decodedProof[i]);
            }
            if (!Arrays.equals(storedKey, MessageDigest.getInstance("SHA-1").digest(clientKey))) {
                throw new SaslException("Authentication failed");
            }
            return ("v=" + DatatypeConverter.printBase64Binary((byte[])serverSignature)).getBytes(StandardCharsets.UTF_8);
        }
        catch (NoSuchAlgorithmException | UserNotFoundException e) {
            throw new SaslException(e.getMessage(), e);
        }
    }

    @Override
    public boolean isComplete() {
        return this.state == State.COMPLETE;
    }

    @Override
    public String getAuthorizationID() {
        if (this.isComplete()) {
            return this.username;
        }
        throw new IllegalStateException("SCRAM-SHA-1 authentication not completed");
    }

    @Override
    public byte[] unwrap(byte[] incoming, int offset, int len) throws SaslException {
        if (this.isComplete()) {
            throw new IllegalStateException("SCRAM-SHA-1 does not support integrity or privacy");
        }
        throw new IllegalStateException("SCRAM-SHA-1 authentication not completed");
    }

    @Override
    public byte[] wrap(byte[] outgoing, int offset, int len) throws SaslException {
        if (this.isComplete()) {
            throw new IllegalStateException("SCRAM-SHA-1 does not support integrity or privacy");
        }
        throw new IllegalStateException("SCRAM-SHA-1 authentication not completed");
    }

    @Override
    public Object getNegotiatedProperty(String propName) {
        if (this.isComplete()) {
            if (propName.equals("javax.security.sasl.qop")) {
                return "auth";
            }
            return null;
        }
        throw new IllegalStateException("SCRAM-SHA-1 authentication not completed");
    }

    @Override
    public void dispose() throws SaslException {
        this.username = null;
        this.state = State.INITIAL;
    }

    private byte[] getSalt(String username) {
        try {
            byte[] salt;
            String saltshaker = AuthFactory.getSalt(username);
            if (saltshaker == null) {
                Log.debug("No salt found, so resetting password.");
                String password = AuthFactory.getPassword(username);
                AuthFactory.setPassword(username, password);
                salt = DatatypeConverter.parseBase64Binary((String)AuthFactory.getSalt(username));
            } else {
                salt = DatatypeConverter.parseBase64Binary((String)saltshaker);
            }
            return salt;
        }
        catch (UnsupportedOperationException | ConnectionException | InternalUnauthenticatedException | UserNotFoundException e) {
            Log.warn("Exception in SCRAM.getSalt():", (Throwable)e);
            byte[] salt = new byte[24];
            this.random.nextBytes(salt);
            return salt;
        }
    }

    private int getIterations(String username) {
        try {
            return AuthFactory.getIterations(username);
        }
        catch (UserNotFoundException e) {
            return JiveGlobals.getIntProperty("sasl.scram-sha-1.iteration-count", 4096);
        }
    }

    private byte[] getServerKey(String username) throws UserNotFoundException {
        String serverKey = AuthFactory.getServerKey(username);
        if (serverKey == null) {
            return null;
        }
        return DatatypeConverter.parseBase64Binary((String)serverKey);
    }

    private byte[] getStoredKey(String username) throws UserNotFoundException {
        String storedKey = AuthFactory.getStoredKey(username);
        if (storedKey == null) {
            return null;
        }
        return DatatypeConverter.parseBase64Binary((String)storedKey);
    }

    private static enum State {
        INITIAL,
        IN_PROGRESS,
        COMPLETE;

    }
}

