/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.code;

import com.oracle.svm.core.util.VMError;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import jdk.vm.ci.code.BytecodeFrame;
import jdk.vm.ci.code.BytecodePosition;
import jdk.vm.ci.code.site.Call;
import jdk.vm.ci.code.site.Infopoint;
import jdk.vm.ci.code.site.InfopointReason;
import jdk.vm.ci.common.JVMCIError;
import jdk.vm.ci.meta.JavaValue;
import jdk.vm.ci.meta.Local;
import jdk.vm.ci.meta.LocalVariableTable;
import jdk.vm.ci.meta.ResolvedJavaMethod;
import org.graalvm.compiler.code.CompilationResult;
import org.graalvm.compiler.code.SourceMapping;
import org.graalvm.compiler.debug.DebugContext;
import org.graalvm.compiler.graph.NodeSourcePosition;

public final class CompilationResultFrameTree {
    public static boolean verify(FrameNode node, Consumer<String> printer) {
        FrameTreeVerifier frameTreeVerifier = new FrameTreeVerifier(printer);
        node.visit(frameTreeVerifier, new Object[0]);
        return frameTreeVerifier.passed();
    }

    public static void dump(FrameNode node, Consumer<String> printer, boolean onlyCallTree, boolean showInfopoints, int maxDepth) {
        if (node != null) {
            printer.accept("\n");
            node.visit(new FrameTreeDumper(printer, onlyCallTree, showInfopoints, maxDepth), 0);
            printer.accept("\n");
        }
    }

    private static final class FrameTreeVerifier
    implements Visitor {
        private final Consumer<String> printer;
        private int issues = 0;

        FrameTreeVerifier(Consumer<String> printer) {
            this.printer = printer;
        }

        boolean passed() {
            return this.issues == 0;
        }

        @Override
        public void apply(FrameNode node, Object ... args) {
            if (node.getStartPos() > node.getEndPos()) {
                this.printer.accept("Error: Node startPos > endPos: ");
                this.printer.accept(node.toString());
                this.printer.accept("\n");
                ++this.issues;
            }
            if (node.nextSibling != null && node.getEndPos() >= node.nextSibling.getStartPos()) {
                this.printer.accept("Error: Overlapping nodes: ");
                this.printer.accept(node.toString());
                this.printer.accept(" with ");
                this.printer.accept(node.nextSibling.toString());
                this.printer.accept("\n");
                ++this.issues;
            }
            node.visitChildren(this, new Object[0]);
        }
    }

    public static class FrameNode {
        public final BytecodePosition frame;
        public final SourcePositionSupplier sourcePos;
        public final FrameNode parent;
        FrameNode nextSibling;

        FrameNode(FrameNode parent, BytecodePosition frame, SourcePositionSupplier sourcePos) {
            this.parent = parent;
            this.frame = frame;
            this.sourcePos = sourcePos;
        }

        boolean hasEqualFrameChain(BytecodePosition otherFrame) {
            return FrameNode.hasEqualCaller(this.frame, otherFrame) && (this.parent == null ? otherFrame.getCaller() == null : this.parent.hasEqualFrameChain(otherFrame.getCaller()));
        }

        static boolean hasEqualCaller(BytecodePosition thisFrame, BytecodePosition thatFrame) {
            return thisFrame.getMethod().equals(thatFrame.getMethod()) && FrameNode.getCallerBCI(thisFrame) == FrameNode.getCallerBCI(thatFrame);
        }

        static int getCallerBCI(BytecodePosition frame) {
            BytecodePosition callerFrame = frame.getCaller();
            if (callerFrame != null) {
                return callerFrame.getBCI();
            }
            return -1;
        }

        public int getStartPos() {
            return this.sourcePos.getStartOffset();
        }

        public int getSpan() {
            if (this.nextSibling != null) {
                return this.nextSibling.getStartPos() - this.getStartPos();
            }
            if (this.parent != null) {
                return this.parent.getStartPos() + this.parent.getSpan() - this.getStartPos();
            }
            throw JVMCIError.shouldNotReachHere();
        }

        public final int getEndPos() {
            return this.getStartPos() + this.getSpan() - 1;
        }

        public final String getPosStr() {
            return String.format("[%d..%d]", this.getStartPos(), this.getEndPos());
        }

        public String toString() {
            return String.format("-%s, span %d, bci %d, method %s", this.getPosStr(), this.getSpan(), this.frame.getBCI(), this.frame.getMethod().format("%h.%n(%p)"));
        }

        public String getLocalsStr() {
            String localsInfo = "";
            if (this.frame instanceof BytecodeFrame) {
                BytecodeFrame bcf = (BytecodeFrame)this.frame;
                Local[] locals = FrameNode.getLocalsBySlot(this.frame.getMethod());
                StringBuilder sb = new StringBuilder();
                if (locals == null) {
                    return localsInfo;
                }
                int combinedLength = Integer.min(locals.length, bcf.numLocals);
                for (int i = 0; i < combinedLength; ++i) {
                    JavaValue value = bcf.getLocalValue(i);
                    if (i > 0) {
                        sb.append(", ");
                    }
                    sb.append("li(");
                    Local local = locals[i];
                    if (local != null) {
                        sb.append(local.getName());
                        sb.append("=");
                    }
                    sb.append(value);
                    sb.append(")");
                }
                localsInfo = sb.toString();
            }
            return localsInfo;
        }

        private static Local[] getLocalsBySlot(ResolvedJavaMethod method) {
            Local[] locals;
            LocalVariableTable lvt = method.getLocalVariableTable();
            Local[] nonEmptySortedLocals = null;
            if (lvt != null && (locals = lvt.getLocals()) != null && locals.length > 0) {
                nonEmptySortedLocals = Arrays.copyOf(locals, locals.length);
                Arrays.sort(nonEmptySortedLocals, (l1, l2) -> l1.getSlot() - l2.getSlot());
            }
            return nonEmptySortedLocals;
        }

        public void visit(Visitor visitor, Object ... varargs) {
            visitor.apply(this, varargs);
        }

        public void visitChildren(Visitor visitor, Object ... varargs) {
        }

        public StackTraceElement getStackTraceElement() {
            return SourcePositionSupplier.getStackTraceElement(this.frame);
        }
    }

    public static interface Visitor {
        public void apply(FrameNode var1, Object ... var2);
    }

    private static final class FrameTreeDumper
    implements Visitor {
        private final Consumer<String> printer;
        private final boolean onlyCallTree;
        private final boolean showSourcePos;
        private final boolean showLocals;
        private final int maxDepth;

        FrameTreeDumper(Consumer<String> printer, boolean onlyCallTree, boolean showSourcePos, int maxDepth) {
            this.printer = printer;
            this.onlyCallTree = onlyCallTree;
            this.showSourcePos = showSourcePos;
            this.showLocals = true;
            this.maxDepth = maxDepth;
        }

        private void indent(int level) {
            this.printer.accept(new String(new char[level * 4]).replace("\u0000", " "));
        }

        @Override
        public void apply(FrameNode node, Object ... args) {
            if (this.onlyCallTree && !(node instanceof CallNode)) {
                return;
            }
            int level = (Integer)args[0];
            if (this.maxDepth > 0 && level > this.maxDepth) {
                return;
            }
            this.indent(level);
            this.printer.accept(node.toString());
            if (this.showSourcePos) {
                this.printer.accept("\n");
                this.indent(level);
                this.printer.accept(" sourcePos: " + node.sourcePos.toString());
            } else {
                StackTraceElement stackTraceElement = node.getStackTraceElement();
                this.printer.accept(" at " + stackTraceElement.getFileName() + ":" + stackTraceElement.getLineNumber());
            }
            if (this.showLocals) {
                this.printer.accept(" locals: ");
                this.printer.accept(node.getLocalsStr());
            }
            this.printer.accept("\n");
            node.visitChildren(this, level + 1);
        }
    }

    public static final class Builder {
        private RootNode root = null;
        private final int targetCodeSize;
        private final int maxDepth;
        private final DebugContext debug;
        private final boolean useSourceMappings;
        private final boolean verify;
        int indexLeft;
        int indexRight;

        public Builder(DebugContext debug, int targetCodeSize, int maxDepth, boolean useSourceMappings, boolean verify) {
            this.targetCodeSize = targetCodeSize;
            this.maxDepth = maxDepth;
            this.useSourceMappings = useSourceMappings;
            this.verify = verify;
            this.debug = debug;
        }

        public CallNode build(CompilationResult compilationResult) {
            try (DebugContext.Scope s = this.debug.scope((Object)"FrameTree.Builder", (Object)compilationResult);){
                SourcePositionSupplier wrapper;
                this.debug.log(3, "Building FrameTree for %s", (Object)compilationResult);
                List infopoints = compilationResult.getInfopoints();
                List sourceMappings = compilationResult.getSourceMappings();
                ArrayList<SourcePositionSupplier> sourcePosData = new ArrayList<SourcePositionSupplier>(infopoints.size() + sourceMappings.size());
                InfopointSourceWrapper infopointForRoot = null;
                for (Infopoint infopoint : infopoints) {
                    wrapper = InfopointSourceWrapper.create(infopoint, this.maxDepth);
                    if (wrapper != null) {
                        sourcePosData.add(wrapper);
                        infopointForRoot = wrapper;
                        continue;
                    }
                    this.debug.log(4, " Discard Infopoint %s", (Object)infopoint);
                }
                if (this.useSourceMappings) {
                    for (SourceMapping sourceMapping : sourceMappings) {
                        wrapper = SourceMappingWrapper.create(sourceMapping, this.maxDepth);
                        if (wrapper != null) {
                            if (((SourceMappingWrapper)wrapper).getStartOffset() > this.targetCodeSize - 1) {
                                if (!this.debug.isLogEnabled(4)) continue;
                                this.debug.log(" Discard SourceMapping outside code-range %s", (Object)SourceMappingWrapper.getSourceMappingString(sourceMapping));
                                continue;
                            }
                            sourcePosData.add(wrapper);
                            continue;
                        }
                        if (!this.debug.isLogEnabled(4)) continue;
                        this.debug.log(" Discard SourceMapping %s", (Object)SourceMappingWrapper.getSourceMappingString(sourceMapping));
                    }
                }
                sourcePosData.sort(Comparator.naturalOrder());
                if (this.useSourceMappings) {
                    this.nullifyOverlappingSourcePositions(sourcePosData);
                }
                if (this.debug.isLogEnabled(4)) {
                    this.debug.log("Sorted input data:");
                    for (SourcePositionSupplier sourcePositionSupplier : sourcePosData) {
                        if (sourcePositionSupplier == null) continue;
                        this.debug.log(" %s", (Object)sourcePositionSupplier);
                    }
                }
                if (infopointForRoot != null) {
                    Object bcp = infopointForRoot.getBytecodePosition();
                    while (bcp.getCaller() != null) {
                        bcp = bcp.getCaller();
                    }
                    this.visitFrame(infopointForRoot, (BytecodePosition)bcp, null);
                } else {
                    this.debug.log("Warning: Constructing FrameTree from SourceMappings only is (currently) unsafe");
                }
                for (SourcePositionSupplier sourcePos : sourcePosData) {
                    if (sourcePos == null) continue;
                    this.visitFrame(sourcePos, sourcePos.getBytecodePosition(), null);
                }
                if (this.root != null) {
                    StringBuilder sb;
                    if (this.debug.isLogEnabled(4)) {
                        sb = new StringBuilder();
                        CompilationResultFrameTree.dump(this.root, sb::append, false, false, 0);
                        this.debug.log("%s", (Object)sb);
                    }
                    if (this.verify) {
                        sb = this.debug.isLogEnabled() ? new StringBuilder() : null;
                        boolean verifyResult = CompilationResultFrameTree.verify(this.root, str -> {
                            if (sb != null) {
                                sb.append((String)str);
                            }
                        });
                        if (this.debug.isLogEnabled()) {
                            this.debug.log("%s", (Object)sb);
                        }
                        VMError.guarantee(verifyResult, "FrameTree verification failed");
                    }
                }
            }
            catch (Throwable e) {
                throw this.debug.handle(e);
            }
            return this.root;
        }

        static boolean overlappingSourcePosition(SourcePositionSupplier left, SourcePositionSupplier right) {
            return left.getEndOffset() >= right.getStartOffset();
        }

        private void nullifyLeft(List<SourcePositionSupplier> sourcePosData) {
            sourcePosData.set(this.indexLeft, null);
            for (int i = this.indexLeft; i < this.indexRight; ++i) {
                VMError.guarantee(sourcePosData.get(i) == null, "Why is there a non-null entry between left and right?");
            }
            this.indexLeft = this.indexRight++;
        }

        private void nullifyRight(List<SourcePositionSupplier> sourcePosData) {
            sourcePosData.set(this.indexRight++, null);
        }

        void nullifyOverlappingSourcePositions(List<SourcePositionSupplier> sourcePosData) {
            this.indexLeft = 0;
            this.indexRight = 1;
            while (this.indexRight < sourcePosData.size()) {
                SourcePositionSupplier rightPos;
                SourcePositionSupplier leftPos = sourcePosData.get(this.indexLeft);
                if (Builder.overlappingSourcePosition(leftPos, rightPos = sourcePosData.get(this.indexRight))) {
                    this.debug.log(4, "Handle Overlapping SourcePositions: %s | %s", (Object)leftPos, (Object)rightPos);
                    if (leftPos instanceof InfopointSourceWrapper && rightPos instanceof InfopointSourceWrapper) {
                        Infopoint leftIP = ((InfopointSourceWrapper)leftPos).infopoint;
                        Infopoint rightIP = ((InfopointSourceWrapper)rightPos).infopoint;
                        if (leftIP.reason == InfopointReason.BYTECODE_POSITION) {
                            this.nullifyRight(sourcePosData);
                            continue;
                        }
                        if (rightIP.reason == InfopointReason.BYTECODE_POSITION) {
                            this.nullifyLeft(sourcePosData);
                            continue;
                        }
                        JVMCIError.shouldNotReachHere((String)"Unhandled overlapping Infopoints");
                        continue;
                    }
                    if (leftPos instanceof SourceMappingWrapper && rightPos instanceof SourceMappingWrapper) {
                        SourceMappingWrapper left = (SourceMappingWrapper)leftPos;
                        SourceMappingWrapper right = (SourceMappingWrapper)rightPos;
                        VMError.guarantee(!left.hasCode() || !right.hasCode(), "Non-empty SourceMappings never overlap");
                        if (left.hasCode()) {
                            this.nullifyRight(sourcePosData);
                            continue;
                        }
                        if (right.hasCode()) {
                            this.nullifyLeft(sourcePosData);
                            continue;
                        }
                        if (right.getCallDepth() > left.getCallDepth()) {
                            this.nullifyLeft(sourcePosData);
                            continue;
                        }
                        this.nullifyRight(sourcePosData);
                        continue;
                    }
                    if (leftPos instanceof InfopointSourceWrapper) {
                        this.nullifyRight(sourcePosData);
                        continue;
                    }
                    if (rightPos instanceof InfopointSourceWrapper) {
                        this.nullifyLeft(sourcePosData);
                        continue;
                    }
                    JVMCIError.shouldNotReachHere((String)"Unhandled overlapping SourcePositions");
                    continue;
                }
                ++this.indexRight;
                do {
                    ++this.indexLeft;
                } while (sourcePosData.get(this.indexLeft) == null);
            }
        }

        private CallNode visitFrame(SourcePositionSupplier sourcePos, BytecodePosition frame, BytecodePosition callee) {
            BytecodePosition caller = frame.getCaller();
            if (caller != null) {
                CallNode parent = this.visitFrame(sourcePos, caller, frame);
                CallNode child = parent.getCallNodeChild(frame, sourcePos);
                if (child == null) {
                    return parent.addCallNode(frame, sourcePos);
                }
                if (callee == null) {
                    child.addFrameInfo(frame, sourcePos);
                }
                return child;
            }
            if (this.root == null) {
                this.root = new RootNode(frame, sourcePos);
                return this.root;
            }
            boolean hasEqualCaller = FrameNode.hasEqualCaller(this.root.frame, frame);
            if (this.debug.isLogEnabled() && !hasEqualCaller) {
                this.debug.log("Bottom frame mismatch for %s", (Object)sourcePos);
            }
            if (callee == null && hasEqualCaller) {
                this.root.addFrameInfo(frame, sourcePos);
            }
            return this.root;
        }

        final class RootNode
        extends CallNode {
            RootNode(BytecodePosition firstFrame, SourcePositionSupplier firstSourcePos) {
                super(null, firstFrame, firstSourcePos);
            }

            @Override
            public int getStartPos() {
                return 0;
            }

            @Override
            public int getSpan() {
                return Builder.this.targetCodeSize;
            }

            @Override
            public String toString() {
                String prefix = "R%s, span %d, method %s";
                String methodName = this.frame.getMethod().format("%h.%n(%p)");
                return String.format(prefix, this.getPosStr(), this.getSpan(), methodName);
            }
        }
    }

    public static class CallNode
    extends FrameNode {
        FrameNode firstChild;

        CallNode(FrameNode parent, BytecodePosition frame, SourcePositionSupplier sourcePos) {
            super(parent, frame, sourcePos);
        }

        CallNode getCallNodeChild(BytecodePosition otherFrame, SourcePositionSupplier otherSourcePos) {
            FrameNode node = this.firstChild;
            while (node != null) {
                if (node instanceof CallNode && node.hasEqualFrameChain(otherFrame) && otherSourcePos.getStartOffset() <= node.getEndPos()) {
                    return (CallNode)node;
                }
                node = node.nextSibling;
            }
            return null;
        }

        private <N extends FrameNode> N addChild(N newNode) {
            FrameNode node;
            assert (newNode.parent == this);
            if (this.firstChild == null) {
                this.firstChild = newNode;
                return newNode;
            }
            FrameNode lastNode = node = this.firstChild;
            while (node != null) {
                lastNode = node;
                node = node.nextSibling;
            }
            lastNode.nextSibling = newNode;
            return newNode;
        }

        CallNode addCallNode(BytecodePosition f, SourcePositionSupplier srcPos) {
            return this.addChild(new CallNode(this, f, srcPos));
        }

        FrameNode addFrameInfo(BytecodePosition f, SourcePositionSupplier srcPos) {
            return this.addChild(new FrameNode(this, f, srcPos));
        }

        @Override
        public void visitChildren(Visitor visitor, Object ... varargs) {
            FrameNode node = this.firstChild;
            while (node != null) {
                visitor.apply(node, varargs);
                node = node.nextSibling;
            }
        }

        @Override
        public String toString() {
            String prefix = "+%s, span %d, method %s, caller bci %d";
            String methodName = this.frame.getMethod().format("%h.%n(%p)");
            return String.format(prefix, this.getPosStr(), this.getSpan(), methodName, CallNode.getCallerBCI(this.frame));
        }
    }

    public static final class SourceMappingWrapper
    extends SourcePositionSupplier {
        public final SourceMapping sourceMapping;

        public static SourceMappingWrapper create(SourceMapping sourceMapping, int maxDepth) {
            if (sourceMapping.getSourcePosition() == null) {
                return null;
            }
            NodeSourcePosition pos = sourceMapping.getSourcePosition();
            int depth = SourceMappingWrapper.getCallDepth((BytecodePosition)pos);
            if (depth > maxDepth) {
                return null;
            }
            if (sourceMapping.getStartOffset() > sourceMapping.getEndOffset()) {
                JVMCIError.shouldNotReachHere((String)("Invalid SourceMapping " + SourceMappingWrapper.getSourceMappingString(sourceMapping)));
            }
            return new SourceMappingWrapper((BytecodePosition)pos, depth, sourceMapping);
        }

        static String getSourceMappingString(SourceMapping sourceMapping) {
            NodeSourcePosition pos = sourceMapping.getSourcePosition();
            SourceMappingWrapper tmp = new SourceMappingWrapper((BytecodePosition)pos, SourceMappingWrapper.getCallDepth((BytecodePosition)pos), sourceMapping);
            return tmp.getPosStr() + " with " + tmp.getStackFrameStr();
        }

        private SourceMappingWrapper(BytecodePosition bytecodePosition, int callDepth, SourceMapping sourceMapping) {
            super(bytecodePosition, callDepth);
            this.sourceMapping = sourceMapping;
        }

        @Override
        public int getStartOffset() {
            return this.sourceMapping.getStartOffset();
        }

        public boolean hasCode() {
            return this.sourceMapping.getEndOffset() > this.sourceMapping.getStartOffset();
        }

        @Override
        public int getSize() {
            if (!this.hasCode()) {
                return 1;
            }
            return this.sourceMapping.getEndOffset() - this.sourceMapping.getStartOffset();
        }

        @Override
        public int compareTo(SourcePositionSupplier o) {
            int res = super.compareTo(o);
            if (res != 0) {
                return res;
            }
            if (o instanceof SourceMappingWrapper) {
                int thisSize = this.getSize();
                int thatSize = o.getSize();
                if (thisSize == 0) {
                    if (thatSize > 0) {
                        return -1;
                    }
                } else if (thatSize == 0 && thisSize > 0) {
                    return 1;
                }
                return 0;
            }
            return -1;
        }
    }

    public static final class InfopointSourceWrapper
    extends SourcePositionSupplier {
        public final Infopoint infopoint;

        public static InfopointSourceWrapper create(Infopoint infopoint, int maxDepth) {
            if (infopoint.debugInfo == null || infopoint.debugInfo.getBytecodePosition() == null) {
                return null;
            }
            BytecodePosition pos = infopoint.debugInfo.getBytecodePosition();
            int depth = InfopointSourceWrapper.getCallDepth(pos);
            if (depth > maxDepth) {
                return null;
            }
            return new InfopointSourceWrapper(pos, depth, infopoint);
        }

        private InfopointSourceWrapper(BytecodePosition bytecodePosition, int callDepth, Infopoint infopoint) {
            super(bytecodePosition, callDepth);
            this.infopoint = infopoint;
        }

        @Override
        public int getStartOffset() {
            return this.infopoint.pcOffset;
        }

        @Override
        public int getSize() {
            if (this.infopoint instanceof Call) {
                return ((Call)this.infopoint).size;
            }
            return 1;
        }

        @Override
        public int compareTo(SourcePositionSupplier o) {
            int res = super.compareTo(o);
            if (res != 0) {
                return res;
            }
            if (o instanceof InfopointSourceWrapper) {
                InfopointSourceWrapper other = (InfopointSourceWrapper)o;
                return this.infopoint.reason.compareTo((Enum)other.infopoint.reason);
            }
            return 1;
        }
    }

    public static abstract class SourcePositionSupplier
    implements Comparable<SourcePositionSupplier> {
        private final BytecodePosition bytecodePosition;
        private final int callDepth;

        protected SourcePositionSupplier(BytecodePosition bytecodePosition, int callDepth) {
            this.bytecodePosition = bytecodePosition;
            this.callDepth = callDepth;
        }

        public abstract int getStartOffset();

        public abstract int getSize();

        public int getEndOffset() {
            return this.getStartOffset() + this.getSize() - 1;
        }

        public BytecodePosition getBytecodePosition() {
            return this.bytecodePosition;
        }

        public static StackTraceElement getStackTraceElement(BytecodePosition pos) {
            return pos.getMethod().asStackTraceElement(pos.getBCI());
        }

        public final StackTraceElement getStackTraceElement() {
            return SourcePositionSupplier.getStackTraceElement(this.getBytecodePosition());
        }

        public final String getPosStr() {
            return String.format("[%d..%d]", this.getStartOffset(), this.getEndOffset());
        }

        public final String getStackFrameStr() {
            StackTraceElement stackTraceElement = this.getStackTraceElement();
            if (stackTraceElement.getFileName() != null && stackTraceElement.getLineNumber() > 0) {
                return stackTraceElement.toString();
            }
            return this.getBytecodePosition().getMethod().format("%h.%n(%p)");
        }

        public int getCallDepth() {
            return this.callDepth;
        }

        protected static int getCallDepth(BytecodePosition startPos) {
            int depth = 0;
            for (BytecodePosition pos = startPos; pos != null; pos = pos.getCaller()) {
                ++depth;
            }
            return depth;
        }

        public final String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append(this.getClass().getName());
            sb.append(" ");
            sb.append(this.getPosStr());
            sb.append(" with ");
            sb.append(this.getStackFrameStr());
            sb.append(" depth ");
            sb.append(this.getCallDepth());
            return sb.toString();
        }

        @Override
        public int compareTo(SourcePositionSupplier o) {
            if (this.getStartOffset() < o.getStartOffset()) {
                return -1;
            }
            if (this.getStartOffset() > o.getStartOffset()) {
                return 1;
            }
            return 0;
        }
    }
}

