/*
 * Copyright (c) 2018, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.nodes.access;

import java.util.Set;

import com.oracle.truffle.api.CompilerDirectives;
import com.oracle.truffle.api.CompilerDirectives.CompilationFinal;
import com.oracle.truffle.api.frame.Frame;
import com.oracle.truffle.api.frame.FrameSlotKind;
import com.oracle.truffle.api.frame.VirtualFrame;
import com.oracle.truffle.api.instrumentation.Tag;
import com.oracle.truffle.api.nodes.UnexpectedResultException;
import com.oracle.truffle.js.nodes.JavaScriptNode;
import com.oracle.truffle.js.runtime.Errors;

public final class TemporalDeadZoneCheckNode extends FrameSlotNode {
    @Child private JavaScriptNode child;
    @Child private ScopeFrameNode levelFrameNode;
    @CompilationFinal private boolean seenDead;

    private TemporalDeadZoneCheckNode(int slot, Object identifier, ScopeFrameNode levelFrameNode, JavaScriptNode child) {
        super(slot, identifier);
        this.levelFrameNode = levelFrameNode;
        this.child = child;
    }

    private void checkNotDead(VirtualFrame frame) {
        Frame levelFrame = levelFrameNode.executeFrame(frame);
        if (CompilerDirectives.injectBranchProbability(CompilerDirectives.SLOWPATH_PROBABILITY, levelFrame.getTag(slot) == FrameSlotKind.Illegal.tag)) {
            if (!seenDead) {
                CompilerDirectives.transferToInterpreterAndInvalidate();
                seenDead = true;
            }
            throw Errors.createReferenceErrorNotDefined(getIdentifier(), this);
        }
    }

    @Override
    public Object execute(VirtualFrame frame) {
        checkNotDead(frame);
        return child.execute(frame);
    }

    @Override
    public int executeInt(VirtualFrame frame) throws UnexpectedResultException {
        checkNotDead(frame);
        return child.executeInt(frame);
    }

    @Override
    public double executeDouble(VirtualFrame frame) throws UnexpectedResultException {
        checkNotDead(frame);
        return child.executeDouble(frame);
    }

    @Override
    public boolean executeBoolean(VirtualFrame frame) throws UnexpectedResultException {
        checkNotDead(frame);
        return child.executeBoolean(frame);
    }

    @Override
    public ScopeFrameNode getLevelFrameNode() {
        return levelFrameNode;
    }

    @Override
    public boolean hasTemporalDeadZone() {
        return true;
    }

    public static TemporalDeadZoneCheckNode create(int slotIndex, Object identifier, ScopeFrameNode levelFrameNode, JavaScriptNode rhs) {
        return new TemporalDeadZoneCheckNode(slotIndex, identifier, levelFrameNode, rhs);
    }

    @Override
    protected JavaScriptNode copyUninitialized(Set<Class<? extends Tag>> materializedTags) {
        return new TemporalDeadZoneCheckNode(getSlotIndex(), getIdentifier(), levelFrameNode, cloneUninitialized(child, materializedTags));
    }
}
