/*
 * Copyright (c) 2023, 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.runtime.util;

import com.oracle.truffle.api.profiles.InlinedBranchProfile;
import com.oracle.truffle.api.profiles.InlinedConditionProfile;

/**
 * Inlined profile builder.
 *
 * @see com.oracle.truffle.api.profiles.InlinedProfile
 */
public class InlinedProfileBuilder {

    protected final int stateFieldStart;
    protected final int stateFieldLength;
    protected int stateFieldCursor;

    /**
     * Allocates a new inlined profile builder for a (partial) StateField.
     */
    public InlinedProfileBuilder(int offset, int length) {
        this.stateFieldStart = offset;
        this.stateFieldCursor = offset;
        this.stateFieldLength = length;
    }

    protected final void maybeAdvanceStateField(int bits) {
        if (stateFieldCursor + bits > stateFieldStart + stateFieldLength) {
            throw new UnsupportedOperationException("Not enough state bits available");
        }
    }

    /**
     * Returns the next state subfield for the required number of bits.
     */
    private int stateFieldOffsetForBits(int bits) {
        maybeAdvanceStateField(bits);
        int offset = stateFieldCursor;
        stateFieldCursor += bits;
        return offset;
    }

    /**
     * Adds and returns a new {@link InlinedConditionProfile}.
     */
    public final int conditionProfile() {
        return stateFieldOffsetForBits(InlinedProfileBag.CONDITION_PROFILE_STATE_BITS);
    }

    /**
     * Adds and returns a new {@link InlinedBranchProfile}.
     */
    public final int branchProfile() {
        return stateFieldOffsetForBits(InlinedProfileBag.BRANCH_PROFILE_STATE_BITS);
    }

}
