/*
 * Decompiled with CFR 0.152.
 */
package org.flywaydb.commandline;

import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.FlywayException;
import org.flywaydb.core.api.MigrationInfo;
import org.flywaydb.core.internal.info.MigrationInfoDumper;
import org.flywaydb.core.internal.util.ClassUtils;
import org.flywaydb.core.internal.util.FileCopyUtils;
import org.flywaydb.core.internal.util.StringUtils;
import org.flywaydb.core.internal.util.VersionPrinter;
import org.flywaydb.core.internal.util.logging.Log;
import org.flywaydb.core.internal.util.logging.LogCreator;
import org.flywaydb.core.internal.util.logging.LogFactory;
import org.flywaydb.core.internal.util.logging.console.ConsoleLog;
import org.flywaydb.core.internal.util.logging.console.ConsoleLogCreator;

public class Main {
    private static Log LOG;
    private static final String PROPERTY_JAR_DIRS = "flyway.jarDirs";

    static void initLogging(ConsoleLog.Level level) {
        LogFactory.setFallbackLogCreator((LogCreator)new ConsoleLogCreator(level));
        LOG = LogFactory.getLog(Main.class);
    }

    public static void main(String[] args) {
        ConsoleLog.Level logLevel = Main.getLogLevel(args);
        Main.initLogging(logLevel);
        try {
            List<String> operations;
            Main.printVersion();
            if (Main.isPrintVersionAndExit(args)) {
                System.exit(0);
            }
            if ((operations = Main.determineOperations(args)).isEmpty()) {
                Main.printUsage();
                return;
            }
            Properties properties = new Properties();
            Main.initializeDefaults(properties);
            Main.loadConfiguration(properties, args);
            Main.overrideConfiguration(properties, args);
            Main.promptForCredentialsIfMissing(properties);
            Main.dumpConfiguration(properties);
            Main.loadJdbcDrivers();
            Main.loadJavaMigrationsFromJarDirs(properties);
            Flyway flyway = new Flyway();
            Main.filterProperties(properties);
            flyway.configure(properties);
            for (String operation : operations) {
                Main.executeOperation(flyway, operation);
            }
        }
        catch (Exception e) {
            if (logLevel == ConsoleLog.Level.DEBUG) {
                LOG.error("Unexpected error", e);
            } else if (e instanceof FlywayException) {
                LOG.error(e.getMessage());
            } else {
                LOG.error(e.toString());
            }
            System.exit(1);
        }
    }

    private static boolean isPrintVersionAndExit(String[] args) {
        for (String arg : args) {
            if (!"-v".equals(arg)) continue;
            return true;
        }
        return false;
    }

    private static void executeOperation(Flyway flyway, String operation) {
        if ("clean".equals(operation)) {
            flyway.clean();
        } else if ("baseline".equals(operation)) {
            flyway.baseline();
        } else if ("migrate".equals(operation)) {
            flyway.migrate();
        } else if ("validate".equals(operation)) {
            flyway.validate();
        } else if ("info".equals(operation)) {
            LOG.info("\n" + MigrationInfoDumper.dumpToAsciiTable((MigrationInfo[])flyway.info().all()));
        } else if ("repair".equals(operation)) {
            flyway.repair();
        } else {
            LOG.error("Invalid operation: " + operation);
            Main.printUsage();
            System.exit(1);
        }
    }

    private static ConsoleLog.Level getLogLevel(String[] args) {
        for (String arg : args) {
            if ("-X".equals(arg)) {
                return ConsoleLog.Level.DEBUG;
            }
            if (!"-q".equals(arg)) continue;
            return ConsoleLog.Level.WARN;
        }
        return ConsoleLog.Level.INFO;
    }

    private static void initializeDefaults(Properties properties) {
        properties.put("flyway.locations", "filesystem:" + new File(Main.getInstallationDir(), "sql").getAbsolutePath());
        properties.put(PROPERTY_JAR_DIRS, new File(Main.getInstallationDir(), "jars").getAbsolutePath());
    }

    private static void filterProperties(Properties properties) {
        properties.remove(PROPERTY_JAR_DIRS);
        properties.remove("flyway.configFile");
        properties.remove("flyway.configFileEncoding");
    }

    private static void printVersion() throws IOException {
        VersionPrinter.printVersion();
        LOG.info("");
        LOG.debug("Java " + System.getProperty("java.version") + " (" + System.getProperty("java.vendor") + ")");
        LOG.debug(System.getProperty("os.name") + " " + System.getProperty("os.version") + " " + System.getProperty("os.arch") + "\n");
    }

    private static void printUsage() {
        LOG.info("Usage");
        LOG.info("=====");
        LOG.info("");
        LOG.info("flyway [options] command");
        LOG.info("");
        LOG.info("By default, the configuration will be read from conf/flyway.conf.");
        LOG.info("Options passed from the command-line override the configuration.");
        LOG.info("");
        LOG.info("Commands");
        LOG.info("--------");
        LOG.info("migrate  : Migrates the database");
        LOG.info("clean    : Drops all objects in the configured schemas");
        LOG.info("info     : Prints the information about applied, current and pending migrations");
        LOG.info("validate : Validates the applied migrations against the ones on the classpath");
        LOG.info("baseline : Baselines an existing database at the baselineVersion");
        LOG.info("repair   : Repairs the metadata table");
        LOG.info("");
        LOG.info("Options (Format: -key=value)");
        LOG.info("-------");
        LOG.info("driver                       : Fully qualified classname of the jdbc driver");
        LOG.info("url                          : Jdbc url to use to connect to the database");
        LOG.info("user                         : User to use to connect to the database");
        LOG.info("password                     : Password to use to connect to the database");
        LOG.info("schemas                      : Comma-separated list of the schemas managed by Flyway");
        LOG.info("table                        : Name of Flyway's metadata table");
        LOG.info("locations                    : Classpath locations to scan recursively for migrations");
        LOG.info("resolvers                    : Comma-separated list of custom MigrationResolvers");
        LOG.info("skipDefaultResolvers         : Skips default resolvers (jdbc, sql and Spring-jdbc)");
        LOG.info("sqlMigrationPrefix           : File name prefix for sql migrations");
        LOG.info("repeatableSqlMigrationPrefix : File name prefix for repeatable sql migrations");
        LOG.info("sqlMigrationSeparator        : File name separator for sql migrations");
        LOG.info("sqlMigrationSuffix           : File name suffix for sql migrations");
        LOG.info("encoding                     : Encoding of sql migrations");
        LOG.info("placeholderReplacement       : Whether placeholders should be replaced");
        LOG.info("placeholders                 : Placeholders to replace in sql migrations");
        LOG.info("placeholderPrefix            : Prefix of every placeholder");
        LOG.info("placeholderSuffix            : Suffix of every placeholder");
        LOG.info("target                       : Target version up to which Flyway should use migrations");
        LOG.info("outOfOrder                   : Allows migrations to be run \"out of order\"");
        LOG.info("callbacks                    : Comma-separated list of FlywayCallback classes");
        LOG.info("skipDefaultCallbacks         : Skips default callbacks (sql)");
        LOG.info("validateOnMigrate            : Validate when running migrate");
        LOG.info("ignoreFutureMigrations       : Allow future migrations when validating");
        LOG.info("cleanOnValidationError       : Automatically clean on a validation error");
        LOG.info("cleanDisabled                : Whether to disable clean");
        LOG.info("baselineVersion              : Version to tag schema with when executing baseline");
        LOG.info("baselineDescription          : Description to tag schema with when executing baseline");
        LOG.info("baselineOnMigrate            : Baseline on migrate against uninitialized non-empty schema");
        LOG.info("configFile                   : Config file to use (default: conf/flyway.properties)");
        LOG.info("configFileEncoding           : Encoding of the config file (default: UTF-8)");
        LOG.info("jarDirs                      : Dirs for Jdbc drivers & Java migrations (default: jars)");
        LOG.info("");
        LOG.info("Add -X to print debug output");
        LOG.info("Add -q to suppress all output, except for errors and warnings");
        LOG.info("Add -v to print the Flyway version and exit");
        LOG.info("");
        LOG.info("Example");
        LOG.info("-------");
        LOG.info("flyway -user=myuser -password=s3cr3t -url=jdbc:h2:mem -placeholders.abc=def migrate");
        LOG.info("");
        LOG.info("More info at https://flywaydb.org/documentation/commandline");
    }

    private static void loadJdbcDrivers() throws IOException {
        File driversDir = new File(Main.getInstallationDir(), "drivers");
        File[] files = driversDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".jar");
            }
        });
        if (files == null) {
            LOG.error("Directory for Jdbc Drivers not found: " + driversDir.getAbsolutePath());
            System.exit(1);
        }
        for (File file : files) {
            Main.addJarOrDirectoryToClasspath(file.getPath());
        }
    }

    private static void loadJavaMigrationsFromJarDirs(Properties properties) throws IOException {
        String[] dirs;
        String jarDirs = properties.getProperty(PROPERTY_JAR_DIRS);
        if (!StringUtils.hasLength((String)jarDirs)) {
            return;
        }
        jarDirs = jarDirs.replace(File.pathSeparator, ",");
        for (String dirName : dirs = StringUtils.tokenizeToStringArray((String)jarDirs, (String)",")) {
            File dir = new File(dirName);
            File[] files = dir.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(".jar");
                }
            });
            if (files == null) {
                LOG.error("Directory for Java Migrations not found: " + dirName);
                System.exit(1);
            }
            for (File file : files) {
                Main.addJarOrDirectoryToClasspath(file.getPath());
            }
        }
    }

    static void addJarOrDirectoryToClasspath(String name) throws IOException {
        LOG.debug("Adding location to classpath: " + name);
        try {
            URL url = new File(name).toURI().toURL();
            URLClassLoader sysloader = (URLClassLoader)ClassLoader.getSystemClassLoader();
            Method method = URLClassLoader.class.getDeclaredMethod("addURL", URL.class);
            method.setAccessible(true);
            method.invoke((Object)sysloader, url);
        }
        catch (Exception e) {
            throw new FlywayException("Unable to load " + name, (Throwable)e);
        }
    }

    static void loadConfiguration(Properties properties, String[] args) {
        String encoding = Main.determineConfigurationFileEncoding(args);
        Main.loadConfigurationFile(properties, Main.getInstallationDir() + "/conf/flyway.conf", encoding, false);
        Main.loadConfigurationFile(properties, System.getProperty("user.home") + "/flyway.conf", encoding, false);
        Main.loadConfigurationFile(properties, "flyway.conf", encoding, false);
        String configFile = Main.determineConfigurationFileArgument(args);
        if (configFile != null) {
            Main.loadConfigurationFile(properties, configFile, encoding, true);
        }
    }

    private static boolean loadConfigurationFile(Properties properties, String file, String encoding, boolean failIfMissing) throws FlywayException {
        File configFile = new File(file);
        String errorMessage = "Unable to load config file: " + configFile.getAbsolutePath();
        if (!configFile.isFile() || !configFile.canRead()) {
            if (!failIfMissing) {
                LOG.debug(errorMessage);
                return false;
            }
            throw new FlywayException(errorMessage);
        }
        LOG.debug("Loading config file: " + configFile.getAbsolutePath());
        try {
            String contents = FileCopyUtils.copyToString((Reader)new InputStreamReader((InputStream)new FileInputStream(configFile), encoding));
            properties.load(new StringReader(contents.replace("\\", "\\\\")));
            return true;
        }
        catch (IOException e) {
            throw new FlywayException(errorMessage, (Throwable)e);
        }
    }

    private static void promptForCredentialsIfMissing(Properties properties) {
        Console console = System.console();
        if (console == null) {
            return;
        }
        if (!properties.contains("flyway.url")) {
            return;
        }
        if (!properties.containsKey("flyway.user")) {
            properties.put("flyway.user", console.readLine("Database user: ", new Object[0]));
        }
        if (!properties.containsKey("flyway.password")) {
            char[] password = console.readPassword("Database password: ", new Object[0]);
            properties.put("flyway.password", password == null ? "" : String.valueOf(password));
        }
    }

    private static void dumpConfiguration(Properties properties) {
        LOG.debug("Using configuration:");
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            String value = entry.getValue().toString();
            value = "flyway.password".equals(entry.getKey()) ? StringUtils.trimOrPad((String)"", (int)value.length(), (char)'*') : value;
            LOG.debug(entry.getKey() + " -> " + value);
        }
    }

    private static String determineConfigurationFileArgument(String[] args) {
        for (String arg : args) {
            if (!Main.isPropertyArgument(arg) || !"configFile".equals(Main.getArgumentProperty(arg))) continue;
            return Main.getArgumentValue(arg);
        }
        return null;
    }

    private static String getInstallationDir() {
        String path = ClassUtils.getLocationOnDisk(Main.class);
        return new File(path).getParentFile().getParentFile().getAbsolutePath();
    }

    private static String determineConfigurationFileEncoding(String[] args) {
        for (String arg : args) {
            if (!Main.isPropertyArgument(arg) || !"configFileEncoding".equals(Main.getArgumentProperty(arg))) continue;
            return Main.getArgumentValue(arg);
        }
        return "UTF-8";
    }

    static void overrideConfiguration(Properties properties, String[] args) {
        for (String arg : args) {
            if (!Main.isPropertyArgument(arg)) continue;
            properties.put("flyway." + Main.getArgumentProperty(arg), Main.getArgumentValue(arg));
        }
    }

    static boolean isPropertyArgument(String arg) {
        return arg.startsWith("-") && arg.contains("=");
    }

    static String getArgumentProperty(String arg) {
        int index = arg.indexOf("=");
        return arg.substring(1, index);
    }

    static String getArgumentValue(String arg) {
        int index = arg.indexOf("=");
        if (index < 0 || index == arg.length()) {
            return "";
        }
        return arg.substring(index + 1);
    }

    private static List<String> determineOperations(String[] args) {
        ArrayList<String> operations = new ArrayList<String>();
        for (String arg : args) {
            if (arg.startsWith("-")) continue;
            operations.add(arg);
        }
        return operations;
    }
}

