/*
 * Decompiled with CFR 0.152.
 */
package org.flywaydb.commandline;

import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import org.flywaydb.commandline.ConsoleLog;
import org.flywaydb.commandline.ConsoleLogCreator;
import org.flywaydb.core.Flyway;
import org.flywaydb.core.api.FlywayException;
import org.flywaydb.core.api.MigrationInfo;
import org.flywaydb.core.internal.info.MigrationInfoDumper;
import org.flywaydb.core.internal.util.ClassUtils;
import org.flywaydb.core.internal.util.FileCopyUtils;
import org.flywaydb.core.internal.util.PropertiesUtils;
import org.flywaydb.core.internal.util.StringUtils;
import org.flywaydb.core.internal.util.VersionPrinter;
import org.flywaydb.core.internal.util.logging.Log;
import org.flywaydb.core.internal.util.logging.LogCreator;
import org.flywaydb.core.internal.util.logging.LogFactory;

public class Main {
    private static Log LOG;

    static void initLogging(ConsoleLog.Level level) {
        LogFactory.setLogCreator((LogCreator)new ConsoleLogCreator(level));
        LOG = LogFactory.getLog(Main.class);
    }

    public static void main(String[] args) {
        ConsoleLog.Level logLevel = Main.getLogLevel(args);
        Main.initLogging(logLevel);
        try {
            Main.printVersion();
            List<String> operations = Main.determineOperations(args);
            if (operations.isEmpty()) {
                Main.printUsage();
                return;
            }
            Properties properties = new Properties();
            Main.initializeDefaults(properties);
            Main.loadConfiguration(properties, args);
            Main.overrideConfiguration(properties, args);
            Main.loadJdbcDrivers();
            Main.loadJavaMigrationsFromJarDir(properties);
            Main.loadJavaMigrationsFromJarDirs(properties);
            Flyway flyway = new Flyway();
            flyway.configure(properties);
            for (String operation : operations) {
                Main.executeOperation(flyway, operation);
            }
        }
        catch (Exception e) {
            if (logLevel == ConsoleLog.Level.DEBUG) {
                LOG.error("Unexpected error", e);
            } else if (e instanceof FlywayException) {
                LOG.error(e.getMessage());
            } else {
                LOG.error(e.toString());
            }
            System.exit(1);
        }
    }

    private static void executeOperation(Flyway flyway, String operation) {
        if ("clean".equals(operation)) {
            flyway.clean();
        } else if ("baseline".equals(operation)) {
            flyway.baseline();
        } else if ("init".equals(operation)) {
            LOG.warn("init is deprecated. Use baseline instead. Will be removed in Flyway 4.0.");
            flyway.baseline();
        } else if ("migrate".equals(operation)) {
            flyway.migrate();
        } else if ("validate".equals(operation)) {
            flyway.validate();
        } else if ("info".equals(operation)) {
            LOG.info("\n" + MigrationInfoDumper.dumpToAsciiTable((MigrationInfo[])flyway.info().all()));
        } else if ("repair".equals(operation)) {
            flyway.repair();
        } else {
            LOG.error("Invalid operation: " + operation);
            Main.printUsage();
            System.exit(1);
        }
    }

    private static ConsoleLog.Level getLogLevel(String[] args) {
        for (String arg : args) {
            if ("-X".equals(arg)) {
                return ConsoleLog.Level.DEBUG;
            }
            if (!"-q".equals(arg)) continue;
            return ConsoleLog.Level.WARN;
        }
        return ConsoleLog.Level.INFO;
    }

    private static void initializeDefaults(Properties properties) {
        properties.put("flyway.locations", "filesystem:" + new File(Main.getInstallationDir(), "sql").getAbsolutePath());
        properties.put("flyway.jarDirs", new File(Main.getInstallationDir(), "jars").getAbsolutePath());
    }

    private static void printVersion() throws IOException {
        VersionPrinter.printVersion((ClassLoader)Thread.currentThread().getContextClassLoader());
        LOG.info("");
    }

    private static void printUsage() {
        LOG.info("********");
        LOG.info("* Usage");
        LOG.info("********");
        LOG.info("");
        LOG.info("flyway [options] command");
        LOG.info("");
        LOG.info("By default, the configuration will be read from conf/flyway.conf.");
        LOG.info("Options passed from the command-line override the configuration.");
        LOG.info("");
        LOG.info("Commands");
        LOG.info("========");
        LOG.info("migrate  : Migrates the database");
        LOG.info("clean    : Drops all objects in the configured schemas");
        LOG.info("info     : Prints the information about applied, current and pending migrations");
        LOG.info("validate : Validates the applied migrations against the ones on the classpath");
        LOG.info("baseline : Baselines an existing database at the baselineVersion");
        LOG.info("repair   : Repairs the metadata table");
        LOG.info("");
        LOG.info("Options (Format: -key=value)");
        LOG.info("=======");
        LOG.info("driver                 : Fully qualified classname of the jdbc driver");
        LOG.info("url                    : Jdbc url to use to connect to the database");
        LOG.info("user                   : User to use to connect to the database");
        LOG.info("password               : Password to use to connect to the database");
        LOG.info("schemas                : Comma-separated list of the schemas managed by Flyway");
        LOG.info("table                  : Name of Flyway's metadata table");
        LOG.info("locations              : Classpath locations to scan recursively for migrations");
        LOG.info("resolvers              : Comma-separated list of custom MigrationResolvers");
        LOG.info("sqlMigrationPrefix     : File name prefix for Sql migrations");
        LOG.info("sqlMigrationSeparator  : File name separator for Sql migrations");
        LOG.info("sqlMigrationSuffix     : File name suffix for Sql migrations");
        LOG.info("encoding               : Encoding of Sql migrations");
        LOG.info("placeholderReplacement : Whether placeholders should be replaced");
        LOG.info("placeholders           : Placeholders to replace in Sql migrations");
        LOG.info("placeholderPrefix      : Prefix of every placeholder");
        LOG.info("placeholderSuffix      : Suffix of every placeholder");
        LOG.info("target                 : Target version up to which Flyway should use migrations");
        LOG.info("outOfOrder             : Allows migrations to be run \"out of order\"");
        LOG.info("callbacks              : Comma-separated list of FlywayCallback classes");
        LOG.info("validateOnMigrate      : Validate when running migrate");
        LOG.info("cleanOnValidationError : Automatically clean on a validation error");
        LOG.info("baselineVersion        : Version to tag schema with when executing baseline");
        LOG.info("baselineDescription    : Description to tag schema with when executing baseline");
        LOG.info("baselineOnMigrate      : Baseline on migrate against uninitialized non-empty schema");
        LOG.info("configFile             : Config file to use (default: conf/flyway.properties)");
        LOG.info("configFileEncoding     : Encoding of the config file (default: UTF-8)");
        LOG.info("jarDirs                : Dirs for Jdbc drivers & Java migrations (default: jars)");
        LOG.info("");
        LOG.info("Add -X to print debug output");
        LOG.info("Add -q to suppress all output, except for errors and warnings");
        LOG.info("");
        LOG.info("Example");
        LOG.info("=======");
        LOG.info("flyway -target=1.5 -placeholders.user=my_user info");
        LOG.info("");
        LOG.info("More info at http://flywaydb.org/documentation/commandline");
    }

    private static void loadJdbcDrivers() throws IOException {
        File driversDir = new File(Main.getInstallationDir(), "drivers");
        File[] files = driversDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".jar");
            }
        });
        if (files == null) {
            LOG.error("Directory for Jdbc Drivers not found: " + driversDir.getAbsolutePath());
            System.exit(1);
        }
        for (File file : files) {
            Main.addJarOrDirectoryToClasspath(file.getPath());
        }
    }

    private static void loadJavaMigrationsFromJarDir(Properties properties) throws IOException {
        String jarDir = properties.getProperty("flyway.jarDir");
        if (!StringUtils.hasLength((String)jarDir)) {
            return;
        }
        LOG.warn("flyway.jarDir is deprecated and will be removed in Flyway 4.0. Use flyway.jarDirs instead.");
        File dir = new File(jarDir);
        File[] files = dir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.endsWith(".jar");
            }
        });
        if (files == null) {
            LOG.error("Directory for Java Migrations not found: " + jarDir);
            System.exit(1);
        }
        for (File file : files) {
            Main.addJarOrDirectoryToClasspath(file.getPath());
        }
    }

    private static void loadJavaMigrationsFromJarDirs(Properties properties) throws IOException {
        String[] dirs;
        String jarDirs = properties.getProperty("flyway.jarDirs");
        if (!StringUtils.hasLength((String)jarDirs)) {
            return;
        }
        jarDirs = jarDirs.replace(File.pathSeparator, ",");
        for (String dirName : dirs = StringUtils.tokenizeToStringArray((String)jarDirs, (String)",")) {
            File dir = new File(dirName);
            File[] files = dir.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(".jar");
                }
            });
            if (files == null) {
                LOG.error("Directory for Java Migrations not found: " + dirName);
                System.exit(1);
            }
            for (File file : files) {
                Main.addJarOrDirectoryToClasspath(file.getPath());
            }
        }
    }

    static void addJarOrDirectoryToClasspath(String name) throws IOException {
        LOG.debug("Adding location to classpath: " + name);
        try {
            URL url = new File(name).toURI().toURL();
            URLClassLoader sysloader = (URLClassLoader)ClassLoader.getSystemClassLoader();
            Method method = URLClassLoader.class.getDeclaredMethod("addURL", URL.class);
            method.setAccessible(true);
            method.invoke((Object)sysloader, url);
        }
        catch (Exception e) {
            throw new FlywayException("Unable to load " + name, (Throwable)e);
        }
    }

    static void loadConfiguration(Properties properties, String[] args) {
        String encoding = Main.determineConfigurationFileEncoding(args);
        if (Main.loadConfigurationFile(properties, Main.getInstallationDir() + "/conf/flyway.properties", encoding, false)) {
            LOG.warn("conf/flyway.properties usage is deprecated and will be removed in Flyway 4.0. Use conf/flyway.conf instead.");
        }
        Main.loadConfigurationFile(properties, Main.getInstallationDir() + "/conf/flyway.conf", encoding, false);
        Main.loadConfigurationFile(properties, System.getProperty("user.home") + "/flyway.conf", encoding, false);
        Main.loadConfigurationFile(properties, "flyway.conf", encoding, false);
        String configFile = Main.determineConfigurationFileArgument(args);
        if (configFile != null) {
            Main.loadConfigurationFile(properties, configFile, encoding, true);
        }
    }

    private static boolean loadConfigurationFile(Properties properties, String file, String encoding, boolean failIfMissing) throws FlywayException {
        File configFile = new File(file);
        String errorMessage = "Unable to load config file: " + configFile.getAbsolutePath();
        if (!configFile.isFile() || !configFile.canRead()) {
            if (!failIfMissing) {
                LOG.debug(errorMessage);
                return false;
            }
            throw new FlywayException(errorMessage);
        }
        LOG.debug("Loading config file: " + configFile.getAbsolutePath());
        try {
            InputStreamReader fileReader = new InputStreamReader((InputStream)new FileInputStream(configFile), encoding);
            String propertiesData = FileCopyUtils.copyToString((Reader)fileReader).replace("\\", "\\\\");
            properties.putAll((Map<?, ?>)PropertiesUtils.loadPropertiesFromString((String)propertiesData));
            return true;
        }
        catch (IOException e) {
            throw new FlywayException(errorMessage, (Throwable)e);
        }
    }

    private static String determineConfigurationFileArgument(String[] args) {
        for (String arg : args) {
            if (!Main.isPropertyArgument(arg) || !"configFile".equals(Main.getArgumentProperty(arg))) continue;
            return Main.getArgumentValue(arg);
        }
        return null;
    }

    private static String getInstallationDir() {
        String path = ClassUtils.getLocationOnDisk(Main.class);
        return new File(path).getParentFile().getParentFile().getAbsolutePath();
    }

    private static String determineConfigurationFileEncoding(String[] args) {
        for (String arg : args) {
            if (!Main.isPropertyArgument(arg) || !"configFileEncoding".equals(Main.getArgumentProperty(arg))) continue;
            return Main.getArgumentValue(arg);
        }
        return "UTF-8";
    }

    static void overrideConfiguration(Properties properties, String[] args) {
        for (String arg : args) {
            if (!Main.isPropertyArgument(arg)) continue;
            properties.put("flyway." + Main.getArgumentProperty(arg), Main.getArgumentValue(arg));
        }
    }

    static boolean isPropertyArgument(String arg) {
        return arg.startsWith("-") && arg.contains("=");
    }

    static String getArgumentProperty(String arg) {
        int index = arg.indexOf("=");
        return arg.substring(1, index);
    }

    static String getArgumentValue(String arg) {
        int index = arg.indexOf("=");
        if (index < 0 || index == arg.length()) {
            return "";
        }
        return arg.substring(index + 1);
    }

    private static List<String> determineOperations(String[] args) {
        ArrayList<String> operations = new ArrayList<String>();
        for (String arg : args) {
            if (arg.startsWith("-")) continue;
            operations.add(arg);
        }
        return operations;
    }
}

