package cdm.product.collateral.functions;

import cdm.base.datetime.Period;
import cdm.base.datetime.PeriodBound;
import cdm.base.datetime.PeriodRange;
import cdm.product.collateral.EligibilityQuery;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.expression.ComparisonResult;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import java.math.BigDecimal;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(CheckMaturity.CheckMaturityDefault.class)
public abstract class CheckMaturity implements RosettaFunction {

	/**
	* @param maturityRange 
	* @param query 
	* @return isEqual 
	*/
	public Boolean evaluate(PeriodRange maturityRange, EligibilityQuery query) {
		Boolean isEqual = doEvaluate(maturityRange, query);
		
		return isEqual;
	}

	protected abstract Boolean doEvaluate(PeriodRange maturityRange, EligibilityQuery query);

	protected abstract Mapper<Boolean> upperBoundCheck(PeriodRange maturityRange, EligibilityQuery query);

	protected abstract Mapper<Boolean> lowerBoundCheck(PeriodRange maturityRange, EligibilityQuery query);

	public static class CheckMaturityDefault extends CheckMaturity {
		@Override
		protected Boolean doEvaluate(PeriodRange maturityRange, EligibilityQuery query) {
			Boolean isEqual = null;
			return assignOutput(isEqual, maturityRange, query);
		}
		
		protected Boolean assignOutput(Boolean isEqual, PeriodRange maturityRange, EligibilityQuery query) {
			isEqual = notExists(MapperS.of(maturityRange)).or(ComparisonResult.of(MapperS.of(upperBoundCheck(maturityRange, query).get())).and(ComparisonResult.of(MapperS.of(lowerBoundCheck(maturityRange, query).get())))).get();
			
			return isEqual;
		}
		
		@Override
		protected Mapper<Boolean> upperBoundCheck(PeriodRange maturityRange, EligibilityQuery query) {
			return MapperUtils.runSingle(() -> {
				if (notExists(MapperS.of(maturityRange).<PeriodBound>map("getUpperBound", periodRange -> periodRange.getUpperBound()).<Period>map("getPeriod", periodBound -> periodBound.getPeriod())).getOrDefault(false)) {
					return MapperS.of(Boolean.valueOf(true));
				}
				else if (ComparisonResult.of(MapperS.of(maturityRange).<PeriodBound>map("getUpperBound", periodRange -> periodRange.getUpperBound()).<Boolean>map("getInclusive", periodBound -> periodBound.getInclusive())).or(notExists(MapperS.of(maturityRange).<PeriodBound>map("getUpperBound", periodRange -> periodRange.getUpperBound()).<Boolean>map("getInclusive", periodBound -> periodBound.getInclusive()))).getOrDefault(false)) {
					return greaterThanEquals(MapperS.of(maturityRange).<PeriodBound>map("getUpperBound", periodRange -> periodRange.getUpperBound()).<Period>map("getPeriod", periodBound -> periodBound.getPeriod()).<Integer>map("getPeriodMultiplier", period -> period.getPeriodMultiplier()), MapperS.of(query).<BigDecimal>map("getMaturity", eligibilityQuery -> eligibilityQuery.getMaturity()), CardinalityOperator.All);
				}
				else {
					return greaterThan(MapperS.of(maturityRange).<PeriodBound>map("getUpperBound", periodRange -> periodRange.getUpperBound()).<Period>map("getPeriod", periodBound -> periodBound.getPeriod()).<Integer>map("getPeriodMultiplier", period -> period.getPeriodMultiplier()), MapperS.of(query).<BigDecimal>map("getMaturity", eligibilityQuery -> eligibilityQuery.getMaturity()), CardinalityOperator.All).asMapper();
				}
			});
		}
		
		@Override
		protected Mapper<Boolean> lowerBoundCheck(PeriodRange maturityRange, EligibilityQuery query) {
			return MapperUtils.runSingle(() -> {
				if (notExists(MapperS.of(maturityRange).<PeriodBound>map("getLowerBound", periodRange -> periodRange.getLowerBound()).<Period>map("getPeriod", periodBound -> periodBound.getPeriod())).getOrDefault(false)) {
					return MapperS.of(Boolean.valueOf(true));
				}
				else if (ComparisonResult.of(MapperS.of(maturityRange).<PeriodBound>map("getLowerBound", periodRange -> periodRange.getLowerBound()).<Boolean>map("getInclusive", periodBound -> periodBound.getInclusive())).or(notExists(MapperS.of(maturityRange).<PeriodBound>map("getLowerBound", periodRange -> periodRange.getLowerBound()).<Boolean>map("getInclusive", periodBound -> periodBound.getInclusive()))).getOrDefault(false)) {
					return lessThanEquals(MapperS.of(maturityRange).<PeriodBound>map("getLowerBound", periodRange -> periodRange.getLowerBound()).<Period>map("getPeriod", periodBound -> periodBound.getPeriod()).<Integer>map("getPeriodMultiplier", period -> period.getPeriodMultiplier()), MapperS.of(query).<BigDecimal>map("getMaturity", eligibilityQuery -> eligibilityQuery.getMaturity()), CardinalityOperator.All);
				}
				else {
					return greaterThan(MapperS.of(maturityRange).<PeriodBound>map("getLowerBound", periodRange -> periodRange.getLowerBound()).<Period>map("getPeriod", periodBound -> periodBound.getPeriod()).<Integer>map("getPeriodMultiplier", period -> period.getPeriodMultiplier()), MapperS.of(query).<BigDecimal>map("getMaturity", eligibilityQuery -> eligibilityQuery.getMaturity()), CardinalityOperator.All).asMapper();
				}
			});
		}
	}
}
