package cdm.product.asset.calculation.functions;

import cdm.base.datetime.daycount.DayCountFractionEnum;
import cdm.base.datetime.daycount.metafields.FieldWithMetaDayCountFractionEnum;
import cdm.base.math.UnitType;
import cdm.observable.asset.Money;
import cdm.product.asset.FixedAmountCalculationDetails;
import cdm.product.asset.FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder;
import cdm.product.asset.InterestRatePayout;
import cdm.product.common.schedule.CalculationPeriodBase;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.MapperMaths;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperS;
import com.rosetta.model.lib.mapper.MapperUtils;
import com.rosetta.model.metafields.FieldWithMetaString;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(FixedAmountCalculation.FixedAmountCalculationDefault.class)
public abstract class FixedAmountCalculation implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected CalculateYearFraction calculateYearFraction;
	@Inject protected GetFixedRate getFixedRate;
	@Inject protected GetNotionalAmount getNotionalAmount;

	/**
	* @param interestRatePayout 
	* @param calculationPeriod 
	* @param notional 
	* @return fixedAmountDetails 
	*/
	public FixedAmountCalculationDetails evaluate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
		FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder fixedAmountDetailsBuilder = doEvaluate(interestRatePayout, calculationPeriod, notional);
		
		final FixedAmountCalculationDetails fixedAmountDetails;
		if (fixedAmountDetailsBuilder == null) {
			fixedAmountDetails = null;
		} else {
			fixedAmountDetails = fixedAmountDetailsBuilder.build();
			objectValidator.validate(FixedAmountCalculationDetails.class, fixedAmountDetails);
		}
		
		return fixedAmountDetails;
	}

	protected abstract FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder doEvaluate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional);

	protected abstract Mapper<BigDecimal> fixedRate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional);

	protected abstract Mapper<? extends Money> calculationAmount(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional);

	protected abstract Mapper<DayCountFractionEnum> dcf(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional);

	protected abstract Mapper<BigDecimal> yearFraction(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional);

	protected abstract Mapper<BigDecimal> calcAmt(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional);

	public static class FixedAmountCalculationDefault extends FixedAmountCalculation {
		@Override
		protected FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder doEvaluate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder fixedAmountDetails = FixedAmountCalculationDetails.builder();
			return assignOutput(fixedAmountDetails, interestRatePayout, calculationPeriod, notional);
		}
		
		protected FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder assignOutput(FixedAmountCalculationDetails.FixedAmountCalculationDetailsBuilder fixedAmountDetails, InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			fixedAmountDetails
				.setCalculationPeriod(MapperS.of(calculationPeriod).get());
			
			fixedAmountDetails
				.getOrCreateCalculationPeriodNotionalAmount()
				.setValue(MapperS.of(calcAmt(interestRatePayout, calculationPeriod, notional).get()).get());
			
			fixedAmountDetails
				.getOrCreateCalculationPeriodNotionalAmount()
				.getOrCreateUnit()
				.setCurrencyValue(MapperS.of(calculationAmount(interestRatePayout, calculationPeriod, notional).get()).<UnitType>map("getUnit", measureBase -> measureBase.getUnit()).<FieldWithMetaString>map("getCurrency", unitType -> unitType.getCurrency()).<String>map("getValue", _f->_f.getValue()).get());
			
			fixedAmountDetails
				.setFixedRate(MapperS.of(fixedRate(interestRatePayout, calculationPeriod, notional).get()).get());
			
			fixedAmountDetails
				.setYearFraction(MapperS.of(yearFraction(interestRatePayout, calculationPeriod, notional).get()).get());
			
			fixedAmountDetails
				.setCalculatedAmount(MapperMaths.<BigDecimal, BigDecimal, BigDecimal>multiply(MapperMaths.<BigDecimal, BigDecimal, BigDecimal>multiply(MapperS.of(calcAmt(interestRatePayout, calculationPeriod, notional).get()), MapperS.of(fixedAmountDetails).<BigDecimal>map("getFixedRate", fixedAmountCalculationDetails -> fixedAmountCalculationDetails.getFixedRate())), MapperS.of(fixedAmountDetails).<BigDecimal>map("getYearFraction", fixedAmountCalculationDetails -> fixedAmountCalculationDetails.getYearFraction())).get());
			
			return Optional.ofNullable(fixedAmountDetails)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<BigDecimal> fixedRate(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			return MapperS.of(getFixedRate.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<? extends Money> calculationAmount(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			return MapperS.of(getNotionalAmount.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<DayCountFractionEnum> dcf(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			return MapperS.of(interestRatePayout).<FieldWithMetaDayCountFractionEnum>map("getDayCountFraction", _interestRatePayout -> _interestRatePayout.getDayCountFraction()).<DayCountFractionEnum>map("getValue", _f->_f.getValue());
		}
		
		@Override
		protected Mapper<BigDecimal> yearFraction(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			return MapperS.of(calculateYearFraction.evaluate(MapperS.of(interestRatePayout).get(), MapperS.of(dcf(interestRatePayout, calculationPeriod, notional).get()).get(), MapperS.of(calculationPeriod).get()));
		}
		
		@Override
		protected Mapper<BigDecimal> calcAmt(InterestRatePayout interestRatePayout, CalculationPeriodBase calculationPeriod, BigDecimal notional) {
			return MapperUtils.runSingle(() -> {
				if (exists(MapperS.of(notional)).getOrDefault(false)) {
					return MapperS.of(notional);
				}
				else {
					return MapperS.of(calculationAmount(interestRatePayout, calculationPeriod, notional).get()).<BigDecimal>map("getValue", measureBase -> measureBase.getValue());
				}
			});
		}
	}
}
