package cdm.event.common.functions;

import cdm.base.datetime.AdjustableOrRelativeDate;
import cdm.base.math.ArithmeticOperationEnum;
import cdm.base.math.DatedValue;
import cdm.base.math.NonNegativeQuantitySchedule;
import cdm.base.math.QuantityChangeDirectionEnum;
import cdm.base.math.UnitType;
import cdm.base.math.metafields.FieldWithMetaNonNegativeQuantitySchedule;
import cdm.base.staticdata.identifier.Identifier;
import cdm.event.common.PrimitiveInstruction;
import cdm.event.common.PrimitiveInstruction.PrimitiveInstructionBuilder;
import cdm.event.common.QuantityChangeInstruction;
import cdm.event.common.Trade;
import cdm.event.common.TradeState;
import cdm.observable.asset.CashPrice;
import cdm.observable.asset.Price;
import cdm.observable.asset.PriceComposite;
import cdm.observable.asset.PriceExpressionEnum;
import cdm.observable.asset.PriceSchedule;
import cdm.observable.asset.PriceTypeEnum;
import cdm.observable.asset.metafields.FieldWithMetaPriceSchedule;
import cdm.product.common.settlement.PriceQuantity;
import cdm.product.template.ContractualProduct;
import cdm.product.template.Product;
import cdm.product.template.TradableProduct;
import cdm.product.template.TradeLot;
import com.google.inject.ImplementedBy;
import com.rosetta.model.lib.expression.CardinalityOperator;
import com.rosetta.model.lib.functions.ModelObjectValidator;
import com.rosetta.model.lib.functions.RosettaFunction;
import com.rosetta.model.lib.mapper.Mapper;
import com.rosetta.model.lib.mapper.MapperC;
import com.rosetta.model.lib.mapper.MapperS;
import java.math.BigDecimal;
import java.util.Optional;
import javax.inject.Inject;

import static com.rosetta.model.lib.expression.ExpressionOperators.*;

@ImplementedBy(Create_RepricePrimitiveInstruction.Create_RepricePrimitiveInstructionDefault.class)
public abstract class Create_RepricePrimitiveInstruction implements RosettaFunction {
	
	@Inject protected ModelObjectValidator objectValidator;
	
	// RosettaFunction dependencies
	//
	@Inject protected Create_EffectiveOrTerminationDateTermChangeInstruction create_EffectiveOrTerminationDateTermChangeInstruction;
	@Inject protected Create_TerminationInstruction create_TerminationInstruction;

	/**
	* @param tradeState The original trade state and trade to be repriced.
	* @param newAllinPrice The collateral new all-in price.
	* @param newCashValue The new cash amount.
	* @param effectiveRepriceDate The date to reprice the collateral
	* @return instruction 
	*/
	public PrimitiveInstruction evaluate(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
		PrimitiveInstruction.PrimitiveInstructionBuilder instructionBuilder = doEvaluate(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate);
		
		final PrimitiveInstruction instruction;
		if (instructionBuilder == null) {
			instruction = null;
		} else {
			instruction = instructionBuilder.build();
			objectValidator.validate(PrimitiveInstruction.class, instruction);
		}
		
		return instruction;
	}

	protected abstract PrimitiveInstruction.PrimitiveInstructionBuilder doEvaluate(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate);

	protected abstract Mapper<? extends PriceQuantity> oldPriceQuantity(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate);

	protected abstract Mapper<? extends PriceSchedule> currentAssetPrice(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate);

	protected abstract Mapper<? extends Price> newPrice(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate);

	protected abstract Mapper<? extends NonNegativeQuantitySchedule> changeCashQuantity(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate);

	protected abstract Mapper<? extends PriceQuantity> newPriceQuantity(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate);

	public static class Create_RepricePrimitiveInstructionDefault extends Create_RepricePrimitiveInstruction {
		@Override
		protected PrimitiveInstruction.PrimitiveInstructionBuilder doEvaluate(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			PrimitiveInstruction.PrimitiveInstructionBuilder instruction = PrimitiveInstruction.builder();
			return assignOutput(instruction, tradeState, newAllinPrice, newCashValue, effectiveRepriceDate);
		}
		
		protected PrimitiveInstruction.PrimitiveInstructionBuilder assignOutput(PrimitiveInstruction.PrimitiveInstructionBuilder instruction, TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			instruction
				.getOrCreateSplit()
				.setBreakdown(MapperC.<PrimitiveInstruction>of(MapperS.of(create_TerminationInstruction.evaluate(MapperS.of(tradeState).get()))).getMulti());
			
			instruction
				.getOrCreateSplit()
				.addBreakdown(MapperC.<PrimitiveInstruction>of(MapperS.of(PrimitiveInstruction.builder()
					.setQuantityChange(MapperS.of(QuantityChangeInstruction.builder()
						.setChange(MapperC.<PriceQuantity>of(MapperS.of(newPriceQuantity(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get())).getMulti())
						.setDirection(MapperS.of(QuantityChangeDirectionEnum.REPLACE).get())
						.setLotIdentifier(MapperC.<Identifier>ofNull().getMulti())
						.build())
					.get())
					.setTermsChange(MapperS.of(create_EffectiveOrTerminationDateTermChangeInstruction.evaluate(MapperS.of(tradeState).<Trade>map("getTrade", _tradeState -> _tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<Product>map("getProduct", tradableProduct -> tradableProduct.getProduct()).<ContractualProduct>map("getContractualProduct", product -> product.getContractualProduct()).get(), MapperS.of(effectiveRepriceDate).get(), null)).get())
					.build())
				).getMulti());
			
			return Optional.ofNullable(instruction)
				.map(o -> o.prune())
				.orElse(null);
		}
		
		@Override
		protected Mapper<? extends PriceQuantity> oldPriceQuantity(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			return MapperS.of(tradeState).<Trade>map("getTrade", _tradeState -> _tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<TradeLot>mapC("getTradeLot", tradableProduct -> tradableProduct.getTradeLot()).<PriceQuantity>mapC("getPriceQuantity", tradeLot -> tradeLot.getPriceQuantity());
		}
		
		@Override
		protected Mapper<? extends PriceSchedule> currentAssetPrice(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			return MapperC.<PriceQuantity>of(oldPriceQuantity(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).getMulti())
				.mapItemToList(item -> (MapperC<PriceSchedule>)item.<FieldWithMetaPriceSchedule>mapC("getPrice", priceQuantity -> priceQuantity.getPrice()).<PriceSchedule>map("getValue", _f->_f.getValue()))
				.apply(item -> item
					.flattenList())
				.apply(item -> item
					.filterItemNullSafe(_item -> (Boolean)areEqual(_item.<PriceTypeEnum>map("getPriceType", priceSchedule -> priceSchedule.getPriceType()), MapperS.of(PriceTypeEnum.ASSET_PRICE), CardinalityOperator.All).get()))
				.apply(item -> MapperS.of(item.get()));
		}
		
		@Override
		protected Mapper<? extends Price> newPrice(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			return MapperS.of(Price.builder()
				.setValue(MapperS.of(newAllinPrice).get())
				.setUnit(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<UnitType>map("getUnit", measureBase -> measureBase.getUnit()).get())
				.setPerUnitOf(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<UnitType>map("getPerUnitOf", priceSchedule -> priceSchedule.getPerUnitOf()).get())
				.setPriceType(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<PriceTypeEnum>map("getPriceType", priceSchedule -> priceSchedule.getPriceType()).get())
				.setPriceExpression(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<PriceExpressionEnum>map("getPriceExpression", priceSchedule -> priceSchedule.getPriceExpression()).get())
				.setComposite(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<PriceComposite>map("getComposite", priceSchedule -> priceSchedule.getComposite()).get())
				.setArithmeticOperator(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<ArithmeticOperationEnum>map("getArithmeticOperator", priceSchedule -> priceSchedule.getArithmeticOperator()).get())
				.setCashPrice(MapperS.of(currentAssetPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get()).<CashPrice>map("getCashPrice", priceSchedule -> priceSchedule.getCashPrice()).get())
				.setDatedValue(MapperC.<DatedValue>ofNull().getMulti())
				.build())
			;
		}
		
		@Override
		protected Mapper<? extends NonNegativeQuantitySchedule> changeCashQuantity(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			return MapperS.of(MapperS.of(tradeState).<Trade>map("getTrade", _tradeState -> _tradeState.getTrade()).<TradableProduct>map("getTradableProduct", trade -> trade.getTradableProduct()).<TradeLot>mapC("getTradeLot", tradableProduct -> tradableProduct.getTradeLot()).get()).<PriceQuantity>mapC("getPriceQuantity", tradeLot -> tradeLot.getPriceQuantity()).<FieldWithMetaNonNegativeQuantitySchedule>mapC("getQuantity", priceQuantity -> priceQuantity.getQuantity()).<NonNegativeQuantitySchedule>map("getValue", _f->_f.getValue())
				.mapItem(item -> (MapperS<NonNegativeQuantitySchedule>)MapperS.of(NonNegativeQuantitySchedule.builder()
					.setValue(MapperS.of(newCashValue).get())
					.setUnit(item.<UnitType>map("getUnit", measureBase -> measureBase.getUnit()).get())
					.build())
				)
				.apply(item -> distinct(item));
		}
		
		@Override
		protected Mapper<? extends PriceQuantity> newPriceQuantity(TradeState tradeState, BigDecimal newAllinPrice, BigDecimal newCashValue, AdjustableOrRelativeDate effectiveRepriceDate) {
			return MapperS.of(PriceQuantity.builder()
				.setPriceValue(MapperC.<Price>of(MapperS.of(newPrice(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).get())).getMulti())
				.setQuantityValue(MapperC.<NonNegativeQuantitySchedule>of(changeCashQuantity(tradeState, newAllinPrice, newCashValue, effectiveRepriceDate).getMulti()).getMulti())
				.build())
			;
		}
	}
}
