/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authz.store;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequestValidationException;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.set.Sets;
import org.elasticsearch.common.xcontent.LoggingDeprecationHandler;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.env.Environment;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.ResourceWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xcontent.DeprecationHandler;
import org.elasticsearch.xcontent.NamedXContentRegistry;
import org.elasticsearch.xcontent.XContentParseException;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.yaml.YamlXContent;
import org.elasticsearch.xpack.core.XPackPlugin;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.security.SecurityField;
import org.elasticsearch.xpack.core.security.authz.RoleDescriptor;
import org.elasticsearch.xpack.core.security.authz.store.ReservedRolesStore;
import org.elasticsearch.xpack.core.security.authz.store.RoleRetrievalResult;
import org.elasticsearch.xpack.core.security.authz.support.DLSRoleQueryValidator;
import org.elasticsearch.xpack.core.security.support.NoOpLogger;
import org.elasticsearch.xpack.core.security.support.Validation;
import org.elasticsearch.xpack.security.PrivilegedFileWatcher;
import org.elasticsearch.xpack.security.authz.FileRoleValidator;

public class FileRolesStore
implements BiConsumer<Set<String>, ActionListener<RoleRetrievalResult>> {
    private static final Pattern IN_SEGMENT_LINE = Pattern.compile("^\\s+.+");
    private static final Pattern SKIP_LINE = Pattern.compile("(^#.*|^\\s*)");
    private static final Logger logger = LogManager.getLogger(FileRolesStore.class);
    private static final RoleDescriptor.Parser ROLE_DESCRIPTOR_PARSER = RoleDescriptor.parserBuilder().allow2xFormat(true).allowDescription(true).build();
    private final Settings settings;
    private final Path file;
    private final FileRoleValidator roleValidator;
    private final XPackLicenseState licenseState;
    private final NamedXContentRegistry xContentRegistry;
    private final List<Consumer<Set<String>>> listeners = new ArrayList<Consumer<Set<String>>>();
    private volatile Map<String, RoleDescriptor> permissions;

    public FileRolesStore(Settings settings, Environment env, ResourceWatcherService watcherService, XPackLicenseState licenseState, NamedXContentRegistry xContentRegistry, FileRoleValidator roleValidator) throws IOException {
        this(settings, env, watcherService, null, roleValidator, licenseState, xContentRegistry);
    }

    FileRolesStore(Settings settings, Environment env, ResourceWatcherService watcherService, Consumer<Set<String>> listener, FileRoleValidator roleValidator, XPackLicenseState licenseState, NamedXContentRegistry xContentRegistry) throws IOException {
        this.settings = settings;
        this.file = FileRolesStore.resolveFile(env);
        this.roleValidator = roleValidator;
        if (listener != null) {
            this.listeners.add(listener);
        }
        this.licenseState = licenseState;
        this.xContentRegistry = xContentRegistry;
        PrivilegedFileWatcher watcher = new PrivilegedFileWatcher(this.file.getParent());
        watcher.addListener(new FileListener());
        watcherService.add((ResourceWatcher)watcher, ResourceWatcherService.Frequency.HIGH);
        this.permissions = FileRolesStore.parseFile(this.file, logger, settings, licenseState, xContentRegistry, roleValidator);
    }

    @Override
    public void accept(Set<String> names, ActionListener<RoleRetrievalResult> listener) {
        listener.onResponse((Object)RoleRetrievalResult.success(this.roleDescriptors(names)));
    }

    Set<RoleDescriptor> roleDescriptors(Set<String> roleNames) {
        Map<String, RoleDescriptor> localPermissions = this.permissions;
        HashSet<RoleDescriptor> descriptors = new HashSet<RoleDescriptor>();
        roleNames.forEach(name -> {
            RoleDescriptor descriptor = (RoleDescriptor)localPermissions.get(name);
            if (descriptor != null) {
                descriptors.add(descriptor);
            }
        });
        return descriptors;
    }

    public boolean exists(String name) {
        Map<String, RoleDescriptor> localPermissions = this.permissions;
        return localPermissions.containsKey(name);
    }

    public Map<String, Object> usageStats() {
        Map<String, RoleDescriptor> localPermissions = this.permissions;
        Map usageStats = Maps.newMapWithExpectedSize((int)3);
        usageStats.put("size", localPermissions.size());
        boolean dls = false;
        boolean fls = false;
        for (RoleDescriptor descriptor : localPermissions.values()) {
            for (RoleDescriptor.IndicesPrivileges indicesPrivileges : descriptor.getIndicesPrivileges()) {
                fls = fls || indicesPrivileges.getGrantedFields() != null || indicesPrivileges.getDeniedFields() != null;
                dls = dls || indicesPrivileges.getQuery() != null;
            }
            if (!fls || !dls) continue;
            break;
        }
        usageStats.put("fls", fls);
        usageStats.put("dls", dls);
        usageStats.put("remote_indices", localPermissions.values().stream().filter(RoleDescriptor::hasRemoteIndicesPrivileges).count());
        usageStats.put("remote_cluster", localPermissions.values().stream().filter(RoleDescriptor::hasRemoteClusterPermissions).count());
        return usageStats;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addListener(Consumer<Set<String>> consumer) {
        Objects.requireNonNull(consumer);
        FileRolesStore fileRolesStore = this;
        synchronized (fileRolesStore) {
            this.listeners.add(consumer);
        }
    }

    public Path getFile() {
        return this.file;
    }

    public Map<String, RoleDescriptor> getAllRoleDescriptors() {
        Map<String, RoleDescriptor> localPermissions = this.permissions;
        return Collections.unmodifiableMap(localPermissions);
    }

    Set<String> getAllRoleNames() {
        return this.permissions.keySet();
    }

    public String toString() {
        return "file roles store (" + String.valueOf(this.file) + ")";
    }

    public static Path resolveFile(Environment env) {
        return XPackPlugin.resolveConfigFile((Environment)env, (String)"roles.yml");
    }

    public static Set<String> parseFileForRoleNames(Path path, Logger logger) {
        if (logger == null) {
            logger = NoOpLogger.INSTANCE;
        }
        HashMap<String, RoleDescriptor> roles = new HashMap<String, RoleDescriptor>();
        logger.trace("attempting to read roles file located at [{}]", (Object)path.toAbsolutePath());
        if (Files.exists(path, new LinkOption[0])) {
            try {
                List<String> roleSegments = FileRolesStore.roleSegments(path);
                for (String segment : roleSegments) {
                    RoleDescriptor rd = FileRolesStore.parseRoleDescriptor(segment, path, logger, false, Settings.EMPTY, NamedXContentRegistry.EMPTY, new FileRoleValidator.Default());
                    if (rd == null) continue;
                    roles.put(rd.getName(), rd);
                }
            }
            catch (IOException ioe) {
                logger.error(() -> Strings.format((String)"failed to read roles file [%s]. skipping all roles...", (Object[])new Object[]{path.toAbsolutePath()}), (Throwable)ioe);
                return Collections.emptySet();
            }
        }
        return Collections.unmodifiableSet(roles.keySet());
    }

    public static Map<String, RoleDescriptor> parseFile(Path path, Logger logger, Settings settings, XPackLicenseState licenseState, NamedXContentRegistry xContentRegistry, FileRoleValidator roleValidator) {
        if (logger == null) {
            logger = NoOpLogger.INSTANCE;
        }
        HashMap<String, RoleDescriptor> roles = new HashMap<String, RoleDescriptor>();
        logger.debug("attempting to read roles file located at [{}]", (Object)path.toAbsolutePath());
        if (Files.exists(path, new LinkOption[0])) {
            try {
                List<String> roleSegments = FileRolesStore.roleSegments(path);
                boolean isDlsLicensed = SecurityField.DOCUMENT_LEVEL_SECURITY_FEATURE.checkWithoutTracking(licenseState);
                for (String segment : roleSegments) {
                    RoleDescriptor descriptor = FileRolesStore.parseRoleDescriptor(segment, path, logger, true, settings, xContentRegistry, roleValidator);
                    if (descriptor == null) continue;
                    if (ReservedRolesStore.isReserved((String)descriptor.getName())) {
                        logger.warn("role [{}] is reserved. the relevant role definition in the mapping file will be ignored", (Object)descriptor.getName());
                        continue;
                    }
                    if (descriptor.isUsingDocumentOrFieldLevelSecurity() && !isDlsLicensed) {
                        logger.warn("role [{}] uses document and/or field level security, which is not enabled by the current license. this role will be ignored", (Object)descriptor.getName());
                        roles.put(descriptor.getName(), descriptor);
                        continue;
                    }
                    roles.put(descriptor.getName(), descriptor);
                }
            }
            catch (IOException ioe) {
                logger.error(() -> Strings.format((String)"failed to read roles file [%s]. skipping all roles...", (Object[])new Object[]{path.toAbsolutePath()}), (Throwable)ioe);
                return Collections.emptyMap();
            }
        } else {
            logger.debug("roles file does not exist");
            return Collections.emptyMap();
        }
        logger.info("parsed [{}] roles from file [{}]", (Object)roles.size(), (Object)path.toAbsolutePath());
        return Collections.unmodifiableMap(roles);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    static RoleDescriptor parseRoleDescriptor(String segment, Path path, Logger logger, boolean resolvePermissions, Settings settings, NamedXContentRegistry xContentRegistry, FileRoleValidator roleValidator) {
        String roleName = null;
        XContentParserConfiguration parserConfig = XContentParserConfiguration.EMPTY.withRegistry(xContentRegistry).withDeprecationHandler((DeprecationHandler)LoggingDeprecationHandler.INSTANCE);
        try (XContentParser parser = YamlXContent.yamlXContent.createParser(parserConfig, segment);){
            XContentParser.Token token = parser.nextToken();
            if (token == XContentParser.Token.START_OBJECT && (token = parser.nextToken()) == XContentParser.Token.FIELD_NAME) {
                roleName = parser.currentName();
                Validation.Error validationError = Validation.Roles.validateRoleName((String)roleName, (boolean)false);
                if (validationError != null) {
                    logger.error("invalid role definition [{}] in roles file [{}]. invalid role name - {}. skipping role...", (Object)roleName, (Object)path.toAbsolutePath(), (Object)validationError);
                    RoleDescriptor roleDescriptor = null;
                    return roleDescriptor;
                }
                if (!resolvePermissions) {
                    RoleDescriptor roleDescriptor = new RoleDescriptor(roleName, null, null, null);
                    return roleDescriptor;
                }
                token = parser.nextToken();
                if (token == XContentParser.Token.START_OBJECT) {
                    RoleDescriptor descriptor = ROLE_DESCRIPTOR_PARSER.parse(roleName, parser);
                    RoleDescriptor roleDescriptor = FileRolesStore.checkDescriptor(descriptor, path, logger, settings, xContentRegistry, roleValidator);
                    return roleDescriptor;
                }
                logger.error("invalid role definition [{}] in roles file [{}]. skipping role...", (Object)roleName, (Object)path.toAbsolutePath());
                RoleDescriptor roleDescriptor = null;
                return roleDescriptor;
            }
            logger.error("invalid role definition [{}] in roles file [{}]. skipping role...", roleName, (Object)path.toAbsolutePath());
            return null;
        }
        catch (ElasticsearchParseException e) {
            assert (roleName != null);
            if (logger.isDebugEnabled()) {
                String finalRoleName = roleName;
                logger.debug(() -> "parsing exception for role [" + finalRoleName + "]", (Throwable)e);
                return null;
            }
            logger.error(e.getMessage() + ". skipping role...");
            return null;
        }
        catch (IOException | XContentParseException e) {
            if (roleName != null) {
                String finalRoleName = roleName;
                logger.error(() -> Strings.format((String)"invalid role definition [%s] in roles file [%s]. skipping role...", (Object[])new Object[]{finalRoleName, path}), e);
                return null;
            }
            logger.error(() -> Strings.format((String)"invalid role definition [%s] in roles file [%s]. skipping role...", (Object[])new Object[]{segment, path}), e);
        }
        return null;
    }

    @Nullable
    private static RoleDescriptor checkDescriptor(RoleDescriptor descriptor, Path path, Logger logger, Settings settings, NamedXContentRegistry xContentRegistry, FileRoleValidator roleValidator) {
        ActionRequestValidationException ex;
        String roleName = descriptor.getName();
        if (descriptor.isUsingDocumentOrFieldLevelSecurity()) {
            if (!((Boolean)XPackSettings.DLS_FLS_ENABLED.get(settings)).booleanValue()) {
                logger.error("invalid role definition [{}] in roles file [{}]. document and field level security is not enabled. set [{}] to [true] in the configuration file. skipping role...", (Object)roleName, (Object)path.toAbsolutePath(), (Object)XPackSettings.DLS_FLS_ENABLED.getKey());
                return null;
            }
            try {
                DLSRoleQueryValidator.validateQueryField((RoleDescriptor.IndicesPrivileges[])descriptor.getIndicesPrivileges(), (NamedXContentRegistry)xContentRegistry);
            }
            catch (IllegalArgumentException | ElasticsearchException e) {
                logger.error(() -> Strings.format((String)"invalid role definition [%s] in roles file [%s]. failed to validate query field. skipping role...", (Object[])new Object[]{roleName, path.toAbsolutePath()}), e);
                return null;
            }
        }
        if ((ex = roleValidator.validatePredefinedRole(descriptor)) != null) {
            throw ex;
        }
        Validation.Error validationError = Validation.Roles.validateRoleDescription((String)descriptor.getDescription());
        if (validationError != null) {
            logger.error("invalid role definition [{}] in roles file [{}]. invalid description - {}. skipping role...", (Object)roleName, (Object)path.toAbsolutePath(), (Object)validationError);
            return null;
        }
        return descriptor;
    }

    private static List<String> roleSegments(Path path) throws IOException {
        ArrayList<String> segments = new ArrayList<String>();
        StringBuilder builder = null;
        for (String line : Files.readAllLines(path, StandardCharsets.UTF_8)) {
            if (SKIP_LINE.matcher(line).matches()) continue;
            if (IN_SEGMENT_LINE.matcher(line).matches()) {
                if (builder == null) continue;
                builder.append(line).append("\n");
                continue;
            }
            if (builder != null) {
                segments.add(builder.toString());
            }
            builder = new StringBuilder(line).append("\n");
        }
        if (builder != null) {
            segments.add(builder.toString());
        }
        return segments;
    }

    private class FileListener
    implements FileChangesListener {
        private FileListener() {
        }

        public void onFileCreated(Path file) {
            this.onFileChanged(file);
        }

        public void onFileDeleted(Path file) {
            this.onFileChanged(file);
        }

        public synchronized void onFileChanged(Path file) {
            if (file.equals(FileRolesStore.this.file)) {
                Map<String, RoleDescriptor> previousPermissions = FileRolesStore.this.permissions;
                try {
                    FileRolesStore.this.permissions = FileRolesStore.parseFile(file, logger, FileRolesStore.this.settings, FileRolesStore.this.licenseState, FileRolesStore.this.xContentRegistry, FileRolesStore.this.roleValidator);
                }
                catch (Exception e) {
                    logger.error(() -> Strings.format((String)"could not reload roles file [%s]. Current roles remain unmodified", (Object[])new Object[]{file.toAbsolutePath()}), (Throwable)e);
                    return;
                }
                Set changedOrMissingRoles = Sets.difference(previousPermissions.entrySet(), FileRolesStore.this.permissions.entrySet()).stream().map(Map.Entry::getKey).collect(Collectors.toSet());
                Set addedRoles = Sets.difference(FileRolesStore.this.permissions.keySet(), previousPermissions.keySet());
                Set changedRoles = Collections.unmodifiableSet(Sets.union(changedOrMissingRoles, (Set)addedRoles));
                if (!changedRoles.isEmpty()) {
                    logger.info("updated roles (roles file [{}] {})", (Object)file.toAbsolutePath(), (Object)(Files.exists(file, new LinkOption[0]) ? "changed" : "removed"));
                    FileRolesStore.this.listeners.forEach(c -> c.accept(changedRoles));
                }
            }
        }
    }
}

