/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.dynamic_config.api.service;

import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.dynamic_config.api.model.Cluster;
import org.terracotta.dynamic_config.api.model.FailoverPriority;
import org.terracotta.dynamic_config.api.model.Node;
import org.terracotta.dynamic_config.api.model.Setting;
import org.terracotta.dynamic_config.api.model.Stripe;
import org.terracotta.dynamic_config.api.model.Version;
import org.terracotta.dynamic_config.api.service.MalformedClusterException;

public class ClusterValidator {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClusterValidator.class);
    private final Cluster cluster;

    public ClusterValidator(Cluster cluster) {
        this.cluster = cluster;
    }

    public void validate() throws MalformedClusterException {
        this.validate(Version.CURRENT);
    }

    public void validate(Version version) throws MalformedClusterException {
        this.validateNodeNames();
        this.validateAddresses();
        this.validateBackupDirs();
        this.validateDataDirs();
        this.validateSecurity();
        this.validateFailoverSetting();
        if (version.amongst(EnumSet.of(Version.V2))) {
            this.validateStripeNames();
        }
    }

    private void validateAddresses() {
        this.checkDuplicateInternalAddresses();
        this.checkPublicAddressContent();
        this.checkDuplicatePublicAddresses();
        this.checkAllOrNoPublicAddresses();
    }

    private void checkAllOrNoPublicAddresses() {
        List nodesWithNoPublicAddresses = this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).filter(node -> !node.getPublicAddress().isPresent()).map(Node::getName).collect(Collectors.toList());
        if (nodesWithNoPublicAddresses.size() != 0 && nodesWithNoPublicAddresses.size() != this.cluster.getNodeCount()) {
            throw new MalformedClusterException("Nodes with names: " + nodesWithNoPublicAddresses + " don't have public addresses defined, but other nodes in the cluster do. Mutative operations on public addresses must be done simultaneously on every node in the cluster");
        }
    }

    private void checkPublicAddressContent() {
        this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).filter(node -> node.getPublicHostname().isConfigured() && !node.getPublicPort().isConfigured() || !node.getPublicHostname().isConfigured() && node.getPublicPort().isConfigured()).findFirst().ifPresent(node -> {
            throw new MalformedClusterException("Public address: '" + node.getPublicHostname().orDefault() + ":" + node.getPublicPort().orDefault() + "' of node with name: " + node.getName() + " isn't well-formed. Public hostname and port need to be set together");
        });
    }

    private void checkDuplicateInternalAddresses() {
        this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).collect(Collectors.groupingBy(Node::getInternalAddress, Collectors.toList())).entrySet().stream().filter(e -> ((List)e.getValue()).size() > 1).findAny().ifPresent(entry -> {
            throw new MalformedClusterException("Nodes with names: " + ((List)entry.getValue()).stream().map(Node::getName).collect(Collectors.joining(", ")) + " have the same address: '" + entry.getKey() + "'");
        });
    }

    private void checkDuplicatePublicAddresses() {
        this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).collect(Collectors.groupingBy(Node::getPublicAddress, Collectors.toList())).entrySet().stream().filter(e -> ((Optional)e.getKey()).isPresent() && ((List)e.getValue()).size() > 1).findAny().ifPresent(entry -> {
            throw new MalformedClusterException("Nodes with names: " + ((List)entry.getValue()).stream().map(Node::getName).collect(Collectors.joining(", ")) + " have the same public address: '" + ((Optional)entry.getKey()).get() + "'");
        });
    }

    private void validateFailoverSetting() {
        FailoverPriority failoverPriority = this.cluster.getFailoverPriority();
        if (failoverPriority == null) {
            throw new MalformedClusterException((Object)((Object)Setting.FAILOVER_PRIORITY) + " setting is missing");
        }
        if (failoverPriority.equals(FailoverPriority.consistency())) {
            int voterCount = failoverPriority.getVoters();
            for (Stripe stripe : this.cluster.getStripes()) {
                int nodeCount = stripe.getNodes().size();
                int sum = voterCount + nodeCount;
                if (sum % 2 != 0) continue;
                LOGGER.warn(System.lineSeparator() + "===================================================================================================================" + System.lineSeparator() + "The sum (" + sum + ") of voter count (" + voterCount + ") and number of nodes (" + nodeCount + ") in stripe '" + stripe.getName() + "' is an even number." + System.lineSeparator() + "An even-numbered configuration is more likely to experience split-brain situations." + System.lineSeparator() + "===================================================================================================================" + System.lineSeparator());
            }
        }
    }

    private void validateNodeNames() {
        this.cluster.getNodes().stream().filter(node -> node.getName() == null).findAny().ifPresent(nodeName -> {
            throw new MalformedClusterException("Found node without name");
        });
        this.cluster.getNodes().stream().map(Node::getName).filter(Objects::nonNull).collect(Collectors.groupingBy(Function.identity(), Collectors.counting())).entrySet().stream().filter(e -> (Long)e.getValue() > 1L).map(Map.Entry::getKey).findAny().ifPresent(nodeName -> {
            throw new MalformedClusterException("Found duplicate node name: " + nodeName);
        });
    }

    private void validateStripeNames() {
        this.cluster.getStripes().stream().filter(stripe -> stripe.getName() == null).findAny().ifPresent(stripeName -> {
            throw new MalformedClusterException("Found stripe without name");
        });
        this.cluster.getStripes().stream().map(Stripe::getName).filter(Objects::nonNull).collect(Collectors.groupingBy(Function.identity(), Collectors.counting())).entrySet().stream().filter(e -> (Long)e.getValue() > 1L).map(Map.Entry::getKey).findAny().ifPresent(stripeName -> {
            throw new MalformedClusterException("Found duplicate stripe name: " + stripeName);
        });
    }

    private void validateDataDirs() {
        Set uniqueDataDirNames = this.cluster.getNodes().stream().map(node -> node.getDataDirs().orDefault().keySet()).collect(Collectors.toSet());
        if (uniqueDataDirNames.size() > 1) {
            throw new MalformedClusterException("Data directory names need to match across the cluster, but found the following mismatches: " + uniqueDataDirNames + ". Mutative operations on data dirs must be done simultaneously on every node in the cluster");
        }
    }

    private void validateBackupDirs() {
        List nodesWithBackupDirs = this.cluster.getNodes().stream().filter(node -> node.getBackupDir().isConfigured()).map(Node::getName).collect(Collectors.toList());
        if (nodesWithBackupDirs.size() != 0 && nodesWithBackupDirs.size() != this.cluster.getNodeCount()) {
            throw new MalformedClusterException("Nodes with names: " + nodesWithBackupDirs + " currently have (or will have) backup directories defined, but some nodes in the cluster do not. Within a cluster, all nodes must have either a backup directory defined or no backup directory defined.");
        }
    }

    private void validateSecurity() {
        this.validateAuthc();
        this.validateSecurityRequirements();
        this.validateAuditLogDir();
    }

    private void validateAuthc() {
        if (this.cluster.getSecurityAuthc().is("certificate") && !this.cluster.getSecuritySslTls().orDefault().booleanValue()) {
            throw new MalformedClusterException((Object)((Object)Setting.SECURITY_SSL_TLS) + " is required for " + (Object)((Object)Setting.SECURITY_AUTHC) + "=certificate");
        }
    }

    private void validateSecurityRequirements() {
        for (Node node : this.cluster.getNodes()) {
            boolean securityDirConfigured = node.getSecurityDir().isConfigured();
            if (!securityDirConfigured && (this.cluster.getSecurityAuthc().isConfigured() || node.getSecurityAuditLogDir().isConfigured() || this.cluster.getSecuritySslTls().orDefault().booleanValue() || this.cluster.getSecurityWhitelist().orDefault().booleanValue())) {
                throw new MalformedClusterException((Object)((Object)Setting.SECURITY_DIR) + " is mandatory for any of the security configuration, but not found on node with name: " + node.getName());
            }
            if (!securityDirConfigured || this.cluster.getSecuritySslTls().orDefault().booleanValue() || this.cluster.getSecurityAuthc().isConfigured() || this.cluster.getSecurityWhitelist().orDefault().booleanValue()) continue;
            throw new MalformedClusterException("One of " + (Object)((Object)Setting.SECURITY_SSL_TLS) + ", " + (Object)((Object)Setting.SECURITY_AUTHC) + ", or " + (Object)((Object)Setting.SECURITY_WHITELIST) + " is required for security configuration, but not found on node with name: " + node.getName());
        }
    }

    private void validateAuditLogDir() {
        List nodesWithNoAuditLogDirs = this.cluster.getStripes().stream().flatMap(s -> s.getNodes().stream()).filter(node -> !node.getSecurityAuditLogDir().isConfigured()).map(Node::getName).collect(Collectors.toList());
        if (nodesWithNoAuditLogDirs.size() != 0 && nodesWithNoAuditLogDirs.size() != this.cluster.getNodeCount()) {
            throw new MalformedClusterException("Nodes with names: " + nodesWithNoAuditLogDirs + " don't have audit log directories defined, but other nodes in the cluster do. Mutative operations on audit log dirs must be done simultaneously on every node in the cluster");
        }
    }
}

